// ---------------------------- font.c ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "font.h"

// Function to initialize the curves of one char
static void InitChar(
  CapyBezierSpline** const ch,
              size_t const nbSegment,
       double const* const c) {
  *ch = CapyBezierSplineAlloc(&nbSegment, 1, 2);
  loop(iSegment, nbSegment) {
    CapyBezier *curve = CapyBezierAlloc(3, 1, 2);
    size_t nbCtrl = 4;
    loop(i, nbCtrl) {
      double coords[2];
      coords[0] = c[iSegment * 8 + i * 2];
      coords[1] = 1.0 - c[iSegment * 8 + i * 2 + 1];
      $(curve, setCtrl)(&i, coords);
    }
    $(*ch, setCurve)(&iSegment, curve);
  }
}

// Create the font dictionary with the default font
// Input:
//   that: the font to initialize
// Output:
//   The font dictionary is freed if necessary and created with the default
//   font.
static void CreateDefaultDictionary(CapyFont* const that) {
  if(that->dict) loop(i, 128) CapyBezierSplineFree(that->dict + i);
  free(that->dict);
  safeMalloc(that->dict, 128);
  loop(i, 128) that->dict[i] = NULL;
  CapyBezierSpline **ch = NULL;
  ch = that->dict + 'A';
  InitChar(
    ch, 3,
    (double[]){
      0.0, 0.0, 0.0, 0.18, 0.32, 1.0, 0.5, 1.0,
      0.5, 1.0, 0.68, 1.0, 1.0, 0.18, 1.0, 0.0,
      0.15, 0.5, 0.15, 0.5, 0.85, 0.5, 0.85, 0.5
    });
  ch = that->dict + 'B';
  InitChar(
    ch, 4,
    (double[]){
      0.00, 0.00, 0.00, 0.00, 0.00, 1.00, 0.00, 1.00,
      0.00, 1.00, 0.77, 1.00, 0.77, 0.58, 0.00, 0.59,
      0.00, 0.59, 0.50, 0.60, 1.01, 0.50, 1.00, 0.26,
      1.00, 0.26, 1.00, 0.00, 0.50, 0.00, 0.00, 0.00
    });
  ch = that->dict + 'C';
  InitChar(
    ch, 4,
    (double[]){
      1.00, 0.67, 1.00, 0.82, 1.00, 1.00, 0.50, 1.00,
      0.50, 1.00, 0.00, 1.00, 0.00, 0.81, 0.00, 0.50,
      0.00, 0.50, 0.00, 0.18, 0.00, 0.00, 0.50, 0.00,
      0.50, 0.00, 1.00, 0.00, 1.00, 0.17, 1.00, 0.33
    });
  ch = that->dict + 'D';
  InitChar(
    ch, 5,
    (double[]){
      0.00, 1.00, 0.00, 1.00, 0.00, 0.00, 0.00, 0.00,
      0.00, 0.00, 1.00, 0.00, 1.00, 0.00, 1.00, 0.50,
      1.00, 0.50, 1.00, 1.00, 0.50, 1.00, 0.00, 1.00,
      0.00, 1.00, -0.11, 1.00, 0.00, 0.00, 0.00, 0.00,
      0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00
    });
  ch = that->dict + 'E';
  InitChar(
    ch, 5,
    (double[]){
      1.00, 1.00, 1.00, 1.00, 0.12, 1.01, 0.06, 0.95,
      0.06, 0.95, -0.01, 0.90, 0.00, 0.10, 0.05, 0.05,
      0.05, 0.05, 0.11, -0.01, 1.00, 0.00, 1.00, 0.00,
      1.00, 0.00, 1.00, 0.00, 0.00, 0.00, 0.00, 0.00,
      0.00, 0.50, 0.00, 0.50, 0.50, 0.50, 0.50, 0.50
    });
  ch = that->dict + 'F';
  InitChar(
    ch, 3,
    (double[]){
      0.00, 0.50, 0.00, 0.50, 0.50, 0.50, 0.50, 0.50,
      1.00, 1.00, 1.00, 1.00, 0.12, 1.01, 0.06, 0.95,
      0.06, 0.95, -0.01, 0.90, 0.00, 0.00, 0.00, 0.00
    });
  ch = that->dict + 'G';
  InitChar(
    ch, 5,
    (double[]){
      1.00, 0.84, 1.00, 1.00, 0.74, 1.00, 0.50, 1.00,
      0.50, 1.00, 0.00, 1.00, 0.00, 0.81, 0.00, 0.50,
      0.00, 0.50, 0.00, 0.18, 0.00, 0.00, 0.50, 0.00,
      0.50, 0.00, 1.00, 0.00, 1.00, 0.50, 1.00, 0.50,
      1.00, 0.50, 1.00, 0.50, 0.50, 0.50, 0.50, 0.50
    });
  ch = that->dict + 'H';
  InitChar(
    ch, 3,
    (double[]){
      1.00, 1.00, 1.00, 1.00, 1.00, 0.00, 1.00, 0.00,
      0.00, 0.50, 0.00, 0.50, 1.00, 0.50, 1.00, 0.50,
      0.00, 1.00, 0.00, 1.00, 0.00, 0.00, 0.00, 0.00
    });
  ch = that->dict + 'I';
  InitChar(
    ch, 3,
    (double[]){
      0.00, 0.00, 0.00, 0.00, 1.00, 0.00, 1.00, 0.00,
      0.50, 1.00, 0.50, 1.00, 0.50, 0.00, 0.50, 0.00,
      0.10, 1.00, 0.10, 1.00, 0.90, 1.00, 0.90, 1.00
    });
  ch = that->dict + 'J';
  InitChar(
    ch, 3,
    (double[]){
      0.66, 1.00, 0.66, 1.00, 1.00, 0.00, 0.50, 0.00,
      0.50, 0.00, 0.00, 0.00, 0.00, 0.33, 0.00, 0.50,
      0.00, 1.00, 0.00, 1.00, 1.00, 1.00, 1.00, 1.00
    });
  ch = that->dict + 'K';
  InitChar(
    ch, 4,
    (double[]){
      0.50, 0.54, 0.50, 0.00, 1.00, 0.00, 1.00, 0.00,
      0.00, 0.50, 0.00, 0.50, 0.00, 0.50, 0.33, 0.50,
      0.33, 0.50, 0.67, 0.51, 1.00, 1.00, 1.00, 1.00,
      0.00, 1.00, 0.00, 1.00, 0.00, 0.00, 0.00, 0.00
    });
  ch = that->dict + 'L';
  InitChar(
    ch, 2,
    (double[]){
      0.00, 1.00, 0.00, 1.00, 0.00, 0.12, 0.05, 0.05,
      0.05, 0.05, 0.08, 0.00, 1.00, 0.00, 1.00, 0.00
    });
  ch = that->dict + 'M';
  InitChar(
    ch, 4,
    (double[]){
      0.00, 0.00, 0.00, 0.00, 0.00, 1.00, 0.00, 1.00,
      0.00, 1.00, 0.00, 1.00, 0.34, 0.67, 0.50, 0.67,
      0.50, 0.67, 0.66, 0.67, 1.00, 1.00, 1.00, 1.00,
      1.00, 1.00, 1.00, 1.00, 1.00, 0.00, 1.00, 0.00
    });
  ch = that->dict + 'N';
  InitChar(
    ch, 3,
    (double[]){
      0.00, 0.00, 0.00, 0.00, 0.00, 1.00, 0.00, 1.00,
      0.00, 1.00, 0.33, 1.00, 0.66, 0.00, 1.00, 0.00,
      1.00, 0.00, 1.00, 0.00, 1.00, 1.00, 1.00, 1.00
    });
  ch = that->dict + 'O';
  InitChar(
    ch, 4,
    (double[]){
      0.50, 1.00, 1.00, 1.00, 1.00, 1.00, 1.00, 0.50,
      1.00, 0.50, 1.00, 0.00, 1.00, 0.00, 0.50, 0.00,
      0.50, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.50,
      0.00, 0.50, 0.00, 1.00, 0.00, 1.00, 0.50, 1.00
    });
  ch = that->dict + 'P';
  InitChar(
    ch, 3,
    (double[]){
      0.00, 0.00, 0.00, 0.00, 0.00, 1.00, 0.00, 1.00,
      0.00, 1.00, 0.50, 1.00, 1.00, 1.00, 1.00, 0.67,
      1.00, 0.67, 1.00, 0.33, 0.50, 0.33, 0.00, 0.33
    });
  ch = that->dict + 'Q';
  InitChar(
    ch, 5,
    (double[]){
      0.66, 0.33, 0.66, 0.33, 1.00, 0.00, 1.00, 0.00,
      0.50, 1.00, 1.00, 1.00, 1.00, 1.00, 1.00, 0.50,
      1.00, 0.50, 1.00, 0.00, 1.00, 0.00, 0.50, 0.00,
      0.50, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.50,
      0.00, 0.50, 0.00, 1.00, 0.00, 1.00, 0.50, 1.00
    });
  ch = that->dict + 'R';
  InitChar(
    ch, 4,
    (double[]){
      0.00, 0.33, 0.33, 0.00, 1.00, 0.00, 1.00, 0.00,
      0.00, 0.00, 0.00, 0.00, 0.00, 1.00, 0.00, 1.00,
      0.00, 1.00, 0.50, 1.00, 1.00, 1.00, 1.00, 0.67,
      1.00, 0.67, 1.00, 0.33, 0.50, 0.33, 0.00, 0.33
    });
  ch = that->dict + 'S';
  InitChar(
    ch, 5,
    (double[]){
      1.00, 0.83, 1.00, 0.99, 1.00, 1.00, 0.50, 1.00,
      0.50, 1.00, 0.00, 1.00, 0.00, 0.83, 0.00, 0.67,
      0.00, 0.67, 0.00, 0.50, 1.00, 0.67, 1.00, 0.50,
      1.00, 0.50, 1.00, 0.33, 1.00, 0.00, 0.50, 0.00,
      0.50, 0.00, 0.00, 0.00, 0.00, 0.16, 0.00, 0.33
    });
  ch = that->dict + 'T';
  InitChar(
    ch, 2,
    (double[]){
      0.50, 1.00, 0.50, 1.00, 0.50, 0.00, 0.50, 0.00,
      0.00, 1.00, 0.00, 1.00, 1.00, 1.00, 1.00, 1.00
    });
  ch = that->dict + 'U';
  InitChar(
    ch, 2,
    (double[]){
      0.00, 1.00, 0.00, 0.50, 0.01, 0.00, 0.50, 0.00,
      0.50, 0.00, 1.00, 0.00, 1.00, 0.51, 1.00, 1.00
    });
  ch = that->dict + 'V';
  InitChar(
    ch, 2,
    (double[]){
      0.00, 1.00, 0.00, 1.00, 0.34, 0.00, 0.50, 0.00,
      0.50, 0.00, 0.67, 0.00, 1.00, 1.00, 1.00, 1.00
    });
  ch = that->dict + 'W';
  InitChar(
    ch, 4,
    (double[]){
      0.00, 1.00, 0.00, 1.00, 0.16, 0.00, 0.33, 0.00,
      0.33, 0.00, 0.50, 0.00, 0.50, 0.50, 0.50, 0.50,
      0.50, 0.50, 0.50, 0.50, 0.50, 0.00, 0.66, 0.00,
      0.66, 0.00, 0.82, 0.00, 1.00, 1.00, 1.00, 1.00
    });
  ch = that->dict + 'X';
  InitChar(
    ch, 4,
    (double[]){
      1.00, 1.00, 1.00, 1.00, 0.50, 0.67, 0.50, 0.51,
      0.50, 0.51, 0.50, 0.33, 0.00, 0.00, 0.00, 0.00,
      0.00, 1.00, 0.00, 1.00, 0.50, 0.67, 0.50, 0.50,
      0.50, 0.50, 0.50, 0.33, 1.00, 0.00, 1.00, 0.00
    });
  ch = that->dict + 'Y';
  InitChar(
    ch, 3,
    (double[]){
      1.00, 1.00, 1.00, 1.00, 0.50, 0.67, 0.50, 0.50,
      0.00, 1.00, 0.00, 1.00, 0.50, 0.67, 0.50, 0.50,
      0.50, 0.50, 0.50, 0.33, 0.50, 0.00, 0.50, 0.00
    });
  ch = that->dict + 'Z';
  InitChar(
    ch, 3,
    (double[]){
      0.00, 1.00, 0.00, 1.00, 1.00, 1.00, 1.00, 1.00,
      1.00, 1.00, 1.00, 0.67, 0.00, 0.33, 0.00, 0.00,
      0.00, 0.00, 0.00, 0.00, 1.00, 0.00, 1.00, 0.00
    });
  ch = that->dict + '0';
  InitChar(
    ch, 5,
    (double[]){
      0.00, 0.00, 0.00, 0.00, 1.00, 1.00, 1.00, 1.00,
      0.50, 1.00, 1.00, 1.00, 1.00, 1.00, 1.00, 0.50,
      1.00, 0.50, 1.00, 0.00, 1.00, 0.00, 0.50, 0.00,
      0.50, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.50,
      0.00, 0.50, 0.00, 1.00, 0.00, 1.00, 0.50, 1.00
    });
  ch = that->dict + '1';
  InitChar(
    ch, 3,
    (double[]){
      0.00, 0.00, 0.00, 0.00, 1.00, 0.00, 1.00, 0.00,
      0.00, 0.67, 0.33, 0.67, 0.50, 1.00, 0.50, 1.00,
      0.50, 1.00, 0.50, 1.00, 0.50, 0.00, 0.50, 0.00
    });
  ch = that->dict + '2';
  InitChar(
    ch, 4,
    (double[]){
      0.00, 0.67, 0.00, 1.00, 0.34, 1.00, 0.50, 1.00,
      0.50, 1.00, 0.66, 1.00, 1.00, 1.00, 1.00, 0.67,
      1.00, 0.67, 1.00, 0.50, 0.00, 0.33, 0.00, 0.00,
      0.00, 0.00, 0.00, 0.00, 1.00, 0.00, 1.00, 0.00
    });
  ch = that->dict + '3';
  InitChar(
    ch, 6,
    (double[]){
      0.00, 0.67, 0.00, 0.83, 0.00, 1.00, 0.50, 1.00,
      0.50, 1.00, 1.00, 1.00, 1.00, 0.83, 1.00, 0.67,
      1.00, 0.67, 1.00, 0.50, 0.50, 0.50, 0.50, 0.50,
      0.50, 0.50, 0.50, 0.50, 1.00, 0.50, 1.00, 0.33,
      1.00, 0.33, 1.00, 0.00, 1.00, 0.00, 0.50, 0.00,
      0.50, 0.00, 0.00, 0.00, 0.00, 0.16, 0.00, 0.33
    });
  ch = that->dict + '4';
  InitChar(
    ch, 3,
    (double[]){
      1.00, 0.33, 1.00, 0.33, 0.00, 0.33, 0.00, 0.33,
      0.00, 0.33, 0.50, 0.50, 0.66, 1.00, 0.66, 1.00,
      0.66, 1.00, 0.66, 1.00, 0.66, 0.00, 0.66, 0.00
    });
  ch = that->dict + '5';
  InitChar(
    ch, 5,
    (double[]){
      1.00, 1.00, 1.00, 1.00, 0.33, 1.00, 0.33, 1.00,
      0.33, 1.00, 0.33, 1.00, 0.00, 0.67, 0.00, 0.67,
      0.00, 0.67, 0.00, 0.67, 1.00, 1.01, 1.00, 0.33,
      1.00, 0.33, 1.00, 0.00, 0.67, 0.00, 0.50, 0.00,
      0.50, 0.00, 0.33, 0.00, 0.00, 0.16, 0.00, 0.33
    });
  ch = that->dict + '6';
  InitChar(
    ch, 6,
    (double[]){
      0.00, 0.33, 0.00, 0.50, 0.33, 0.50, 0.50, 0.50,
      0.50, 0.50, 0.67, 0.50, 1.00, 0.50, 1.00, 0.33,
      1.00, 0.33, 1.00, 0.16, 1.00, 0.00, 0.50, 0.00,
      0.50, 0.00, 0.00, 0.00, 0.00, 0.33, 0.00, 0.50,
      0.00, 0.50, 0.00, 1.00, 0.50, 1.00, 0.50, 1.00,
      0.50, 1.00, 0.50, 1.00, 1.00, 1.00, 1.00, 0.67
    });
  ch = that->dict + '7';
  InitChar(
    ch, 2,
    (double[]){
      0.00, 1.00, 0.00, 1.00, 1.00, 1.00, 1.00, 1.00,
      1.00, 1.00, 1.00, 1.00, 0.33, 0.67, 0.33, 0.00
    });
  ch = that->dict + '8';
  InitChar(
    ch, 6,
    (double[]){
      0.50, 1.00, 1.00, 1.00, 1.00, 0.67, 0.50, 0.67,
      0.50, 0.67, 0.33, 0.67, 0.00, 0.50, 0.00, 0.33,
      0.00, 0.33, 0.00, 0.00, 0.33, 0.00, 0.50, 0.00,
      0.50, 0.00, 0.66, 0.00, 1.00, 0.00, 1.00, 0.33,
      1.00, 0.33, 1.00, 0.50, 0.66, 0.67, 0.50, 0.67,
      0.50, 0.67, 0.00, 0.67, 0.00, 1.00, 0.50, 1.00
    });
  ch = that->dict + '9';
  InitChar(
    ch, 5,
    (double[]){
      0.33, 0.00, 0.50, 0.00, 1.00, 0.00, 1.00, 0.50,
      1.00, 0.50, 1.00, 1.00, 0.66, 1.00, 0.50, 1.00,
      0.50, 1.00, 0.33, 1.00, 0.00, 1.00, 0.00, 0.67,
      0.00, 0.67, 0.00, 0.50, 0.33, 0.50, 0.50, 0.50,
      0.50, 0.50, 0.67, 0.50, 1.00, 0.50, 1.00, 0.67
    });
  ch = that->dict + '!';
  InitChar(
    ch, 3,
    (double[]){
      0.50, 0.18, 0.44, 0.18, 0.44, 0.07, 0.50, 0.07,
      0.50, 0.07, 0.56, 0.07, 0.56, 0.18, 0.50, 0.18,
      0.50, 1.00, 0.50, 1.00, 0.50, 0.33, 0.50, 0.33
    });
  ch = that->dict + '"';
  InitChar(
    ch, 2,
    (double[]){
      0.66, 1.00, 0.66, 1.00, 0.66, 0.75, 0.66, 0.75,
      0.33, 1.00, 0.33, 1.00, 0.33, 0.75, 0.33, 0.75
    });
  ch = that->dict + '\'';
  InitChar(
    ch, 1,
    (double[]){
      0.25, 1.00, 0.25, 1.00, 0.25, 0.49, 0.00, 0.50
    });
  ch = that->dict + '#';
  InitChar(
    ch, 4,
    (double[]){
      0.75, 1.00, 0.75, 1.00, 0.66, 0.00, 0.66, 0.00,
      0.33, 1.00, 0.33, 1.00, 0.25, 0.00, 0.25, 0.00,
      0.00, 0.25, 0.00, 0.25, 1.00, 0.25, 1.00, 0.25,
      0.00, 0.67, 0.00, 0.67, 1.00, 0.67, 1.00, 0.67
    });
  ch = that->dict + '$';
  InitChar(
    ch, 6,
    (double[]){
      0.50, 1.00, 0.50, 1.00, 0.50, 0.00, 0.50, 0.00,
      1.00, 0.83, 1.00, 0.99, 1.00, 1.00, 0.50, 1.00,
      0.50, 1.00, 0.00, 1.00, 0.00, 0.83, 0.00, 0.67,
      0.00, 0.67, 0.00, 0.50, 1.00, 0.67, 1.00, 0.50,
      1.00, 0.50, 1.00, 0.33, 1.00, 0.00, 0.50, 0.00,
      0.50, 0.00, 0.00, 0.00, 0.00, 0.16, 0.00, 0.33
    });
  ch = that->dict + '%';
  InitChar(
    ch, 9,
    (double[]){
      0.75, 0.50, 1.00, 0.50, 1.00, 0.50, 1.00, 0.25,
      1.00, 0.25, 1.00, 0.00, 1.00, 0.00, 0.75, 0.00,
      0.75, 0.00, 0.50, 0.00, 0.50, 0.00, 0.50, 0.25,
      0.50, 0.25, 0.50, 0.50, 0.50, 0.50, 0.75, 0.50,
      0.25, 1.00, 0.50, 1.00, 0.50, 1.00, 0.50, 0.75,
      0.50, 0.75, 0.50, 0.50, 0.50, 0.50, 0.25, 0.50,
      0.25, 0.50, 0.00, 0.50, 0.00, 0.50, 0.00, 0.75,
      0.00, 0.75, 0.00, 1.00, 0.00, 1.00, 0.25, 1.00,
      0.00, 0.00, 0.00, 0.00, 1.00, 1.00, 1.00, 1.00
    });
  ch = that->dict + '&';
  InitChar(
    ch, 6,
    (double[]){
      1.00, 0.00, 0.76, 0.33, 0.50, 0.67, 0.50, 0.67,
      0.50, 0.67, 0.00, 0.66, 0.00, 1.00, 0.50, 1.00,
      0.50, 1.00, 1.00, 1.00, 1.00, 0.67, 0.50, 0.67,
      0.50, 0.67, 0.33, 0.67, 0.00, 0.50, 0.00, 0.33,
      0.00, 0.33, 0.00, 0.00, 0.33, 0.00, 0.50, 0.00,
      0.50, 0.00, 0.66, 0.00, 1.00, 0.17, 1.00, 0.50
    });
  ch = that->dict + '(';
  InitChar(
    ch, 1,
    (double[]){
      1.00, 1.00, 0.75, 0.75, 0.75, 0.25, 1.00, 0.00
    });
  ch = that->dict + ')';
  InitChar(
    ch, 1,
    (double[]){
      0.00, 1.00, 0.25, 0.75, 0.25, 0.25, 0.00, 0.00
    });
  ch = that->dict + '=';
  InitChar(
    ch, 2,
    (double[]){
      0.00, 0.33, 0.00, 0.33, 1.00, 0.33, 1.00, 0.33,
      0.00, 0.67, 0.00, 0.67, 1.00, 0.67, 1.00, 0.67
    });
  ch = that->dict + '~';
  InitChar(
    ch, 1,
    (double[]){
      0.00, 0.50, 0.33, 0.75, 0.66, 0.25, 1.00, 0.50
    });
  ch = that->dict + '`';
  InitChar(
    ch, 1,
    (double[]){
      0.75, 1.00, 0.75, 1.00, 0.75, 0.49, 1.00, 0.50
    });
  ch = that->dict + '{';
  InitChar(
    ch, 2,
    (double[]){
      1.00, 1.00, 0.75, 1.00, 1.00, 0.50, 0.75, 0.50,
      0.75, 0.50, 1.00, 0.50, 0.76, 0.00, 1.00, 0.00
    });
  ch = that->dict + '}';
  InitChar(
    ch, 2,
    (double[]){
      0.00, 1.00, 0.25, 1.00, 0.00, 0.50, 0.25, 0.50,
      0.25, 0.50, -0.02, 0.50, 0.25, 0.00, 0.00, 0.00
    });
  ch = that->dict + '*';
  InitChar(
    ch, 2,
    (double[]){
      0.00, 0.00, 0.00, 0.00, 1.00, 1.00, 1.00, 1.00,
      0.00, 1.00, 0.00, 1.00, 1.00, 0.00, 1.00, 0.00
    });
  ch = that->dict + '+';
  InitChar(
    ch, 2,
    (double[]){
      0.00, 0.50, 0.00, 0.50, 1.00, 0.50, 1.00, 0.50,
      0.50, 1.00, 0.50, 1.00, 0.50, 0.00, 0.50, 0.00
    });
  ch = that->dict + '<';
  InitChar(
    ch, 2,
    (double[]){
      1.00, 1.00, 1.00, 1.00, 0.00, 0.50, 0.00, 0.50,
      0.00, 0.50, 0.00, 0.50, 1.00, 0.00, 1.00, 0.00
    });
  ch = that->dict + '>';
  InitChar(
    ch, 2,
    (double[]){
      0.00, 1.00, 0.00, 1.00, 1.00, 0.50, 1.00, 0.50,
      1.00, 0.50, 1.00, 0.50, 0.00, 0.00, 0.00, 0.00
    });
  ch = that->dict + '?';
  InitChar(
    ch, 5,
    (double[]){
      0.00, 0.67, 0.00, 1.00, 0.34, 1.00, 0.50, 1.00,
      0.50, 1.00, 0.66, 1.00, 1.00, 1.00, 1.00, 0.67,
      1.00, 0.67, 1.00, 0.33, 0.50, 0.66, 0.50, 0.33,
      0.50, 0.18, 0.44, 0.18, 0.44, 0.07, 0.50, 0.07,
      0.50, 0.07, 0.56, 0.07, 0.56, 0.18, 0.50, 0.18
    });
  ch = that->dict + '.';
  InitChar(
    ch, 2,
    (double[]){
      0.13, 0.25, 0.00, 0.25, 0.00, 0.00, 0.13, 0.00,
      0.13, 0.00, 0.25, 0.00, 0.25, 0.25, 0.13, 0.25
    });
  ch = that->dict + ',';
  InitChar(
    ch, 1,
    (double[]){
      0.25, 0.18, 0.25, 0.18, 0.25, -0.33, 0.00, -0.32
    });
  ch = that->dict + '/';
  InitChar(
    ch, 1,
    (double[]){
      1.00, 1.00, 1.00, 1.00, 0.00, 0.00, 0.00, 0.00
    });
  ch = that->dict + '\\';
  InitChar(
    ch, 1,
    (double[]){
      0.00, 1.00, 0.00, 1.00, 1.00, 0.00, 1.00, 0.00
    });
  ch = that->dict + '[';
  InitChar(
    ch, 3,
    (double[]){
      1.00, 1.00, 1.00, 1.00, 0.75, 1.00, 0.75, 1.00,
      0.75, 1.00, 0.75, 1.00, 0.75, 0.00, 0.75, 0.00,
      0.75, 0.00, 0.75, 0.00, 1.00, 0.00, 1.00, 0.00
    });
  ch = that->dict + ']';
  InitChar(
    ch, 3,
    (double[]){
      0.00, 1.00, 0.00, 1.00, 0.25, 1.00, 0.25, 1.00,
      0.25, 1.00, 0.25, 1.00, 0.25, 0.0, 0.25, 0.0,
      0.25, 0.0, 0.25, 0.0, 0.00, 0.0, 0.00, 0.0
    });
  ch = that->dict + '-';
  InitChar(
    ch, 1,
    (double[]){
      0.00, 0.50, 0.00, 0.50, 1.00, 0.50, 1.00, 0.50
    });
  ch = that->dict + '|';
  InitChar(
    ch, 1,
    (double[]){
      0.50, 1.00, 0.50, 1.00, 0.50, 0.00, 0.50, 0.00
    });
  ch = that->dict + '_';
  InitChar(
    ch, 1,
    (double[]){
      0.00, 0.00, 0.00, 0.00, 1.00, 0.00, 1.00, 0.00,
    });
  ch = that->dict + ';';
  InitChar(
    ch, 3,
    (double[]){
      0.25, 0.47, 0.18, 0.47, 0.18, 0.36, 0.25, 0.36,
      0.25, 0.36, 0.30, 0.36, 0.30, 0.47, 0.25, 0.47,
      0.25, 0.18, 0.25, 0.18, 0.25, -0.33, 0.00, -0.32,
    });
  ch = that->dict + ':';
  InitChar(
    ch, 4,
    (double[]){
      0.50, 0.72, 0.44, 0.72, 0.44, 0.61, 0.50, 0.61,
      0.50, 0.61, 0.56, 0.61, 0.56, 0.72, 0.50, 0.72,
      0.50, 0.39, 0.44, 0.39, 0.44, 0.28, 0.50, 0.28,
      0.50, 0.28, 0.56, 0.28, 0.56, 0.39, 0.50, 0.39
    });
  ch = that->dict + 'a';
  InitChar(
    ch, 4,
    (double[]){
      0.66, 0.67, 0.25, 0.67, 0.00, 0.66, 0.00, 0.33,
      0.00, 0.33, 0.00, 0.00, 0.26, 0.01, 0.49, 0.01,
      0.49, 0.01, 0.74, 0.01, 0.75, 0.33, 0.75, 0.67,
      0.75, 0.67, 0.75, 0.25, 0.75, 0.01, 1.00, 0.00
    });
  ch = that->dict + 'b';
  InitChar(
    ch, 4,
    (double[]){
      0.00, 1.00, 0.00, 0.50, 0.00, 0.00, 0.50, 0.00,
      0.50, 0.00, 1.00, 0.00, 1.00, 0.33, 1.00, 0.50,
      1.00, 0.50, 1.00, 0.67, 0.59, 0.67, 0.42, 0.67,
      0.42, 0.67, 0.25, 0.67, 0.06, 0.58, 0.06, 0.33
    });
  ch = that->dict + 'c';
  InitChar(
    ch, 4,
    (double[]){
      1.00, 0.50, 1.00, 0.67, 0.67, 0.67, 0.50, 0.67,
      0.50, 0.67, 0.33, 0.67, 0.00, 0.66, 0.00, 0.33,
      0.00, 0.33, 0.00, 0.00, 0.34, 0.00, 0.50, 0.00,
      0.50, 0.00, 0.66, 0.00, 1.00, 0.00, 1.00, 0.25
    });
  ch = that->dict + 'd';
  InitChar(
    ch, 4,
    (double[]){
      1.00, 1.00, 1.01, 0.50, 1.00, 0.00, 0.50, 0.00,
      0.50, 0.00, 0.00, 0.00, 0.00, 0.33, 0.00, 0.50,
      0.00, 0.50, 0.00, 0.67, 0.44, 0.66, 0.59, 0.66,
      0.59, 0.66, 0.75, 0.66, 0.95, 0.59, 0.95, 0.34
    });
  ch = that->dict + 'e';
  InitChar(
    ch, 6,
    (double[]){
      1.00, 0.25, 1.00, 0.00, 0.66, 0.00, 0.50, 0.00,
      0.50, 0.00, 0.34, 0.00, 0.00, 0.00, 0.00, 0.33,
      0.00, 0.33, 0.00, 0.66, 0.33, 0.67, 0.50, 0.67,
      0.50, 0.67, 0.67, 0.67, 1.00, 0.67, 1.00, 0.50,
      1.00, 0.50, 1.00, 0.33, 0.67, 0.33, 0.50, 0.33,
      0.50, 0.33, 0.33, 0.33, 0.00, 0.33, 0.00, 0.33
    });
  ch = that->dict + 'f';
  InitChar(
    ch, 4,
    (double[]){
      0.00, 0.50, 0.00, 0.50, 0.66, 0.50, 0.66, 0.50,
      1.00, 0.75, 1.00, 1.00, 0.75, 1.00, 0.50, 1.00,
      0.50, 1.00, 0.25, 1.00, 0.25, 0.83, 0.25, 0.67,
      0.25, 0.67, 0.25, 0.50, 0.25, 0.00, 0.25, 0.00
    });
  ch = that->dict + 'g';
  InitChar(
    ch, 6,
    (double[]){
      1.00, 0.33, 1.00, 0.00, 0.67, 0.00, 0.50, 0.00,
      0.50, 0.00, 0.33, 0.00, 0.00, -0.01, 0.00, 0.33,
      0.00, 0.33, 0.00, 0.67, 0.25, 0.67, 0.50, 0.67,
      0.50, 0.67, 0.75, 0.67, 1.00, 0.66, 1.00, 0.33,
      1.00, 0.33, 1.00, 0.00, 1.00, -0.33, 0.50, -0.33,
      0.50, -0.33, 0.41, -0.33, 0.33, -0.33, 0.33, -0.33
    });
  ch = that->dict + 'h';
  InitChar(
    ch, 3,
    (double[]){
      0.00, 0.33, 0.25, 0.67, 1.00, 1.00, 1.00, 0.50,
      1.00, 0.50, 1.00, 0.25, 1.00, 0.00, 1.00, 0.00,
      0.00, 1.00, 0.00, 1.00, 0.00, 0.00, 0.00, 0.00
    });
  ch = that->dict + 'i';
  InitChar(
    ch, 5,
    (double[]){
      0.25, 0.87, 0.19, 0.87, 0.19, 0.76, 0.25, 0.76,
      0.25, 0.76, 0.31, 0.76, 0.31, 0.87, 0.25, 0.87,
      0.00, 0.00, 0.25, 0.00, 0.25, 0.42, 0.25, 0.50,
      0.25, 0.50, 0.25, 0.25, 0.26, 0.00, 0.50, 0.00,
      0.50, 0.00, 0.72, 0.00, 1.00, 0.00, 1.00, 0.00
    });
  ch = that->dict + 'j';
  InitChar(
    ch, 5,
    (double[]){
      0.75, 0.87, 0.69, 0.87, 0.69, 0.76, 0.75, 0.76,
      0.75, 0.76, 0.81, 0.76, 0.81, 0.87, 0.76, 0.87,
      0.00, 0.00, 0.00, -0.33, 0.33, -0.33, 0.50, -0.33,
      0.50, -0.33, 0.75, -0.33, 0.75, 0.33, 0.75, 0.65,
      0.75, 0.65, 0.75, 0.33, 0.76, 0.00, 1.00, 0.00
    });
  ch = that->dict + 'k';
  InitChar(
    ch, 4,
    (double[]){
      0.00, 0.50, 0.25, 0.67, 1.00, 0.75, 1.00, 0.50,
      1.00, 0.50, 1.00, 0.25, 0.50, 0.33, 0.00, 0.33,
      0.00, 0.33, 0.32, 0.33, 0.75, 0.25, 1.00, 0.00,
      0.00, 1.00, 0.00, 1.00, 0.00, 0.00, 0.00, 0.00
    });
  ch = that->dict + 'l';
  InitChar(
    ch, 6,
    (double[]){
      0.00, 0.00, 0.25, 0.00, 0.25, 0.34, 0.25, 0.50,
      0.25, 0.50, 0.25, 0.66, 0.25, 1.00, 0.50, 1.00,
      0.50, 1.00, 0.66, 1.00, 0.75, 1.00, 0.75, 0.76,
      0.75, 0.76, 0.75, 0.51, 0.50, 0.33, 0.25, 0.33,
      0.25, 0.33, 0.26, 0.00, 0.33, 0.00, 0.66, 0.00,
      0.66, 0.00, 0.76, 0.00, 1.00, 0.00, 1.00, 0.00
    });
  ch = that->dict + 'm';
  InitChar(
    ch, 5,
    (double[]){
      0.00, 0.67, 0.00, 0.67, 0.00, 0.00, 0.00, 0.00,
      0.00, 0.25, 0.00, 0.59, 0.25, 0.67, 0.33, 0.67,
      0.33, 0.67, 0.50, 0.66, 0.50, 0.00, 0.50, 0.00,
      0.50, 0.00, 0.50, 0.00, 0.50, 0.67, 0.74, 0.67,
      0.74, 0.67, 1.00, 0.67, 1.00, 0.00, 1.00, 0.00
    });
  ch = that->dict + 'n';
  InitChar(
    ch, 3,
    (double[]){
      0.00, 0.67, 0.00, 0.67, 0.00, 0.00, 0.00, 0.00,
      0.00, 0.25, 0.00, 0.50, 0.25, 0.67, 0.66, 0.67,
      0.66, 0.67, 1.00, 0.67, 1.00, 0.24, 1.00, 0.00
    });
  ch = that->dict + 'o';
  InitChar(
    ch, 4,
    (double[]){
      0.50, 0.67, 1.00, 0.67, 1.00, 0.66, 1.00, 0.33,
      1.00, 0.33, 1.00, 0.00, 1.00, 0.00, 0.50, 0.00,
      0.50, 0.00, 0.00, 0.00, 0.00, -0.01, 0.00, 0.33,
      0.00, 0.33, 0.00, 0.67, 0.00, 0.67, 0.50, 0.67
    });
  ch = that->dict + 'p';
  InitChar(
    ch, 5,
    (double[]){
      0.00, -0.33, 0.00, -0.33, 0.00, 0.16, 0.00, 0.33,
      0.00, 0.33, 0.00, 0.50, 0.00, 0.67, 0.50, 0.67,
      0.50, 0.67, 1.00, 0.67, 1.00, 0.50, 1.00, 0.33,
      1.00, 0.33, 1.00, 0.16, 1.00, 0.00, 0.50, 0.00,
      0.50, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00
    });
  ch = that->dict + 'q';
  InitChar(
    ch, 5,
    (double[]){
      1.00, 0.00, 1.00, 0.00, 0.75, 0.00, 0.50, 0.00,
      0.50, 0.00, 0.25, 0.00, 0.00, -0.01, 0.00, 0.33,
      0.00, 0.33, 0.00, 0.67, 0.25, 0.67, 0.50, 0.67,
      0.50, 0.67, 0.75, 0.67, 1.00, 0.66, 1.00, 0.33,
      1.00, 0.33, 1.00, 0.00, 1.00, -0.33, 1.00, -0.33
    });
  ch = that->dict + 'r';
  InitChar(
    ch, 2,
    (double[]){
      0.00, 0.67, 0.00, 0.67, 0.00, 0.00, 0.00, 0.00,
      0.00, 0.00, 0.25, 0.67, 1.00, 1.00, 1.00, 0.50
    });
  ch = that->dict + 's';
  InitChar(
    ch, 5,
    (double[]){
      1.00, 0.50, 1.00, 0.66, 1.00, 0.67, 0.50, 0.67,
      0.50, 0.67, 0.00, 0.67, 0.00, 0.66, 0.00, 0.50,
      0.00, 0.50, 0.00, 0.33, 1.00, 0.50, 1.00, 0.33,
      1.00, 0.33, 1.00, 0.16, 1.00, 0.00, 0.50, 0.00,
      0.50, 0.00, 0.00, 0.00, 0.00, 0.08, 0.00, 0.25
    });
  ch = that->dict + 't';
  InitChar(
    ch, 4,
    (double[]){
      0.00, 0.00, 0.25, 0.00, 0.25, 0.17, 0.25, 0.25,
      0.00, 0.67, 0.00, 0.67, 0.50, 0.67, 0.50, 0.67,
      0.25, 1.00, 0.25, 1.00, 0.25, 0.33, 0.25, 0.25,
      0.25, 0.25, 0.25, 0.01, 0.50, 0.00, 1.00, 0.00
    });
  ch = that->dict + 'u';
  InitChar(
    ch, 3,
    (double[]){
      0.00, 0.67, 0.00, 0.33, 0.00, 0.00, 0.50, 0.00,
      0.50, 0.00, 1.00, 0.00, 1.00, 0.33, 1.00, 0.67,
      1.00, 0.67, 1.00, 0.33, 1.00, 0.00, 1.00, 0.00
    });
  ch = that->dict + 'v';
  InitChar(
    ch, 2,
    (double[]){
      0.00, 0.67, 0.00, 0.67, 0.34, 0.00, 0.50, 0.00,
      0.50, 0.00, 0.66, 0.00, 1.00, 0.67, 1.00, 0.67
    });
  ch = that->dict + 'w';
  InitChar(
    ch, 4,
    (double[]){
      0.00, 0.67, 0.00, 0.67, 0.16, 0.00, 0.33, 0.00,
      0.33, 0.00, 0.50, 0.00, 0.50, 0.50, 0.50, 0.50,
      0.50, 0.50, 0.50, 0.50, 0.50, 0.00, 0.66, 0.00,
      0.66, 0.00, 0.82, 0.00, 1.00, 0.67, 1.00, 0.67
    });
  ch = that->dict + 'x';
  InitChar(
    ch, 4,
    (double[]){
      0.00, 0.00, 0.25, 0.00, 0.51, 0.24, 0.50, 0.33,
      0.50, 0.33, 0.50, 0.41, 0.76, 0.67, 1.00, 0.67,
      0.00, 0.67, 0.25, 0.67, 0.50, 0.41, 0.50, 0.33,
      0.50, 0.33, 0.50, 0.25, 0.75, 0.00, 1.00, 0.00
    });
  ch = that->dict + 'y';
  InitChar(
    ch, 3,
    (double[]){
      0.00, 0.67, 0.00, 0.67, 0.00, 0.00, 0.66, 0.00,
      1.00, 0.67, 1.00, 0.67, 0.82, 0.33, 0.66, 0.00,
      0.66, 0.00, 0.50, -0.33, 0.50, -0.33, 0.25, -0.33
    });
  ch = that->dict + 'z';
  InitChar(
    ch, 3,
    (double[]){
      0.00, 0.67, 0.00, 0.67, 1.00, 0.67, 1.00, 0.67,
      1.00, 0.67, 1.00, 0.50, 0.00, 0.25, 0.00, 0.00,
      0.00, 0.00, 0.00, 0.00, 1.00, 0.00, 1.00, 0.00
    });
  ch = that->dict + '@';
  InitChar(
    ch, 8,
    (double[]){
      0.61, 0.66, 0.36, 0.66, 0.21, 0.65, 0.21, 0.45,
      0.21, 0.45, 0.21, 0.25, 0.36, 0.25, 0.51, 0.25,
      0.51, 0.25, 0.66, 0.25, 0.67, 0.45, 0.67, 0.66,
      0.67, 0.66, 0.66, 0.40, 0.66, 0.25, 0.82, 0.25,
      0.82, 0.25, 0.97, 0.24, 0.94, 0.72, 0.75, 0.79,
      0.75, 0.79, 0.56, 0.85, 0.36, 0.84, 0.25, 0.78,
      0.25, 0.78, 0.03, 0.66, 0.05, 0.21, 0.25, 0.11,
      0.25, 0.11, 0.45, 0.01, 0.67, 0.07, 0.75, 0.13
    });
  ch = that->dict + '^';
  InitChar(
    ch, 2,
    (double[]){
      0.00, 0.75, 0.00, 0.75, 0.50, 1.00, 0.50, 1.00,
      0.50, 1.00, 0.50, 1.00, 1.00, 0.75, 1.00, 0.75
    });
}

// Convert a string to a CapyListBezierSpline
// Input:
//   text: the text to convert
// Output:
//   Return a list of Bezier spline.
static CapyListBezierSpline* TextToBezierSpline(char const* const text) {
  methodOf(CapyFont);

  // Create the result list
  CapyListBezierSpline* splines = CapyListBezierSplineAlloc();

  // Current position
  CapyVec pos = CapyVecCreate(2);

  // Up vector
  CapyVec up = CapyVecCreate(2);
  up.vals[0] = that->forward.vals[1];
  up.vals[1] = -that->forward.vals[0];

  // Variable to memorise the line index
  size_t iLine = 0;

  // Loop on the characters in the text
  char const* chr = text;
  while(*chr) {

    // Convert from char to subscript
    if(*chr < 0) raiseExc(CapyExc_InvalidElemIdx);
    ssize_t iDict = *chr;

    // Process special characters
    if(*chr == ' ' || *chr == '\t') {

      // Update the position
      pos.vals[0] += that->scale.vals[0] * that->spacing.vals[0];
    } else if(*chr == '\n') {

      // Update the position
      ++iLine;
      pos.vals[0] = 0.0;
      pos.vals[1] = that->scale.vals[1] * that->spacing.vals[1] * (double)iLine;

    // If the character is defined in the dictionary
    } else if(that->dict[iDict] != NULL) {

      // Get a clone of the spline for the character
      CapyBezierSpline* splineChr = $(that->dict[iDict], clone)();

      // Apply scaling, rotation, translation
      $(splineChr, scale)(&(that->scale));
      $(splineChr, translate)(&pos);
      double theta = atan2(that->forward.vals[1], that->forward.vals[0]);
      $(splineChr, rotate2D)(theta);

      // Append the transformed spline to the result array
      $(splines, add)(splineChr);

      // Update the position
      pos.vals[0] += that->scale.vals[0] * that->spacing.vals[0];
    }

    // Move to the next character
    ++chr;
  }

  // Free memory
  CapyVecDestruct(&pos);
  CapyVecDestruct(&up);

  // Return the result list of spline
  return splines;
}

// Free the memory used by a CapyFont
static void Destruct(void) {
  methodOf(CapyFont);
  if(that->dict) loop(i, 128) CapyBezierSplineFree(that->dict + i);
  free(that->dict);
  CapyVecDestruct(&(that->scale));
  CapyVecDestruct(&(that->forward));
  CapyVecDestruct(&(that->spacing));
  return;
}

// Create a CapyFont
// Output:
//   Return a CapyFont
CapyFont CapyFontCreate(void) {
  CapyFont that = {
    .scale = CapyVecCreate(2),
    .forward = CapyVecCreate(2),
    .spacing = CapyVecCreate(2),
    .destruct = Destruct,
    .textToBezierSpline = TextToBezierSpline,
  };
  that.scale.vals[0] = 1.0;
  that.scale.vals[1] = 1.0;
  that.forward.vals[0] = 1.0;
  that.forward.vals[1] = 0.0;
  that.spacing.vals[0] = 1.0;
  that.spacing.vals[1] = 1.0;
  CreateDefaultDictionary(&that);
  return that;
}

// Allocate memory for a new CapyFont and create it
// Output:
//   Return a CapyFont
// Exception:
//   May raise CapyExc_MallocFailed.
CapyFont* CapyFontAlloc(void) {
  CapyFont* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyFontCreate();
  return that;
}

// Free the memory used by a CapyFont* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyFont to free
void CapyFontFree(CapyFont** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
