// ---------------------------------- font.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_FONT_H
#define CAPY_FONT_H
#include "externalHeaders.h"
#include "cext.h"
#include "bezier.h"

// Description:
// Font class.

// Font object
typedef struct CapyFont {

  // Font dictionary (array of 128 Bezier splines for the 128 ASCII characters,
  // null for undefined charachters, ordered accordingly to ascii code)
  CapyBezierSpline** dict;

  // Scale (default (1,1))
  CapyVec scale;

  // Forward vector of the text (default (1,0))
  CapyVec forward;

  // Spacing of letter (default (1,1))
  CapyVec spacing;

  // Destructor
  void (*destruct)(void);

  // Convert a string to a CapyListBezierSpline
  // Input:
  //   text: the text to convert
  // Output:
  //   Return a list of Bezier spline.
  CapyListBezierSpline* (*textToBezierSpline)(char const* const text);
} CapyFont;

// Create a CapyFont
// Output:
//   Return a CapyFont
CapyFont CapyFontCreate(void);

// Allocate memory for a new CapyFont and create it
// Output:
//   Return a CapyFont
// Exception:
//   May raise CapyExc_MallocFailed.
CapyFont* CapyFontAlloc(void);

// Free the memory used by a CapyFont* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyFont to free
void CapyFontFree(CapyFont** const that);
#endif
