// ------------------------------ elo.c ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "frequentistHypothesisTesting.h"

// Check if an event matches the null hypothesis
// Input:
//   evt: the event
// Output:
//   Return true if the event matches, else false.
static bool IsEvtMatchingNullHypo(CapyDistEvt const* const evt) {
  methodOf(CapyFHT);
  bool checkNullHypo =
    $(that->nullHypo, isEvtInMostProbable)(evt, that->statisticalPower);
  bool checkAltHypo =
    $(that->altHypo, isEvtInMostProbable)(evt, 1.0 - that->pValue);
  return checkNullHypo && !checkAltHypo;
}

// Free the memory used by a CapyFHT* and reset '*that' to NULL
static void Destruct(void) {
  return;
}

// Create a CapyFHT
// Input:
//   statisticalPower: confidence that an event doesn't match the alternate
//                     hypothesis (in [0,1])
//             pValue: confidence that an event matches the null hypothesis
//                     (in [0,1])
//           nullHypo: distribution of the null hypothesis
//             altHyp: distribution of the alternate hypothesis
// Output:
//   Return a CapyFHT
CapyFHT CapyFHTCreate(
           double const statisticalPower,
           double const pValue,
  CapyDist const* const nullHypo,
  CapyDist const* const altHypo) {
  return (CapyFHT){
    .pValue = pValue,
    .statisticalPower = statisticalPower,
    .nullHypo = nullHypo,
    .altHypo = altHypo,
    .destruct = Destruct,
    .isEvtMatchingNullHypo = IsEvtMatchingNullHypo,
  };
}

// Allocate memory for a new CapyFHT and create it.
// Input:
//   statisticalPower: confidence that an event doesn't match the alternate
//                     hypothesis (in [0,1])
//             pValue: confidence that an event matches the null hypothesis
//                     (in [0,1])
//           nullHypo: distribution of the null hypothesis
//             altHyp: distribution of the alternate hypothesis
// Output:
//   Return a CapyFHT
// Exception:
//   May raise CapyExc_MallocFailed.
CapyFHT* CapyFHTAlloc(
           double const statisticalPower,
           double const pValue,
  CapyDist const* const nullHypo,
  CapyDist const* const altHypo) {
  CapyFHT* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyFHTCreate(statisticalPower, pValue, nullHypo, altHypo);
  return that;
}

//   that: a pointer to the CapyFHT to free
void CapyFHTFree(CapyFHT** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
