// -------------------------- galeshapleypairing.c -------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache info@baillehachepascal.dev
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "galeshapleypairing.h"
#include "sort.h"
#include "comparator.h"

// Comparator for sorting the preferences
typedef struct CapyGaleShapleyComparator {
  CapyComparatorDef;
  double const* weight;
} CapyGaleShapleyComparator;

// Evaluation function the comparator for sorting preferences
static int Eval(
  void const* a,
  void const* b) {
  methodOf(CapyGaleShapleyComparator);
  size_t const* elemA = a;
  size_t const* elemB = b;
  if(that->weight[*elemA] > that->weight[*elemB]) {
    return -1;
  } else if(that->weight[*elemA] < that->weight[*elemB]) {
    return 1;
  } else {
    return 0;
  }
}

// Find the best pairing for elements in two sets A and B of same size
// according to the Gale-Shapley algorithm.
// Input:
//   weightA: matrix representing the pairing preferences of elements in
//            set A.The value of the i-th row and j-th column is equal to
//            the preference of the i-th element in set A to be paired with
//            the j-th element in set B. The higher the stronger the
//            preference.
//   weightB: matrix representing the pairing preferences of elements in
//            set B.The value of the i-th row and j-th column is equal to
//            the preference of the i-th element in set B to be paired with
//            the j-th element in set A. The higher the stronger the
//            preference.
// Output:
//   Return an array of indices describing the pairing. The i-th element j of
//   the array indicates the i-th element in the set A is paired with the 
//   the j-th element in the set B. Note that the pairing is biased toward
//   preferences of set A.
static size_t* Run(
  CapyMat const* const weightA,
  CapyMat const* const weightB) {

  // Allocate memory for the result
  size_t* pairing = NULL;
  size_t nb = weightA->nbRow;
  safeMalloc(pairing, nb);
  if(pairing == NULL) return NULL;
  loop(i, nb) pairing[i] = nb;

  // Create the matrices of ordered preferences
  size_t* prefs = NULL;
  safeMalloc(prefs, nb * nb);
  if(prefs == NULL) return NULL;
  loop(i, nb) {
    loop(j, nb) prefs[i * nb + j] = j;
    CapyGaleShapleyComparator cmp =
      {.weight = weightA->vals + i * nb, .eval = Eval};
    CapyQuickSort(prefs + i * nb, sizeof(prefs[0]), nb, (CapyComparator*)&cmp);
  }

  // Create the vectors of current bests
  size_t* curBests[2] = {NULL, NULL};
  loop(i, 2) {
    safeMalloc(curBests[i], nb);
    loop(j, nb) curBests[i][j] = (i == 0 ? 0 : nb);
  }

  // Loop on the elements of set A until we have all the pairings
  size_t iElemA = 0;
  while(iElemA < nb) {

    // If that element is not paired
    if(pairing[iElemA] == nb) {

      // Get the index in set B of the current best for the elem in set A
      size_t candidate = prefs[iElemA * nb + curBests[0][iElemA]];

      // If the candidate in set B has not been paired
      if(curBests[1][candidate] == nb) {

        // Set the new pairing
        pairing[iElemA] = candidate;
        curBests[1][candidate] = iElemA;

        // Move to the next elem in set A
        ++iElemA;

      // Else, if the candidate in set B prefers the element in set A
      } else if(
        weightB->vals[candidate * nb + iElemA] >
        weightB->vals[candidate * nb + curBests[1][candidate]]
      ) {

        // Remove the previous pairing
        pairing[curBests[1][candidate]] = nb;
        curBests[0][curBests[1][candidate]] += 1;

        // Set the new pairing
        size_t jElemA = curBests[1][candidate];
        pairing[iElemA] = candidate;
        curBests[1][candidate] = iElemA;

        // Move back to the element which has lost its pairing
        iElemA = jElemA;

      // Else, the element in set A can't choose the element in set B
      } else {

        // Try the next preference of the element in set A
        curBests[0][iElemA] += 1;
      }

    // Else, that element in set A is paired
    } else {

      // Move to the next element in set A
      ++iElemA;
    }
  }

  // Free memory
  free(prefs);
  loop(i, 2) free(curBests[i]);

  // Return the best pairing
  return pairing;
}

// Free the memory used by a CapyGaleShapleyPairing
static void Destruct(void) {
  return;
}

// Create a CapyGaleShapleyPairing
// Output:
//   Return a CapyGaleShapleyPairing
CapyGaleShapleyPairing CapyGaleShapleyPairingCreate(void) {
  CapyGaleShapleyPairing that = {
    .destruct = Destruct,
    .run = Run,
  };
  return that;
}

// Allocate memory for a new CapyGaleShapleyPairing and create it
// Output:
//   Return a CapyGaleShapleyPairing
// Exception:
//   May raise CapyExc_MallocFailed.
CapyGaleShapleyPairing* CapyGaleShapleyPairingAlloc(void) {
  CapyGaleShapleyPairing* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyGaleShapleyPairingCreate();
  return that;
}

// Free the memory used by a CapyGaleShapleyPairing* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyGaleShapleyPairing to free
void CapyGaleShapleyPairingFree(CapyGaleShapleyPairing** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
