// ---------------------------- galeshapleypairing.h ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache info@baillehachepascal.dev
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_GALESHAPLEYPAIRING_H
#define CAPY_GALESHAPLEYPAIRING_H
#include "externalHeaders.h"
#include "cext.h"
#include "capymath.h"

// Description:
// GaleShapleyPairing class.

// GaleShapleyPairing object
typedef struct CapyGaleShapleyPairing {

  // Destructor
  void (*destruct)(void);

  // Find the best pairing for elements in two sets A and B of same size
  // according to the Gale-Shapley algorithm.
  // Input:
  //   weightA: matrix representing the pairing preferences of elements in
  //            set A.The value of the i-th row and j-th column is equal to
  //            the preference of the i-th element in set A to be paired with
  //            the j-th element in set B. The higher the stronger the
  //            preference.
  //   weightB: matrix representing the pairing preferences of elements in
  //            set B.The value of the i-th row and j-th column is equal to
  //            the preference of the i-th element in set B to be paired with
  //            the j-th element in set A. The higher the stronger the
  //            preference.
  // Output:
  //   Return an array of indices describing the pairing. The i-th element j of
  //   the array indicates the i-th element in the set A is paired with the 
  //   the j-th element in the set B. Note that the pairing is biased toward
  //   preferences of set A.
  size_t* (*run)(
    CapyMat const* const weightA,
    CapyMat const* const weightB);
} CapyGaleShapleyPairing;

// Create a CapyGaleShapleyPairing
// Output:
//   Return a CapyGaleShapleyPairing
CapyGaleShapleyPairing CapyGaleShapleyPairingCreate(void);

// Allocate memory for a new CapyGaleShapleyPairing and create it
// Output:
//   Return a CapyGaleShapleyPairing
// Exception:
//   May raise CapyExc_MallocFailed.
CapyGaleShapleyPairing* CapyGaleShapleyPairingAlloc(void);

// Free the memory used by a CapyGaleShapleyPairing* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyGaleShapleyPairing to free
void CapyGaleShapleyPairingFree(CapyGaleShapleyPairing** const that);
#endif
