// ---------------------------- geomap.c ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "geomap.h"

// Range of latitude/longitude
CapyDefRange(LongLat, CapyLongLat_t)

// Load the image of the map from a path.
// Input:
//   path: the path to the image
// Output:
//   The image is loaded. If a previous image existed it is freed.
static void LoadImgFromPath(char const* const path) {
  methodOf(CapyGeoMap);
  CapyImgFree(&(that->img));
  that->img = CapyImgLoadFromPath(path);
}

// Set the latitude range.
// Input:
//   range: the range
// Output:
//   The latitude range is updated.
static void SetLatitudeRange(CapyRangeLongLat const range) {
  methodOf(CapyGeoMap);
  loop(i, 2) that->rangeLatitude.vals[i] = range.vals[i];
}

// Set the longitude range.
// Input:
//   range: the range
// Output:
//   The longitude range is updated.
static void SetLongitudeRange(CapyRangeLongLat const range) {
  methodOf(CapyGeoMap);
  loop(i, 2) that->rangeLongitude.vals[i] = range.vals[i];
}

// Get the position in the image for a given latitude and longitude
// Input:
//   coord: the latitude and longitude
// Output:
//   Return a CapyImgPos.
static CapyImgPos GetImgPos(CapyGeoMapCoord const coord) {
  methodOf(CapyGeoMap);
  CapyImgPos pos = {0};
  CapyRangeDouble imgRange[2];
  CapyRangeDouble mapRange[2];
  imgRange[0].min = 0;
  imgRange[0].max = that->img->dims.width;
  mapRange[0].min = that->rangeLongitude.min;
  mapRange[0].max = that->rangeLongitude.max;
  imgRange[1].min = 0;
  imgRange[1].max = that->img->dims.height;
  mapRange[1].min = that->rangeLatitude.max;
  mapRange[1].max = that->rangeLatitude.min;
  loop(i, 2) {
    pos.coords[i] =
      (CapyImgPos_t)CapyLerp(coord.vals[i], mapRange + i, imgRange + i);
  }
  return pos;
}

// Get the latitude and longitude for a given position in the image
// Input:
//   pos: the position in the image
// Output:
//   Return a CapyGeoMapCoord.
static CapyGeoMapCoord GetImgCoord(CapyImgPos const pos) {
  methodOf(CapyGeoMap);
  CapyGeoMapCoord coord = {0};
  CapyRangeDouble imgRange[2];
  CapyRangeDouble mapRange[2];
  imgRange[0].min = 0;
  imgRange[0].max = that->img->dims.width;
  mapRange[0].min = that->rangeLongitude.min;
  mapRange[0].max = that->rangeLongitude.max;
  imgRange[1].min = 0;
  imgRange[1].max = that->img->dims.height;
  mapRange[1].min = that->rangeLatitude.max;
  mapRange[1].max = that->rangeLatitude.min;
  loop(i, 2) {
    coord.vals[i] = CapyLerp(pos.coords[i], imgRange + i, mapRange + i);
  }
  return coord;
}

// Convert a distance in meter to a distance in pixel in the image
// Input:
//   distMeter: the distance in meter
// Output:
//   Return the distance in pixel, assuming the same scale along latitude and
//   longitude.
static double ConvertMeterToPixel(double const distMeter) {
  methodOf(CapyGeoMap);

  // Convert from meter to degree
  double const degree = distMeter / CAPY_EARTH_ONE_DEGREE_DISTANCE;

  // Convert from degree to pixel
  double const dist =
    degree *
    (double)(that->img->dims.width) /
    (that->rangeLongitude.max - that->rangeLongitude.min);
  return dist;
}

// Convert a distance in meter to a distance in degree
// Input:
//   distMeter: the distance in meter
// Output:
//   Return the distance in degree, assuming the earth is a perfect sphere.
static double ConvertMeterToDegree(double const distMeter) {
  return distMeter / CAPY_EARTH_CIRCUMFERENCE * 360.0;
}

// Free the memory used by a CapyGeoMap
static void Destruct(void) {
  methodOf(CapyGeoMap);
  CapyImgFree(&(that->img));
  loop(i, 2) $(that->rangeLongLat + i, destruct)();
}

// Create a CapyGeoMap
// Output:
//   Return a CapyGeoMap
CapyGeoMap CapyGeoMapCreate(void) {
  CapyGeoMap that = {
    .img = NULL,
    .rangeLatitude = CapyRangeLongLatCreate(0.0, 0.0),
    .rangeLongitude = CapyRangeLongLatCreate(0.0, 0.0),
    .destruct = Destruct,
    .loadImgFromPath = LoadImgFromPath,
    .setLatitudeRange = SetLatitudeRange,
    .setLongitudeRange = SetLongitudeRange,
    .getImgPos = GetImgPos,
    .getImgCoord = GetImgCoord,
    .convertMeterToDegree = ConvertMeterToDegree,
    .convertMeterToPixel = ConvertMeterToPixel,
  };
  return that;
}

// Allocate memory for a new CapyGeoMap and create it
// Output:
//   Return a CapyGeoMap
// Exception:
//   May raise CapyExc_MallocFailed.
CapyGeoMap* CapyGeoMapAlloc(void) {
  CapyGeoMap* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyGeoMapCreate();
  return that;
}

// Free the memory used by a CapyGeoMap* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyGeoMap to free
void CapyGeoMapFree(CapyGeoMap** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
