// ---------------------------------- geomap.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_GEOMAP_H
#define CAPY_GEOMAP_H
#include "externalHeaders.h"
#include "cext.h"
#include "image.h"
#include "range.h"

// Approximate circumference of earth at the equator in meter
#define CAPY_EARTH_CIRCUMFERENCE 40075017.0

// Approximate distance in meter corresponding to one degree on the surface
// of earth at the equator
#define CAPY_EARTH_ONE_DEGREE_DISTANCE 111319.49166666667

// Description:
// GeoMap class.

// Type for latitude (north-south position in degree from the equator) and
// longitude (east-west position in degree from a reference meridian)
typedef double CapyLongLat_t;

// Latitude/longitude coordinate
typedef struct CapyGeoMapCoord {
  union {
    CapyLongLat_t vals[2];
    struct __attribute__((packed)) {CapyLongLat_t longitude, latitude;};
  };
} CapyGeoMapCoord;

// Range of latitude/longitude
CapyDecRange(LongLat, CapyLongLat_t)

// GeoMap object
typedef struct CapyGeoMap {

  // Image of the map (default: none). Assuming North at the top, South at the
  // bottom, West at the left, East at the right, Mercator projection.
  CapyImg* img;

  // Range of latitude and longitude (default: [0, 0])
  union {
    CapyRangeLongLat rangeLongLat[2];
    struct __attribute__((packed)) {
      CapyRangeLongLat rangeLongitude, rangeLatitude;
    };
  };

  // Destructor
  void (*destruct)(void);

  // Load the image of the map from a path.
  // Input:
  //   path: the path to the image
  // Output:
  //   The image is loaded. If a previous image existed it is freed.
  void (*loadImgFromPath)(char const* const path);

  // Set the latitude range.
  // Input:
  //   range: the range
  // Output:
  //   The latitude range is updated.
  void (*setLatitudeRange)(CapyRangeLongLat const range);

  // Set the longitude range.
  // Input:
  //   range: the range
  // Output:
  //   The longitude range is updated.
  void (*setLongitudeRange)(CapyRangeLongLat const range);

  // Get the position in the image for a given latitude and longitude
  // Input:
  //   coord: the latitude and longitude
  // Output:
  //   Return a CapyImgPos.
  CapyImgPos (*getImgPos)(CapyGeoMapCoord const coord);

  // Get the latitude and longitude for a given position in the image
  // Input:
  //   pos: the position in the image
  // Output:
  //   Return a CapyGeoMapCoord.
  CapyGeoMapCoord (*getImgCoord)(CapyImgPos const pos);

  // Convert a distance in meter to a distance in pixel in the image
  // Input:
  //   distMeter: the distance in meter
  // Output:
  //   Return the distance in pixel, assuming the same scale along latitude and
  //   longitude.
  double (*convertMeterToPixel)(double const distMeter);

  // Convert a distance in meter to a distance in degree
  // Input:
  //   distMeter: the distance in meter
  // Output:
  //   Return the distance in degree, assuming the earth is a perfect sphere.
  double (*convertMeterToDegree)(double const distMeter);
} CapyGeoMap;

// Create a CapyGeoMap
// Output:
//   Return a CapyGeoMap
CapyGeoMap CapyGeoMapCreate(void);

// Allocate memory for a new CapyGeoMap and create it
// Output:
//   Return a CapyGeoMap
// Exception:
//   May raise CapyExc_MallocFailed.
CapyGeoMap* CapyGeoMapAlloc(void);

// Free the memory used by a CapyGeoMap* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyGeoMap to free
void CapyGeoMapFree(CapyGeoMap** const that);
#endif
