// --------------------------- geometricShape.c ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "geometricShape.h"
#include "mathfun.h"

// Get the area of the geometry
// Output:
//   Return the area
static double GetArea(void) {
  raiseExc(CapyExc_UndefinedExecution);
  assert(false && "Geometry2D.getArea is undefined.");
  return 0.0;
}

// Get the perimeter of the geometry
// Output:
//   Return the perimeter
static double GetPerimeter(void) {
  raiseExc(CapyExc_UndefinedExecution);
  assert(false && "Geometry2D.getPerimeter is undefined.");
  return 0.0;
}

// Convert the geometry into a Bezier spline
// Output:
//   Return the Bezier spline approximating the geometry.
static CapyBezierSpline GetBezierSpline(void) {
  raiseExc(CapyExc_UndefinedExecution);
  assert(false && "Geometry2D.getBezierSpline is undefined.");
  return (CapyBezierSpline){0};
}

// Free the memory used by a CapyGeometry2D
static void Geometry2DDestruct(void) {
  return;
}

// Create a CapyGeometry2D
// Output:
//   Return a CapyGeometry2D
CapyGeometry2D CapyGeometry2DCreate(void) {
  CapyGeometry2D that;
  that.destruct = Geometry2DDestruct;
  that.getArea = GetArea;
  that.getPerimeter = GetPerimeter;
  that.getBezierSpline = GetBezierSpline;
  return that;
}

// Get the area of the segment
// Output:
//   Return the area
static double Segment2DGetArea(void) {
  return 0.0;
}

// Get the perimeter of the segment
// Output:
//   Return the perimeter
static double Segment2DGetPerimeter(void) {
  return 0.0;
}

// Free the memory used by a CapySegment
static void Segment2DDestruct(void) {
  methodOf(CapySegment);
  $(that, destructCapyGeometry2D)();
}

// Convert the triangle into a Bezier spline
// Output:
//   Return the Bezier spline representing the triangle.
static CapyBezierSpline Segment2DGetBezierSpline(void) {
  methodOf(CapySegment);
  size_t nbSegment = 1;
  CapyBezierSpline spline = CapyBezierSplineCreate(&nbSegment, 1, 2);
  CapyBezier* curve = CapyBezierAlloc(1, 1, 2);
  double vals[2];
  loop(iCtrl, (size_t)2) {
    loop(i, 2) vals[i] = that->points[iCtrl].coords[i];
    $(curve, setCtrlById)(iCtrl, vals);
  }
  size_t idx = 0;
  $(&spline, setCurve)(&idx, curve);
  return spline;
}

// Create a CapySegment
// Output:
//   Return a CapySegment, points initialised to ((0, 0), (0, 1))
CapySegment CapySegmentCreate(void) {
  CapySegment that;
  CapyInherits(that, CapyGeometry2D, ());
  that.points[0] = (CapyPoint2D){{{0.0, 0.0}}};
  that.points[1] = (CapyPoint2D){{{0.0, 1.0}}};
  that.destruct = Segment2DDestruct;
  that.getArea = Segment2DGetArea;
  that.getPerimeter = Segment2DGetPerimeter;
  that.getBezierSpline = Segment2DGetBezierSpline;
  return that;
}

// Allocate memory for a new CapySegment and create it
// Output:
//   Return a CapySegment, points initialised to ((0, 0), (0, 1))
// Exception:
//   May raise CapyExc_MallocFailed.
CapySegment* CapySegmentAlloc(void) {
  CapySegment* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapySegmentCreate();
  return that;
}

// Free the memory used by a CapySegment* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapySegment to free
void CapySegmentFree(CapySegment** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Get the area of the triangle using Heron's formula
// Output:
//   Return the area
static double TriangleGetArea(void) {
  methodOf(CapyTriangle);
  double l[3] = {0.0, 0.0, 0.0};
  loop(i, 3) {
    loop(j, 2) {
      l[i] += pow(
        that->corners[i].coords[j] - that->corners[(i + 1) % 3].coords[j],
        2.0);
    }
    l[i] = sqrt(l[i]);
  }
  double s = 0.5 * (l[0] + l[1] + l[2]);
  double area = sqrt(s * (s - l[0]) * (s - l[1]) * (s - l[2]));
  return area;
}

// Get the perimeter of the triangle
// Output:
//   Return the perimeter
static double TriangleGetPerimeter(void) {
  methodOf(CapyTriangle);
  double perimeter = 0.0;
  loop(i, 2) {
    double l = 0.0;
    loop(j, 2) {
      double d =
        that->corners[(i + 1) % 2].coords[j] - that->corners[i].coords[j];
      l += d * d;
    }
    perimeter += sqrt(l);
  }
  return perimeter;
}

// Free the memory used by a CapyTriangle
static void TriangleDestruct(void) {
  methodOf(CapyTriangle);
  $(that, destructCapyGeometry2D)();
}

// Convert the triangle into a Bezier spline
// Output:
//   Return the Bezier spline representing the triangle.
static CapyBezierSpline TriangleGetBezierSpline(void) {
  methodOf(CapyTriangle);
  size_t nbSegment = 3;
  CapyBezierSpline spline = CapyBezierSplineCreate(&nbSegment, 1, 2);
  loop(iSeg, (size_t)3) {
    CapyBezier* curve = CapyBezierAlloc(1, 1, 2);
    double vals[2];
    loop(iCtrl, (size_t)2) {
      loop(i, 2) vals[i] = that->corners[(iSeg + iCtrl) % 3].coords[i];
      $(curve, setCtrlById)(iCtrl, vals);
    }
    size_t idx = iSeg;
    $(&spline, setCurve)(&idx, curve);
  }
  return spline;
}

// Get the barycentric coordinates of a position relative to the triangle
// Input:
//   pos: the position
//   coord: the barycentric coordinates
// Output:
//   'coord' is updated. If at least one of the three barycentric coordinates
//   is negative, the position is outside the triangle. Barycentric
//   coordinates in [-inf, 1].
static void GetBarycentricCoord(
  CapyPoint2D const* const pos,
             double* const coord) {
  methodOf(CapyTriangle);
  coord[0] =
    (
      (that->corners[1].y - that->corners[2].y) *
      (pos->x - that->corners[2].x) +
      (that->corners[2].x - that->corners[1].x) *
      (pos->y - that->corners[2].y)
    ) / (
      (that->corners[1].y - that->corners[2].y) *
      (that->corners[0].x - that->corners[2].x) +
      (that->corners[2].x - that->corners[1].x) *
      (that->corners[0].y - that->corners[2].y)
    );
  coord[1] =
    (
      (that->corners[2].y - that->corners[0].y) *
      (pos->x - that->corners[2].x) +
      (that->corners[0].x - that->corners[2].x) *
      (pos->y - that->corners[2].y)
    ) / (
      (that->corners[1].y - that->corners[2].y) *
      (that->corners[0].x - that->corners[2].x) +
      (that->corners[2].x - that->corners[1].x) *
      (that->corners[0].y - that->corners[2].y)
    );
  coord[2] = 1.0 - coord[0] - coord[1];
}

// Create a CapyTriangle
// Output:
//   Return a CapyTriangle, corners initialised to ((0, 0), (0, 1), (1, 0))
CapyTriangle CapyTriangleCreate(void) {
  CapyTriangle that;
  CapyInherits(that, CapyGeometry2D, ());
  that.corners[0] = (CapyPoint2D){{{0.0, 0.0}}};
  that.corners[1] = (CapyPoint2D){{{0.0, 1.0}}};
  that.corners[2] = (CapyPoint2D){{{1.0, 0.0}}};
  that.destruct = TriangleDestruct;
  that.getArea = TriangleGetArea;
  that.getPerimeter = TriangleGetPerimeter;
  that.getBezierSpline = TriangleGetBezierSpline;
  that.getBarycentricCoord = GetBarycentricCoord;
  return that;
}

// Allocate memory for a new CapyTriangle and create it
// Output:
//   Return a CapyTriangle, corners initialised to ((0, 0), (0, 1), (1, 0))
// Exception:
//   May raise CapyExc_MallocFailed.
CapyTriangle* CapyTriangleAlloc(void) {
  CapyTriangle* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyTriangleCreate();
  return that;
}

// Free the memory used by a CapyTriangle* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyTriangle to free
void CapyTriangleFree(CapyTriangle** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Get the area of the quadrilateral as the sum of the two subtriangles
// Output:
//   Return the area
static double QuadrilateralGetArea(void) {
  methodOf(CapyQuadrilateral);
  double area[2] = {0.0, 0.0};
  loop(i, 2) {
    CapyTriangle tri = CapyTriangleCreate();
    loop(j, 3) loop(k, 2) {
      tri.corners[j].coords[k] = that->corners[(j + i * 2) % 4].coords[k];
    }
    area[i] = $(&tri, getArea)();
    $(&tri, destruct)();
  }
  return area[0] + area[1];
}

// Get the perimeter of the quadrilateral
// Output:
//   Return the perimeter
static double QuadrilateralGetPerimeter(void) {
  methodOf(CapyQuadrilateral);
  double perimeter = 0.0;
  loop(i, 3) {
    double l = 0.0;
    loop(j, 2) {
      double d =
        that->corners[(i + 1) % 3].coords[j] - that->corners[i].coords[j];
      l += d * d;
    }
    perimeter += sqrt(l);
  }
  return perimeter;
}

// Functions to solve the bilinear inverse using Newton's method
typedef struct BilinearResidual {
  struct CapyMathFunDef;
  double const* pos;
  CapyQuadrilateral* quad;
  void (*destructCapyMathFun)(void);
} BilinearResidual;

static void BilinearResidualEval(
  double const* const in,
        double* const out) {
  BilinearResidual* that = (BilinearResidual*)capyThat;
  loop(i, 2) {
    out[i] =
      (
        (1.0 - in[1]) * (
          (1.0 - in[0]) * (that->quad->corners[0].coords[i]) +
          in[0] * that->quad->corners[1].coords[i]
        ) + in[1] * (
          (1.0 - in[0]) * (that->quad->corners[3].coords[i]) +
          in[0] * that->quad->corners[2].coords[i]
        )
      ) - that->pos[i];
  }
}

static void BilinearResidualEvalDerivative(
  double* const in,
   size_t const iDim,
  double* const out) {
  BilinearResidual* that = (BilinearResidual*)capyThat;
  if(iDim == 0) {
    loop(i, 2) {
      out[i] =
        that->quad->corners[1].coords[i] -
        that->quad->corners[0].coords[i] + (
          that->quad->corners[0].coords[i] -
          that->quad->corners[1].coords[i] -
          that->quad->corners[3].coords[i] +
          that->quad->corners[2].coords[i]
        ) * in[1];
    }
  } else {
    loop(i, 2) {
      out[i] =
        that->quad->corners[3].coords[i] -
        that->quad->corners[0].coords[i] + (
          that->quad->corners[0].coords[i] -
          that->quad->corners[1].coords[i] -
          that->quad->corners[3].coords[i] +
          that->quad->corners[2].coords[i]
        ) * in[0];
    }
  }
}

// Free a BilinearResidual
static void BilinearResidualDestruct(void) {
  methodOf(BilinearResidual);
  $(that, destructCapyMathFun)();
}

// Create a BilinearResidual
static BilinearResidual BilinearResidualCreate(
  CapyQuadrilateral* const quad,
       double const* const pos) {
  BilinearResidual that;
  CapyInherits(that, CapyMathFun, (2, 2));
  that.destruct = BilinearResidualDestruct;
  that.eval = BilinearResidualEval;
  that.evalDerivative = BilinearResidualEvalDerivative;
  that.quad = quad;
  that.pos = pos;
  return that;
}

// Get the bilinear coordinates of a point in the quadrilateral from
// its coordinates in world coordinate system
// Input:
//   pos: the position in world coordinate system
//   coords: double[2] updated with the bilinear coordinates
static void GetBilinearCoords(
  double const* const pos,
        double* const coords) {
  methodOf(CapyQuadrilateral);

  // Cf https://stackoverflow.com/questions/808441/
  //    inverse-bilinear-interpolation
  // Create the residual function
  BilinearResidual residual = BilinearResidualCreate(that, pos);

  // Solve the equation: residual(coords) = 0.0 . Starting with any
  // coords is fine, supposing the requested pos is in the quadrilateral
  // setting coords to (0.5, 0.5) seems legit.
  coords[0] = 0.5; coords[1] = 0.5;
  double out[2] = {0.0, 0.0};
  $(&residual, solveByNewtonMethod)(coords, out);

  // Free memory
  $(&residual, destruct)();
}

// Free the memory used by a CapyQuadrilateral
static void QuadrilateralDestruct(void) {
  methodOf(CapyQuadrilateral);
  $(that, destructCapyGeometry2D)();
}

// Convert the quadrilateral into a Bezier spline
// Output:
//   Return the Bezier spline representing the quadrilateral.
static CapyBezierSpline QuadrilateralGetBezierSpline(void) {
  methodOf(CapyQuadrilateral);
  size_t nbSegment = 4;
  CapyBezierSpline spline = CapyBezierSplineCreate(&nbSegment, 1, 2);
  loop(iSeg, (size_t)4) {
    CapyBezier* curve = CapyBezierAlloc(1, 1, 2);
    double vals[2];
    loop(iCtrl, (size_t)2) {
      loop(i, 2) vals[i] = that->corners[(iSeg + iCtrl) % 4].coords[i];
      $(curve, setCtrlById)(iCtrl, vals);
    }
    size_t idx = iSeg;
    $(&spline, setCurve)(&idx, curve);
  }
  return spline;
}

// Create a CapyQuadrilateral
// Output:
//   Return a CapyQuadrilateral, corners initialised to
//   ((0, 0), (1, 0), (1, 1), (0, 1))
CapyQuadrilateral CapyQuadrilateralCreate(void) {
  CapyQuadrilateral that;
  CapyInherits(that, CapyGeometry2D, ());
  that.corners[0] = (CapyPoint2D){{{0.0, 0.0}}};
  that.corners[1] = (CapyPoint2D){{{1.0, 0.0}}};
  that.corners[2] = (CapyPoint2D){{{1.0, 1.0}}};
  that.corners[3] = (CapyPoint2D){{{0.0, 1.0}}};
  that.destruct = QuadrilateralDestruct;
  that.getArea = QuadrilateralGetArea;
  that.getPerimeter = QuadrilateralGetPerimeter;
  that.getBilinearCoords = GetBilinearCoords;
  that.getBezierSpline = QuadrilateralGetBezierSpline;
  return that;
}

// Allocate memory for a new CapyQuadrilateral and create it
// Output:
//   Return a CapyQuadrilateral, corners initialised to
//   ((0, 0), (0, 1), (1, 1), (1, 0))
// Exception:
//   May raise CapyExc_MallocFailed.
CapyQuadrilateral* CapyQuadrilateralAlloc(void) {
  CapyQuadrilateral* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyQuadrilateralCreate();
  return that;
}

// Free the memory used by a CapyQuadrilateral* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyQuadrilateral to free
void CapyQuadrilateralFree(CapyQuadrilateral** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Free the memory used by a CapyRectangle
static void RectangleDestruct(void) {
  methodOf(CapyRectangle);
  $(that, destructCapyGeometry2D)();
}

// Convert the quadrilateral into a Bezier spline
// Output:
//   Return the Bezier spline representing the quadrilateral.
static CapyBezierSpline RectangleGetBezierSpline(void) {
  methodOf(CapyRectangle);
  size_t nbSegment = 4;
  CapyBezierSpline spline = CapyBezierSplineCreate(&nbSegment, 1, 2);
  loop(iSeg, (size_t)4) {
    CapyBezier* curve = CapyBezierAlloc(1, 1, 2);
    double vals[2];
    loop(iCtrl, (size_t)2) {
      size_t idx[5][2] = {{0, 0}, {1, 0}, {1, 1}, {0, 1}, {0, 0}};
      loop(i, 2) vals[i] = that->corners[idx[iSeg + iCtrl][i]].coords[i];
      $(curve, setCtrlById)(iCtrl, vals);
    }
    size_t idx = iSeg;
    $(&spline, setCurve)(&idx, curve);
  }
  return spline;
}

// Get the area of the rectangle
// Output:
//   Return the area
static double RectangleGetArea(void) {
  methodOf(CapyRectangle);
  double area =
    (that->corners[1].x - that->corners[0].x) *
    (that->corners[1].y - that->corners[0].y);
  return area;
}

// Get the perimeter of the rectangle
// Output:
//   Return the perimeter
static double RectangleGetPerimeter(void) {
  methodOf(CapyRectangle);
  double perimeter =
    2.0 * (
      that->corners[1].x - that->corners[0].x +
      that->corners[1].y - that->corners[0].y);
  return perimeter;
}

// Create a CapyRectangle
// Output:
//   Return a CapyRectangle, corners initialised to
//   ((0, 0), (1, 0), (1, 1), (0, 1))
CapyRectangle CapyRectangleCreate(void) {
  CapyRectangle that;
  CapyInherits(that, CapyGeometry2D, ());
  that.corners[0] = (CapyPoint2D){{{0.0, 0.0}}};
  that.corners[1] = (CapyPoint2D){{{1.0, 1.0}}};
  that.destruct = RectangleDestruct;
  that.getArea = RectangleGetArea;
  that.getPerimeter = RectangleGetPerimeter;
  that.getBezierSpline = RectangleGetBezierSpline;
  return that;
}

// Allocate memory for a new CapyRectangle and create it
// Output:
//   Return a CapyRectangle, corners initialised to
//   ((0, 0), (0, 1), (1, 1), (1, 0))
// Exception:
//   May raise CapyExc_MallocFailed.
CapyRectangle* CapyRectangleAlloc(void) {
  CapyRectangle* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyRectangleCreate();
  return that;
}

// Free the memory used by a CapyRectangle* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyRectangle to free
void CapyRectangleFree(CapyRectangle** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Get the area of the circle
// Output:
//   Return the area
static double CircleGetArea(void) {
  methodOf(CapyCircle);
  return M_PI * that->radius * that->radius;
}

// Get the perimeter of the circle
// Output:
//   Return the perimeter
static double CircleGetPerimeter(void) {
  methodOf(CapyCircle);
  return 2.0 * M_PI * that->radius;
}

// Free the memory used by a CapyCircle
static void CircleDestruct(void) {
  methodOf(CapyCircle);
  $(that, destructCapyGeometry2D)();
}

// Convert the circle into a Bezier spline
// Output:
//   Return the Bezier spline approximating the circle.
//   https://spencermortensen.com/articles/bezier-circle/
static CapyBezierSpline CircleGetBezierSpline(void) {
  methodOf(CapyCircle);
  size_t const nbSegment = 4;
  CapyBezierSpline spline = CapyBezierSplineCreate(&nbSegment, 1, 2);
  double c = 4.0 / 3.0 * (M_SQRT2 - 1.0);
  double coeff[4][4][2] = {
    {{0, 1}, {c, 1}, {1, c}, {1, 0}},
    {{1, 0}, {1, -c}, {c, -1}, {0, -1}},
    {{0, -1}, {-c, -1}, {-1, -c}, {-1, 0}},
    {{-1, 0}, {-1, c}, {-c, 1}, {0, 1}},
  };
  loop(iSeg, nbSegment) {
    CapyBezier* curve = CapyBezierAlloc(3, 1, 2);
    double vals[2];
    loop(iCtrl, (size_t)4) {
      loop(i, 2) {
        vals[i] = that->center.coords[i] + that->radius * coeff[iSeg][iCtrl][i];
      }
      $(curve, setCtrlById)(iCtrl, vals);
    }
    $(&spline, setCurve)(&iSeg, curve);
  }
  return spline;
}

// Get an arc of the circle as a Bezier spline
// Input:
//   pointA: the beginning of the arc
//   pointB: the end of the arc
// Output:
//   Return a Bezier spine approximating the arc from 'pointA' to 'pointB'
//   counter-clockwise. If the points are not on the circle, their projection
//   on the circle is used instead. Based on:
// https://stackoverflow.com/questions/734076/
// how-to-best-approximate-a-geometrical-arc-with-a-bezier-curve
static CapyBezierSpline GetArcAsBezierSpline(
  CapyPoint2D const* const pointA,
  CapyPoint2D const* const pointB) {
  methodOf(CapyCircle);

  // Project the points on the circle translated to the origin
  CapyVec u = CapyVecCreateLocal2D;
  CapyVec v = CapyVecCreateLocal2D;
  loop(i, 2) {
    u.vals[i] = pointA->coords[i] - that->center.coords[i];
    v.vals[i] = pointB->coords[i] - that->center.coords[i];
  }
  CapyVecNormalise(&u);
  CapyVecNormalise(&v);
  loop(i, 2) {
    u.vals[i] *= that->radius;
    v.vals[i] *= that->radius;
  }

  // Get the angles of the two points
  double angleA = CapyVec2DGetAngle(&u, &capyXAxis2D);
  if(angleA < 0.0) angleA = 2.0 * M_PI + angleA;
  double angleB = CapyVec2DGetAngle(&v, &capyXAxis2D);
  if(angleB < 0.0) angleB = 2.0 * M_PI + angleB;

  // Get the relative angle between the two points
  double theta = angleB - angleA;
  if(angleA > angleB) {
    theta = 2.0 * M_PI + theta;
  }

  // Variable to memorise the points at the extremities of the sub-arcs
  CapyPoint2D points[5] = {0};

  // Variable to memorise the number of points at the extremities of the
  // sub-arcs
  size_t nbPoint = 0;

  // Set the first point
  points[nbPoint] = (CapyPoint2D){.x = u.vals[0], .y = u.vals[1]};
  nbPoint += 1;

  // Set the mid points
  while(theta > M_PI_2 + 1e-9) {
    points[nbPoint] = (CapyPoint2D){
      .x = -points[nbPoint - 1].y,
      .y = points[nbPoint - 1].x,
    };
    nbPoint += 1;
    theta -= M_PI_2;
  }

  // Set the last point
  points[nbPoint] = (CapyPoint2D){.x = v.vals[0], .y = v.vals[1]};
  nbPoint += 1;

  // Translate the points back to the circle position
  loop(iPoint, nbPoint) {
    points[iPoint].x += that->center.coords[0];
    points[iPoint].y += that->center.coords[1];
  }

  // Create the Bezier cubic spline
  size_t nbSegment = nbPoint - 1;
  CapyBezierSpline spline = CapyBezierSplineCreate(&nbSegment, 1, 2);

  // Loop on the sub-arcs
  loop(iArc, nbSegment) {

    // Create the sub-arc to the spline
    CapyBezier* curve = CapyBezierAlloc(3, 1, 2);

    // Set the first and last controls
    $(curve, setCtrlById)(0, points[iArc].coords);
    $(curve, setCtrlById)(3, points[iArc + 1].coords);

    // Set the intermediate controls
    double vals[2];
    double ax = points[iArc].x - that->center.coords[0];
    double ay = points[iArc].y - that->center.coords[1];
    double bx = points[iArc + 1].x - that->center.coords[0];
    double by = points[iArc + 1].y - that->center.coords[1];
    double q1 = ax * ax + ay * ay;
    double q2 = q1 + ax * bx + ay * by;
    double k2 = (4.0/3.0) * (sqrt(2.0 * q1 * q2) - q2) / (ax * by - ay * bx);
    vals[0] = that->center.coords[0] + ax - k2 * ay;
    vals[1] = that->center.coords[1] + ay + k2 * ax;
    $(curve, setCtrlById)(1, vals);
    vals[0] = that->center.coords[0] + bx + k2 * by;
    vals[1] = that->center.coords[1] + by - k2 * bx;
    $(curve, setCtrlById)(2, vals);

    // Add the sub-arc to the spline
    $(&spline, setCurve)(&iArc, curve);
  }

  // Return the spline
  return spline;
}

// Create a CapyCircle
// Output:
//   Return a CapyCircle, center initialised to (0, 0) and radius
//   initialised to 1.
CapyCircle CapyCircleCreate(void) {
  CapyCircle that;
  CapyInherits(that, CapyGeometry2D, ());
  that.center = (CapyPoint2D){{{0.0, 0.0}}};
  that.radius = 1.0;
  that.destruct = CircleDestruct;
  that.getArea = CircleGetArea;
  that.getPerimeter = CircleGetPerimeter;
  that.getBezierSpline = CircleGetBezierSpline;
  that.getArcAsBezierSpline = GetArcAsBezierSpline;
  return that;
}

// Allocate memory for a new CapyCircle and create it
// Output:
//   Return a CapyCircle, center initialised to (0, 0) and radius
//   initialised to 1.
// Exception:
//   May raise CapyExc_MallocFailed.
CapyCircle* CapyCircleAlloc(void) {
  CapyCircle* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyCircleCreate();
  return that;
}

// Free the memory used by a CapyCircle* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyCircle to free
void CapyCircleFree(CapyCircle** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Get the intersecting point between two segments
// Inputs:
//   segmentA: first segment
//   segmentB: second segment
//   flagStrict: flag to restrict to the intersection inside the segments
// Output:
//   Update the intersection points of 'that'
static void GetIntersectSegments(
  CapySegment const* const segmentA,
  CapySegment const* const segmentB,
                bool const flagStrict) {
  methodOf(CapyGeometry2DIntersection);
  CapyMat m = CapyMatCreateLocal2x2;
  m.vals[0] = segmentA->points[0].x - segmentA->points[1].x;
  m.vals[1] = segmentB->points[1].x - segmentB->points[0].x;
  m.vals[2] = segmentA->points[0].y - segmentA->points[1].y;
  m.vals[3] = segmentB->points[1].y - segmentB->points[0].y;
  CapyMat mInv = CapyMatCreateLocal2x2;
  try {
    CapyMatInv(&m, &mInv);
  } catch(CapyExc_MatrixInversionFailed) {
    that->nbPoint = 0;
    return;
  } endCatch;
  CapyVec u = CapyVecCreateLocal2D;
  u.vals[0] = segmentB->points[1].x - segmentA->points[1].x;
  u.vals[1] = segmentB->points[1].y - segmentA->points[1].y;
  CapyVec v = CapyVecCreateLocal2D;
  CapyMatProdVec(&mInv, &u, &v);
  if(
    flagStrict && (
      v.vals[0] < 0.0 ||
      v.vals[0] > 1.0 ||
      v.vals[1] < 0.0 ||
      v.vals[1] > 1.0
    )
  ) {
    that->nbPoint = 0;
    return;
  }
  that->nbPoint = 1;
  loop(i, 2) {
    that->points[0].coords[i] =
      v.vals[0] * segmentA->points[0].coords[i] +
      (1.0 - v.vals[0]) * segmentA->points[1].coords[i];
  }
}

// Get the intersecting point(s) between two circles
// Inputs:
//   circleA: first circle
//   circleB: second circle
// Output:
//   Update the intersection points of 'circleA' and 'circleB'. Points are
//   ordered counter-clockwise on 'circleA' from the intersection of the
//   segment connecting the center the circles
static void GetIntersectCircles(
  CapyCircle const* const circleA,
  CapyCircle const* const circleB) {
  methodOf(CapyGeometry2DIntersection);
  double const d =
    sqrt(
      (circleA->center.x - circleB->center.x) *
      (circleA->center.x - circleB->center.x) +
      (circleA->center.y - circleB->center.y) *
      (circleA->center.y - circleB->center.y));
  if(equald(d, 0.0) || d > circleA->radius + circleB->radius) {
    that->nbPoint = 0;
  } else {
    double const l =
      (
        circleA->radius * circleA->radius - circleB->radius * circleB->radius +
        d * d
      ) / (
        2.0 * d
      );
    double const h = sqrt(circleA->radius * circleA->radius - l * l);
    that->points[0].x =
      l / d * (circleB->center.x - circleA->center.x) +
      h / d * (circleB->center.y - circleA->center.y) +
      circleA->center.x;
    that->points[0].y =
      l / d * (circleB->center.y - circleA->center.y) -
      h / d * (circleB->center.x - circleA->center.x) +
      circleA->center.y;
    if(equald(d, circleA->radius + circleB->radius)) {
      that->nbPoint = 1;
    } else {
      that->nbPoint = 2;
      that->points[1].x =
        l / d * (circleB->center.x - circleA->center.x) -
        h / d * (circleB->center.y - circleA->center.y) +
        circleA->center.x;
      that->points[1].y =
        l / d * (circleB->center.y - circleA->center.y) +
        h / d * (circleB->center.x - circleA->center.x) +
        circleA->center.y;

      // Order the points
      CapyVec u = CapyVecCreateLocal2D;
      CapyVec vA = CapyVecCreateLocal2D;
      CapyVec vB = CapyVecCreateLocal2D;
      loop(i, 2) {
        u.vals[i] = circleB->center.coords[i] - circleA->center.coords[i];
        vA.vals[i] = that->points[0].coords[i] - circleA->center.coords[i];
        vB.vals[i] = that->points[1].coords[i] - circleA->center.coords[i];
      }
      double thetaA = CapyVec2DGetAngle(&u, &vA);
      double thetaB = CapyVec2DGetAngle(&u, &vB);
      if(thetaA > thetaB) {
        CapyPoint2D pt = that->points[1];
        that->points[1] = that->points[0];
        that->points[0] = pt;
      }
    }
  }
}

// Get the intersecting point(s) between a segment and a circle
// Inputs:
//   seg: the segment
//   circle: the circle
//   flagStrict: flag to restrict to the intersection inside the segment
// Output:
//   Update the intersection points of 'seg' and 'circle'. Points are
//   ordered according to the parameter 't' of the segment.
static void GetIntersectSegmentCircle(
  CapySegment const* const seg,
   CapyCircle const* const circle,
                bool const flagStrict) {
  methodOf(CapyGeometry2DIntersection);
  double const Ax = seg->points[1].x - seg->points[0].x;
  double const Ay = seg->points[1].y - seg->points[0].y;
  double const Bx = seg->points[0].x - circle->center.x;
  double const By = seg->points[0].y - circle->center.y;
  double const coeffs[3] = {
    Bx * Bx + By * By - circle->radius * circle->radius,
    2.0 * (Ax * Bx + Ay * By),
    Ax * Ax + Ay * Ay,
  };
  double roots[2] = {0};
  bool hasSolution = CapySolveQuadratic(coeffs, roots);
  if(hasSolution) {
    that->nbPoint = 0;
    int nbRoot = 1 + (equald(roots[0], roots[1]) ? 0 : 1);
    loop(iRoot, nbRoot) {
      if(flagStrict == false || (roots[iRoot] >= 0.0 && roots[iRoot] <= 1.0)) {
        loop(iCoord, 2) {
          that->points[that->nbPoint].coords[iCoord] =
            (1.0 - roots[iRoot]) * seg->points[0].coords[iCoord] +
            roots[iRoot] * seg->points[1].coords[iCoord];
        }
        that->nbPoint += 1;
      }
    }
  } else {
    that->nbPoint = 0;
  }
}

// Free the memory used by a CapyCircle
static void Geometry2DIntersectionDestruct(void) {
  methodOf(CapyGeometry2DIntersection);
  *that = (CapyGeometry2DIntersection){0};
}

// Create a CapyGeometry2DIntersection
// Output:
//   Return a CapyGeometry2DIntersection initialised to 0
CapyGeometry2DIntersection CapyGeometry2DIntersectionCreate(void) {
  CapyGeometry2DIntersection that = {
    .nbPoint = 0,
    .destruct = Geometry2DIntersectionDestruct,
    .getIntersectSegments = GetIntersectSegments,
    .getIntersectCircles = GetIntersectCircles,
    .getIntersectSegmentCircle = GetIntersectSegmentCircle,
  };
  return that;
}
