// ----------------------------- geometricShape.h ----------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_GEOMETRICSHAPE_H
#define CAPY_GEOMETRICSHAPE_H
#include "externalHeaders.h"
#include "cext.h"
#include "bezier.h"

// Description:
// Basic geometric shape classes.

// Definition of the parent class for 2D geometric shapes
// Destructor
// void (*destruct)(void);
//
// Get the area of the geometry
// Output:
//   Return the area
// double (*getArea)(void);
//
// Get the perimeter of the geometry
// Output:
//   Return the perimeter
// double (*getPerimeter)(void);
//
// Convert the geometry into a Bezier spline
// Output:
//   Return the Bezier spline approximating the geometry.
// CapyBezierSpline (*getBezierSpline)(void);
#define CapyGeometry2D_ struct {             \
  void (*destruct)(void);                    \
  double (*getArea)(void);                   \
  double (*getPerimeter)(void);              \
  CapyBezierSpline (*getBezierSpline)(void); \
}

// Parent class for 2D geometric shapes
typedef CapyGeometry2D_ CapyGeometry2D;

// Create a CapyGeometry2D
// Output:
//   Return a CapyGeometry2D
CapyGeometry2D CapyGeometry2DCreate(void);

// Point
typedef struct CapyPoint2D {

  // Location of the corner
  union {
    double coords[2];
    struct __attribute__((packed)) {double x; double y;};
  };
} CapyPoint2D;

// Segment2D object
typedef struct CapySegment {

  // Inherits CapyGeometry2D
  CapyGeometry2D_;

  // Array of two positions, the locations of the segment extremities.
  CapyPoint2D points[2];

  // Destructor
  void (*destructCapyGeometry2D)(void);
} CapySegment;

// Create a CapySegment
// Output:
//   Return a CapySegment, points initialised to ((0, 0), (0, 1))
CapySegment CapySegmentCreate(void);

// Allocate memory for a new CapySegment and create it
// Output:
//   Return a CapySegment, points initialised to ((0, 0), (0, 1))
// Exception:
//   May raise CapyExc_MallocFailed.
CapySegment* CapySegmentAlloc(void);

// Free the memory used by a CapySegment* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapySegment to free
void CapySegmentFree(CapySegment** const that);

// Triangle object
typedef struct CapyTriangle {

  // Inherits CapyGeometry2D
  CapyGeometry2D_;

  // Array of three positions, the locations of the triangle's corners.
  CapyPoint2D corners[3];

  // Destructor
  void (*destructCapyGeometry2D)(void);

  // Get the barycentric coordinates of a position relative to the triangle
  // Input:
  //   pos: the position
  //   coord: the barycentric coordinates
  // Output:
  //   'coord' is updated. If at least one of the three barycentric coordinates
  //   is negative, the position is outside the triangle. Barycentric
  //   coordinates in [-inf, 1].
  void (*getBarycentricCoord)(
    CapyPoint2D const* const pos,
               double* const coord);
} CapyTriangle;

// Create a CapyTriangle
// Output:
//   Return a CapyTriangle, corners initialised to ((0, 0), (0, 1), (1, 0))
CapyTriangle CapyTriangleCreate(void);

// Allocate memory for a new CapyTriangle and create it
// Output:
//   Return a CapyTriangle, corners initialised to ((0, 0), (0, 1), (1, 0))
// Exception:
//   May raise CapyExc_MallocFailed.
CapyTriangle* CapyTriangleAlloc(void);

// Free the memory used by a CapyTriangle* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyTriangle to free
void CapyTriangleFree(CapyTriangle** const that);

// Quadrilateral object
typedef struct CapyQuadrilateral {

  // Inherits CapyGeometry2D
  CapyGeometry2D_;

  // Array of four positions, the location of the quadrilateral's
  // corners, in clockwise order.
  CapyPoint2D corners[4];

  // Destructor
  void (*destructCapyGeometry2D)(void);

  // Get the bilinear coordinates of a point in the quadrilateral from
  // its coordinates in world coordinate system
  // Input:
  //   pos: the position in world coordinate system
  //   coords: double[2] updated with the bilinear coordinates
  void (*getBilinearCoords)(
    double const* const pos,
          double* const coords);
} CapyQuadrilateral;

// Create a CapyQuadrilateral
// Output:
//   Return a CapyQuadrilateral, corners initialised to
//   ((0, 0), (0, 1), (1, 1), (1, 0))
CapyQuadrilateral CapyQuadrilateralCreate(void);

// Allocate memory for a new CapyQuadrilateral and create it
// Output:
//   Return a CapyQuadrilateral, corners initialised to
//   ((0, 0), (0, 1), (1, 1), (1, 0))
// Exception:
//   May raise CapyExc_MallocFailed.
CapyQuadrilateral* CapyQuadrilateralAlloc(void);

// Free the memory used by a CapyQuadrilateral* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyQuadrilateral to free
void CapyQuadrilateralFree(CapyQuadrilateral** const that);

// Rectangle object
typedef struct CapyRectangle {

  // Inherits CapyGeometry2D
  CapyGeometry2D_;

  // Array of two positions, the location of the quadrilateral's
  // corners (such as corners[0].x < corners[1].x and
  // corners[0].y < corners[1].y).
  CapyPoint2D corners[2];

  // Destructor
  void (*destructCapyGeometry2D)(void);
} CapyRectangle;

// Create a CapyRectangle
// Output:
//   Return a CapyRectangle, corners initialised to
//   ((0, 0), (0, 1), (1, 1), (1, 0))
CapyRectangle CapyRectangleCreate(void);

// Allocate memory for a new CapyRectangle and create it
// Output:
//   Return a CapyRectangle, corners initialised to
//   ((0, 0), (0, 1), (1, 1), (1, 0))
// Exception:
//   May raise CapyExc_MallocFailed.
CapyRectangle* CapyRectangleAlloc(void);

// Free the memory used by a CapyRectangle* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyRectangle to free
void CapyRectangleFree(CapyRectangle** const that);

// Circle object
typedef struct CapyCircle {

  // Inherits CapyGeometry2D
  CapyGeometry2D_;

  // Center
  CapyPoint2D center;

  // Radius
  double radius;

  // Destructor
  void (*destructCapyGeometry2D)(void);

  // Get an arc of the circle as a Bezier spline
  // Input:
  //   pointA: the beginning of the arc
  //   pointB: the end of the arc
  // Output:
  //   Return a Bezier spine approximating the arc from 'pointA' to 'pointB'
  //   counter-clockwise. If the points are not on the circle, their projection
  //   on the circle is used instead.
  CapyBezierSpline (*getArcAsBezierSpline)(
    CapyPoint2D const* const pointA,
    CapyPoint2D const* const pointB);
} CapyCircle;

// Create a CapyCircle
// Output:
//   Return a CapyCircle, center initialised to (0, 0) and radius
//   initialised to 1.
CapyCircle CapyCircleCreate(void);

// Allocate memory for a new CapyCircle and create it
// Output:
//   Return a CapyCircle, center initialised to (0, 0) and radius
//   initialised to 1.
// Exception:
//   May raise CapyExc_MallocFailed.
CapyCircle* CapyCircleAlloc(void);

// Free the memory used by a CapyCircle* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyCircle to free
void CapyCircleFree(CapyCircle** const that);

// Maximum number of intersecting points in a CapyGeometry2DIntersection
#define CAPY_NB_MAX_INTER_POINT_GEOMETRY 2

// Geometries intersection
typedef struct CapyGeometry2DIntersection {

  // Number of intersection point
  size_t nbPoint;

  // Intersection points
  CapyPoint2D points[CAPY_NB_MAX_INTER_POINT_GEOMETRY];

  // Destructor
  void (*destruct)(void);

  // Get the intersecting point between two segments
  // Inputs:
  //   segmentA: first segment
  //   segmentB: second segment
  //   flagStrict: flag to restrict to the intersection inside the segments
  // Output:
  //   Update the intersection points of 'segmentA' and 'segmentB'
  void (*getIntersectSegments)(
    CapySegment const* const segmentA,
    CapySegment const* const segmentB,
                  bool const flagStrict);

  // Get the intersecting point(s) between two circles
  // Inputs:
  //   circleA: first circle
  //   circleB: second circle
  // Output:
  // Output:
  //   Update the intersection points of 'circleA' and 'circleB'. Points are
  //   ordered counter-clockwise on 'circleA' from the intersection of the
  //   segment connecting the center the circles
  void (*getIntersectCircles)(
    CapyCircle const* const circleA,
    CapyCircle const* const circleB);

  // Get the intersecting point(s) between a segment and a circle
  // Inputs:
  //   seg: the segment
  //   circle: the circle
  //   flagStrict: flag to restrict to the intersection inside the segment
  // Output:
  //   Update the intersection points of 'seg' and 'circle'. Points are
  //   ordered according to the parameter 't' of the segment.
  void (*getIntersectSegmentCircle)(
    CapySegment const* const seg,
     CapyCircle const* const circle,
                  bool const flagStrict);
} CapyGeometry2DIntersection;

// Create a CapyGeometry2DIntersection
// Output:
//   Return a CapyGeometry2DIntersection initialised to 0
CapyGeometry2DIntersection CapyGeometry2DIntersectionCreate(void);
#endif
