// ------------------------------ graph.c ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "graph.h"

// Destruct a CapyGraphNode
static void GraphNodeDestruct(void) {
  methodOf(CapyGraph);
  (void)that;
}

// Create a CapyGraphNode
// Input:
//   id: id of the node in the graph
// Output:
//   Return the node.
CapyGraphNode CapyGraphNodeCreate(size_t id) {
  CapyGraphNode that = {
    .id = id, .pageRank = 0.0, .data = NULL,
    .destruct = GraphNodeDestruct,
  };
  return that;
}

// List of nodes
CapyDefList(CapyListGraphNode, CapyGraphNode)

// List of links
CapyDefList(CapyListGraphLink, CapyGraphLink)

// Add a node to the graph.
// Input:
//   node: the node to add
// Exception:
//   May return CapyExc_MallocFailed
static void AddNode(CapyGraphNode const node) {
  methodOf(CapyGraph);
  $(that->nodes, add)(node);
}

// Add a link to the graph.
// Input:
//   link: the link to add
// Exception:
//   May return CapyExc_MallocFailed
static void AddLink(CapyGraphLink const link) {
  methodOf(CapyGraph);
  $(that->links, add)(link);
}

// Link two nodes in the graph given their id.
// Input:
//   idA: ID of the first node
//   idB: ID of the second node
// Exception:
//   May return CapyExc_MallocFailed, CapyExc_InvalidNodeIdx
static void LinkNodes(
  size_t const idA,
  size_t const idB) {
  methodOf(CapyGraph);

  // Variable to memorise the link to add
  CapyGraphLink link = {.nodes = {NULL, NULL}};

  // Search the nodes of the link
  forEach(node, that->nodes->iter) {
    if(node.id == idA) link.nodes[0] = nodePtr;
    if(node.id == idB) link.nodes[1] = nodePtr;
  }

  // If the nodes weren't found, raise an exception
  if(link.nodes[0] == NULL || link.nodes[1] == NULL) {
    raiseExc(CapyExc_InvalidNodeIdx);
  }

  // Add the link
  $(that, addLink)(link);
}

// Get the index of a node in the list of nodes of a graph
// Input:
//   searchedNode: the node
// Output:
//   Return the index of the node, or raise CapyExc_InvalidNodeIdx
//   if the nodes wasn't found
static size_t GetIdxOfNode(CapyGraphNode const* const searchedNode) {
  methodOf(CapyGraph);
  forEach(node, that->nodes->iter) {
    if(nodePtr == searchedNode) return that->nodes->iter.idx;
  }
  raiseExc(CapyExc_InvalidNodeIdx);
  return 0;
}

// Update the connectivity matrix of the graph
// Exception:
//   May return CapyExc_MallocFailed, CapyExc_InvalidNodeIdx
static void UpdateConnectivityMat(void) {
  methodOf(CapyGraph);

  // Free the eventual previous matrix
  free(that->connectivityMat);

  // Allocate memory for the matrix
  size_t nbNode = $(that->nodes, getSize)();
  safeMalloc(that->connectivityMat, nbNode * nbNode);
  if(!(that->connectivityMat)) return;
  loop(i, nbNode * nbNode) that->connectivityMat[i] = false;

  // Loop on the links
  forEach(link, that->links->iter) {

    // Get the index of the nodes in the list of nodes
    size_t idx[2] = {0, 0};
    loop(i, 2) idx[i] = $(that, getIdxOfNode)(link.nodes[i]);

    // Update the connectivity matrix
    that->connectivityMat[idx[0] * nbNode + idx[1]] = true;
    if(that->directed == false) {
      that->connectivityMat[idx[1] * nbNode + idx[0]] = true;
    }
  }
}

// Update the link distance matrix of the graph based on the current
// connectivity matrix (update it with updateConnectivityMat if necessary
// before using this method)
// Exception:
//   May return CapyExc_MallocFailed
static void UpdateLinkDistanceMat(void) {
  methodOf(CapyGraph);

  // Free the eventual previous matrix
  free(that->linkDistanceMat);

  // Allocate memory for the matrix
  size_t nbNode = $(that->nodes, getSize)();
  safeMalloc(that->linkDistanceMat, nbNode * nbNode);

  // Initialise the matrix with the connectivity matrix
  loop(i, nbNode * nbNode) {
    that->linkDistanceMat[i] = (that->connectivityMat[i] ? 1 : 0);
  }

  // Loop until the distance matrix has stabilized
  bool flagStable = false;
  while(!flagStable) {
    flagStable = true;

    // Loop on the distance matrix
    loop(idxFrom, nbNode) loop(idxTo, nbNode) if(idxFrom != idxTo) {

      // If idxTo can be reached from idxFrom
      if(that->linkDistanceMat[idxFrom * nbNode + idxTo] > 0) {

        // Loop on the nodes reachable from idxTo
        loop(idxNext, nbNode) if(idxNext != idxFrom && idxNext != idxTo) {
          if(that->linkDistanceMat[idxTo * nbNode + idxNext] > 0) {

            // Update the distance from idxFrom to idxNextNode as necessary
            size_t dist =
              that->linkDistanceMat[idxFrom * nbNode + idxTo] +
              that->linkDistanceMat[idxTo * nbNode + idxNext];
            if(
              that->linkDistanceMat[idxFrom * nbNode + idxNext] == 0 ||
              that->linkDistanceMat[idxFrom * nbNode + idxNext] > dist
            ) {
              that->linkDistanceMat[idxFrom * nbNode + idxNext] = dist;
              flagStable = false;
            }
          }
        }
      }
    }
  }
}

// Run the PageRank algorithm to evaluate nodes in the graph
// Input:
//   nbIter: number of iteration of the algorithm
// Output:
//   The pageRank properties of nodes is updated
static void PageRank(size_t const nbIter) {
  methodOf(CapyGraph);

  // Get the number of node in the graph
  size_t nbNode = $(that->nodes, getSize)();

  // Coefficients for calculation
  double d = 0.85;
  double dp = (1.0 - d) / (double)nbNode;

  // Temporary vector for the calculation
  CapyVec rank = CapyVecCreate(nbNode);
  double initRank = 1.0 / (double)nbNode;
  loop(iNode, nbNode) rank.vals[iNode] = initRank;
  CapyVec newRank = CapyVecCreate(nbNode);

  // Calculate the connectivity matrix
  CapyMat mat = CapyMatCreate(nbNode, nbNode);
  $(that, updateConnectivityMat)();
  loop(iNode, nbNode) loop(jNode, nbNode) {
    if(that->connectivityMat[iNode * nbNode + jNode]) {
      mat.vals[iNode * nbNode + jNode] = 1.0;
    }
  }
  loop(jNode, nbNode) {
    double v = 0.0;
    loop(iNode, nbNode) v += mat.vals[iNode * nbNode + jNode];
    v = 1.0 / v;
    loop(iNode, nbNode) mat.vals[iNode * nbNode + jNode] *= v;
  }

  // Apply the algorithm
  loop(iIter, nbIter) {
    CapyMatProdVec(&mat, &rank, &newRank);
    loop(iNode, nbNode) rank.vals[iNode] = dp + d * newRank.vals[iNode];
  }

  // Set the node's pageRank property
  forEach(node, that->nodes->iter) {
    nodePtr->pageRank = rank.vals[that->nodes->iter.idx];
  }

  // Free memory
  CapyVecDestruct(&rank);
  CapyMatDestruct(&mat);
}

// Comparator for RemoveLinksFromNode
static int CmpLinkFirstNode(
  void const* a,
  void const* b) {
  CapyGraphLink const* const linkA = a;
  CapyGraphLink const* const linkB = b;
  if(linkA->nodes[0] == NULL) return -1;
  else if(linkB->nodes[0] == NULL) return -1;
  else if(linkA->nodes[0]->id < linkB->nodes[0]->id) return -1;
  else if(linkA->nodes[0]->id > linkB->nodes[0]->id) return 1;
  else return 0;
}

// Remove all links whose first node is equal to the given one
// Input:
//   id: id of the first node of the links to be removed
// Output:
//   All links whose first node is equal to the given one are removed
static void RemoveLinksFromNode(size_t const id) {
  methodOf(CapyGraph);
  CapyComparator cmp = CapyComparatorCreate();
  cmp.eval = CmpLinkFirstNode;
  CapyGraphNode* const node = $(that, getNodeById)(id);
  CapyGraphLink link = {.nodes = {node, NULL}};
  $(that->links, remove)(&cmp, &link);
  $(&cmp, destruct)();
}

// Comparator for GetNodeById
static int CmpNodeById(
  void const* a,
  void const* b) {
  CapyGraphNode const* const nodeA = a;
  CapyGraphNode const* const nodeB = b;
  if(nodeA->id < nodeB->id) return -1;
  else if(nodeA->id > nodeB->id) return 1;
  else return 0;
}

// Get a node by its id
// Input:
//   id: the node id
// Output:
//   Return the node, or null if the given id couldn't be found
static CapyGraphNode* GetNodeById(size_t const id) {
  methodOf(CapyGraph);
  CapyComparator cmp = CapyComparatorCreate();
  cmp.eval = CmpNodeById;
  CapyGraphNode node = {.id = id};
  CapyGraphNode* const res = $(that->nodes, find)(&cmp, &node);
  $(&cmp, destruct)();
  return res;
}

// Comparator for GetLinkBetweenNode
static int CmpLinkByNodesId(
  void const* a,
  void const* b) {
  CapyGraphLink const* const linkA = a;
  CapyGraphLink const* const linkB = b;
  if(
    linkA->nodes[0]->id == linkB->nodes[0]->id &&
    linkA->nodes[1]->id == linkB->nodes[1]->id
  ) {
    return 0;
  } else if(
    linkA->nodes[1]->id == linkB->nodes[0]->id &&
    linkA->nodes[0]->id == linkB->nodes[1]->id
  ) {
    return 0;
  } else return -1;
}

// Find a link between two nodes
// Input:
//   idA: the first node id
//   idB: the second node id
// Output:
//   Return the link, or null if there is no link between the two nodes.
//   The order of ids is irrelevant.
static CapyGraphLink* GetLinkBetweenNodes(
  size_t const idA,
  size_t const idB) {
  methodOf(CapyGraph);
  CapyComparator cmp = CapyComparatorCreate();
  cmp.eval = CmpLinkByNodesId;
  CapyGraphLink link;
  link.nodes[0] = $(that, getNodeById)(idA);
  link.nodes[1] = $(that, getNodeById)(idB);
  CapyGraphLink* const res = $(that->links, find)(&cmp, &link);
  $(&cmp, destruct)();
  return res;
}

// Free the memory used by a CapyGraph
static void Destruct(void) {
  methodOf(CapyGraph);
  CapyListGraphNodeFree(&(that->nodes));
  CapyListGraphLinkFree(&(that->links));
  free(that->connectivityMat);
  free(that->linkDistanceMat);
}

// Create a CapyGraph
// Output:
//   Return a CapyGraph
CapyGraph CapyGraphCreate(void) {
  return (CapyGraph){
    .nodes = CapyListGraphNodeAlloc(),
    .links = CapyListGraphLinkAlloc(),
    .directed = false,
    .connectivityMat = NULL,
    .linkDistanceMat = NULL,
    .destruct = Destruct,
    .addNode = AddNode,
    .addLink = AddLink,
    .linkNodes = LinkNodes,
    .getIdxOfNode = GetIdxOfNode,
    .updateConnectivityMat = UpdateConnectivityMat,
    .updateLinkDistanceMat = UpdateLinkDistanceMat,
    .pageRank = PageRank,
    .removeLinksFromNode = RemoveLinksFromNode,
    .getNodeById = GetNodeById,
    .getLinkBetweenNodes = GetLinkBetweenNodes,
  };
}

// Allocate memory for a new CapyGraph and create it
// Output:
//   Return a CapyGraph
// Exception:
//   May raise CapyExc_MallocFailed.
CapyGraph* CapyGraphAlloc(void) {
  CapyGraph* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyGraphCreate();
  return that;
}

// Free the memory used by a CapyGraph* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyGraph to free
void CapyGraphFree(CapyGraph** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
