// ---------------------------------- graph.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_GRAPH_H
#define CAPY_GRAPH_H
#include "externalHeaders.h"
#include "list.h"
#include "random.h"
#include "comparator.h"

// Description:
// Graph class.

// Node of a CapyGraph
typedef struct CapyGraphNode {

  // ID of the node
  size_t id;

  // Page rank value (default: 0.0, updated by CapyGraph.pageRank)
  double pageRank;

  // User's data
  void* data;

  // Destructor
  void (*destruct)(void);
} CapyGraphNode;

// Create a CapyGraphNode
// Input:
//   id: id of the node in the graph
// Output:
//   Return the node.
CapyGraphNode CapyGraphNodeCreate(size_t id);

// Declaration of a list of nodes
CapyDecList(CapyListGraphNode, CapyGraphNode)

// Link in a CapyGraph
typedef struct CapyGraphLink {

  // Pointer to the two nodes of the link
  CapyGraphNode* nodes[2];

  // User's data
  void* data;
} CapyGraphLink;

CapyDecList(CapyListGraphLink, CapyGraphLink)

// Graph object
typedef struct CapyGraph {

  // List of nodes
  CapyListGraphNode* nodes;

  // List of links
  CapyListGraphLink* links;

  // Flag to memorise if the graph is directed (default: false)
  bool directed;
  CapyPad(bool, directed);

  // Connectivity matrix (updated with updateConnectivityMat())
  // connectivityMat[i*nbNode+j] is true if there is a link from i to j
  bool* connectivityMat;

  // Link distance matrix (updated with updateLinkDistanceMat())
  // linkDistanceMat[i*nbNode+j] is the smallest number of steps (in term of
  // links) from i to j, 0 means there is no path from i to j
  size_t* linkDistanceMat;

  // Destructor
  void (*destruct)(void);

  // Add a node to the graph.
  // Input:
  //   node: the node to add
  // Exception:
  //   May return CapyExc_MallocFailed
  void (*addNode)(CapyGraphNode const node);

  // Add a link to the graph.
  // Input:
  //   link: the link to add
  // Exception:
  //   May return CapyExc_MallocFailed
  void (*addLink)(CapyGraphLink const link);

  // Link two nodes in the graph given their id.
  // Input:
  //   idA: ID of the first node
  //   idB: ID of the second node
  // Exception:
  //   May return CapyExc_MallocFailed, CapyExc_InvalidNodeIdx
  void (*linkNodes)(
    size_t const idA,
    size_t const idB);

  // Update the connectivity matrix of the graph
  // Exception:
  //   May return CapyExc_MallocFailed, CapyExc_InvalidNodeIdx
  void (*updateConnectivityMat)(void);

  // Get the index of a node in the list of nodes of a graph
  // Input:
  //   node: the node
  // Output:
  //   Return the index of the node, or raise CapyExc_InvalidNodeIdx
  //   if the nodes wasn't found
  size_t (*getIdxOfNode)(CapyGraphNode const* const node);

  // Update the link distance matrix of the graph based on the current
  // connectivity matrix (update it with updateConnectivityMat if necessary
  // before using this method)
  // Exception:
  //   May return CapyExc_MallocFailed
  void (*updateLinkDistanceMat)(void);

  // Run the PageRank algorithm to evaluate nodes in the graph
  // Input:
  //   nbIter: number of iteration of the algorithm
  // Output:
  //   The pageRank properties of nodes is updated
  void (*pageRank)(size_t const nbIter);

  // Remove all links whose first node is equal to the given one
  // Input:
  //   id: id of the first node of the links to be removed
  // Output:
  //   All links whose first node is equal to the given one are removed
  void (*removeLinksFromNode)(size_t const id);

  // Get a node by its id
  // Input:
  //   id: the node id
  // Output:
  //   Return the node, or null if the given id couldn't be found
  CapyGraphNode* (*getNodeById)(size_t const id);

  // Find a link between two nodes
  // Input:
  //   idA: the first node id
  //   idB: the second node id
  // Output:
  //   Return the link, or null if there is no link between the two nodes.
  //   The order of ids is irrelevant.
  CapyGraphLink* (*getLinkBetweenNodes)(
    size_t const idA,
    size_t const idB);
} CapyGraph;

// Create a CapyGraph
// Output:
//   Return a CapyGraph
CapyGraph CapyGraphCreate(void);

// Allocate memory for a new CapyGraph and create it
// Output:
//   Return a CapyGraph
// Exception:
//   May raise CapyExc_MallocFailed.
CapyGraph* CapyGraphAlloc(void);

// Free the memory used by a CapyGraph* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyGraph to free
void CapyGraphFree(CapyGraph** const that);
#endif
