// ---------------------------------- graphplotter.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_GRAPHPLOTTER_H
#define CAPY_GRAPHPLOTTER_H
#include "externalHeaders.h"
#include "cext.h"
#include "image.h"
#include "dataset.h"
#include "pen.h"
#include "geometricShape.h"

// Description:
// GraphPlotter class.

// Legend data
typedef struct CapyGraphPlotterLegendData {

  // Title for the abciss
  char const* titleAbciss;

  // Title for the ordinate
  char const* titleOrdinate;
} CapyGraphPlotterLegendData;

// Type of correlation used in plot
typedef enum CapyGraphPlotterTypeCorrelation {
  capyGraphPlotterTypeCorrelation_pearson,
  capyGraphPlotterTypeCorrelation_distance,
  capyGraphPlotterTypeCorrelation_nb,
} CapyGraphPlotterTypeCorrelation;

// GraphPlotter object
typedef struct CapyGraphPlotter {

  // Margin (in pixel, default 10)
  CapyImgPos_t margin;

  // Margin for the legend (in pixel, default 20). Space used beside the
  // axii for axis name and values
  CapyImgPos_t marginLegend;

  // Pen used to plot the data (default: default pen and blue color)
  CapyPen penData;

  // Pen used to draw axii and legend (default: default pen and black)
  CapyPen penLegend;

  // Flags for log base 10 scale
  union {
    bool logScales[2];
    struct __attribute__((packed)) {bool logScaleX, logScaleY;};
  };
  CapyPad(bool[2], logScales);

  // Destructor
  void (*destruct)(void);

  // Get the bounding box of the area containing data (without margin, axii, etc,
  // ...) for a given image.
  // Input:
  //   img: the image
  // Output:
  //   Return the coordinates (in pixel) as a CapyRectangle.
  CapyRectangle* (*getBoundingBoxDataArea)(CapyImg const* const img);

  // Draw the legend of a graph
  // Input:
  //   img: the image on which to draw
  //   legend: data about the legend
  // Output:
  //   The image is updated with the legend.
  void (*drawLegend)(
                             CapyImg* const img,
    CapyGraphPlotterLegendData const* const legend);

  // Plot the density distributions for one field from a CapyDataset.
  // Input:
  //   dataset: the CapyDataset
  //   iField: the index of the plotted field
  //   dims: the dimension of the graph
  //   nbBin: number of bins for the density distribution
  // Output:
  //   Return a CapyImg. The ordinate corresponds to the bins from the min to the
  //   max value of the plotted field and the abciss corresponds to the number
  //   of records in the dataset for that bin.
  CapyImg* (*plotDensityDistributions)(
    CapyDataset const* const dataset,
                size_t const iField,
    CapyImgDims const* const dims,
                size_t const nbBin);

  // Plot the density distributions for one categorical and one other
  // field from a CapyDataset.
  // Input:
  //   dataset: the CapyDataset
  //   iCatField: the index of the categorical field for filtering
  //   valCatField: the value of the categorical field for filtering
  //   iField: the index of the plotted field
  //   dims: the dimension of the graph
  //   nbBin: number of bins for the density distribution
  // Output:
  //   Return a CapyImg. The ordinate corresponds to the bins from the min to the
  //   max value of the plotted field and the abciss corresponds to the number
  //   of records in the dataset for that bin. Records are filtered on the value
  //   of the categorical field.
  CapyImg* (*plotDensityDistributionsGivenCatValue)(
    CapyDataset const* const dataset,
                size_t const iCatField,
           char const* const valCatField,
                size_t const iField,
    CapyImgDims const* const dims,
                size_t const nbBin);

  // Plot the density distributions of all fields in a CapyDataset filtered per
  // value of a given categorical field.
  // Input:
  //   dataset: the CapyDataset
  //   iCatField: the index of the categorical field
  //   dims: the dimension of one graph
  //   nbBin: number of bins for the density distribution
  // Output:
  //   Return a CapyImg containing the result graphs. Each graph represent
  //   the combination of one value of the categorical field and one other
  //   field. Categorical field values are aligned top-down and other fields
  //   values are aligned left-right. In one plot, the ordinate corresponds to
  //   the bins from the min to the max value of the numerical value and the
  //   abciss corresponds to the number of record in the dataset for that bin.
  CapyImg* (*plotAllDensityDistributionsGivenCatField)(
    CapyDataset const* const dataset,
                size_t const iCatField,
    CapyImgDims const* const dims,
                size_t const nbBin);

  // Plot the records of a dataset on a 2D graph using the values of two
  // given fields.
  // Input:
  //   dataset: the CapyDataset
  //   iField: the index of the field in abciss
  //   jField: the index of the field in ordinate
  //   dims: the dimension the graph
  // Output:
  //   Return a CapyImg containing the result graph.
  CapyImg* (*plotValuesForGivenPairOfFields)(
    CapyDataset const* const dataset,
                size_t const iField,
                size_t const jField,
    CapyImgDims const* const dims);

  // Plot the correlation graph between all pair of variables.
  // Input:
  //   dataset: the CapyDataset
  //   dims: the dimension of one graph
  //   typeCorrelation: type of correlation
  // Output:
  //   Return a CapyImg containing the result graphs. Each graph represent
  //   a pair of fields. Bottom left graphs contains a plot of records in
  //   the dataset.
  CapyImg* (*plotAllCorrelationGraph)(
                 CapyDataset const* const dataset,
                 CapyImgDims const* const dims,
    CapyGraphPlotterTypeCorrelation const typeCorrelation);

  // Plot an histogram.
  // Input:
  //   vals: values to plot
  //   dims: dimension of the result image
  //   legend: data for the legend
  // Output:
  //   Return an image of the plot of the histogram
  CapyImg* (*plotHistogram)(
                 CapyArrDouble const* const vals,
                   CapyImgDims const* const dims,
    CapyGraphPlotterLegendData const* const legend);
} CapyGraphPlotter;

// Create a CapyGraphPlotter
// Output:
//   Return a CapyGraphPlotter
CapyGraphPlotter CapyGraphPlotterCreate(void);

// Allocate memory for a new CapyGraphPlotter and create it
// Output:
//   Return a CapyGraphPlotter
// Exception:
//   May raise CapyExc_MallocFailed.
CapyGraphPlotter* CapyGraphPlotterAlloc(void);

// Free the memory used by a CapyGraphPlotter* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyGraphPlotter to free
void CapyGraphPlotterFree(CapyGraphPlotter** const that);
#endif
