// ------------------------------ greedy.c ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "greedy.h"
#include "comparator.h"
#include "random.h"

// Array of objects
CapyDefArray(CapyGreedyObjects, CapyGreedyObject)

// Comparator to sort the objects by gain
static int CmpGreedyObjects(
  void const* a,
  void const* b) {
  CapyGreedyObject const* objA = a;
  CapyGreedyObject const* objB = b;
  if(objA->gain < objB->gain) return 1;
  else if(objA->gain > objB->gain) return -1;
  else return 0;
}

// Run the Greedy algorithm to select the best set of objects amongst
// that->objects.
// Input:
//   budget: the budget allowed to select objects
// Output:
//   Return the total gain and cost , and that->objects is modified to
//   represent the selected objects and their quantity.
static CapyGreedyResult Select(double const budget) {
  methodOf(CapyGreedy);

  // Sort the list by gain per unit
  CapyComparator cmpObjects = CapyComparatorCreate();
  cmpObjects.eval = CmpGreedyObjects;
  $(&(that->objects), qsort)(&cmpObjects);
  $(&cmpObjects, destruct)();

  // Variable to memorise the total cost and total gain
  double totalCost = 0.0;
  double totalGain = 0.0;

  // Loop on the objects
  $(&(that->objects), initIterator)();
  forEach(object, that->objects.iter) {

    // Update the quantity of that object with as much it is possible to
    // take of it while staying in the limits of bugdet and available quantity
    if (totalCost < budget) {
      size_t qty = (size_t)floor((budget - totalCost) / object.cost);
      if (qty > object.qty) qty = object.qty;
      totalCost += object.cost * (double)qty;
      objectPtr->qty = qty;
      totalGain += object.gain * (double)qty;
    } else objectPtr->qty = 0;
  }
  return (CapyGreedyResult){.gain = totalGain, .cost = totalCost};
}

// Free the memory used by a CapyGreedy
static void Destruct(void) {
  methodOf(CapyGreedy);
  $(&(that->objects), destruct)();
}

// Create a CapyGreedy
// Output:
//   Return a CapyGreedy
CapyGreedy CapyGreedyCreate(void) {
  return (CapyGreedy){
    .objects = CapyGreedyObjectsCreate(0),
    .destruct = Destruct,
    .select = Select,
  };
}

// Allocate memory for a new CapyGreedy and create it
// Output:
//   Return a CapyGreedy
// Exception:
//   May raise CapyExc_MallocFailed.
CapyGreedy* CapyGreedyAlloc(void) {
  CapyGreedy* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyGreedyCreate();
  return that;
}

// Free the memory used by a CapyGreedy* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyGreedy to free
void CapyGreedyFree(CapyGreedy** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
