// ---------------------------------- greedy.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_GREEDY_H
#define CAPY_GREEDY_H
#include "externalHeaders.h"
#include "cext.h"
#include "array.h"

// Description:
// Class implementing the greedy algoirhtm.

// Structure to memorise object used by the Greedy algorithm
typedef struct CapyGreedyObject {

  // Gain per unit
  double gain;

  // Cost per unit
  double cost;

  // Quantity
  size_t qty;

  // Pointer or ID to identify the object
  union {void const* ptr; size_t id;};
} CapyGreedyObject;

CapyDecArray(CapyGreedyObjects, CapyGreedyObject)

// Structure to memorise the result (total gain and cost) of the greedy
// algorithm.
typedef struct CapyGreedyResult {

  // Gain
  double gain;

  // Cost
  double cost;
} CapyGreedyResult;

// Greedy class
typedef struct CapyGreedy {

  // Array of objects
  CapyGreedyObjects objects;

  // Destructor
  void (*destruct)(void);

  // Run the Greedy algorithm to select the best set of objects amongst
  // that->objects.
  // Input:
  //   budget: the budget allowed to select objects
  // Output:
  //   Return the total gain and cost , and that->objects is modified to
  //   represent the selected objects and their quantity.
  CapyGreedyResult (*select)(double const budget);
} CapyGreedy;

// Create a CapyGreedy
// Output:
//   Return a CapyGreedy
CapyGreedy CapyGreedyCreate(void);

// Allocate memory for a new CapyGreedy and create it
// Output:
//   Return a CapyGreedy
// Exception:
//   May raise CapyExc_MallocFailed.
CapyGreedy* CapyGreedyAlloc(void);

// Free the memory used by a CapyGreedy* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyGreedy to free
void CapyGreedyFree(CapyGreedy** const that);
#endif
