// ---------------------------- hashFun.c ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "hashFun.h"

// Evaluate the hash function for a given input.
// Input:
//   data: the input
//   sizeData: the size in byte of the data
// Output:
//   Return the hash.
static CapyHashFunValue_t HashFunEval(
  char const* const data,
       size_t const sizeData) {
  (void)data; (void)sizeData;
  raiseExc(CapyExc_UndefinedExecution);
  assert(false && "HashFun.eval is undefined.");
  return 0;
}

// Free the memory used by a CapyHashFun
static void HashFunDestruct(void) {
  return;
}

// Create a CapyHashFun
// Output:
//   Return a CapyHashFun
CapyHashFun CapyHashFunCreate(void) {
  CapyHashFun that = {
    .destruct = HashFunDestruct,
    .eval = HashFunEval,
  };
  return that;
}

// Allocate memory for a new CapyHashFun and create it
// Output:
//   Return a CapyHashFun
// Exception:
//   May raise CapyExc_MallocFailed.
CapyHashFun* CapyHashFunAlloc(void) {
  CapyHashFun* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyHashFunCreate();
  return that;
}

// Free the memory used by a CapyHashFun* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyHashFun to free
void CapyHashFunFree(CapyHashFun** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Evaluate the FNV1a hash function for a given input.
// Input:
//   data: the input
//   sizeData: the size in byte of the data
// Output:
//   Return the hash.
static CapyHashFunValue_t FNV1aEval(
  char const* const data,
       size_t const sizeData) {
  methodOf(CapyFNV1aHashFun);
  CapyHashFunValue_t hash = that->offset;
  loop(i, sizeData) {
    hash ^= (CapyHashFunValue_t)(data[i]);
    hash *= that->prime;
  }
  return hash;
}

// Free the memory used by a CapyFNV1aHashFun
static void FNV1aHashFunDestruct(void) {
  methodOf(CapyFNV1aHashFun);
  $(that, destructCapyHashFun)();
}

// Create a CapyFNV1aHashFun
// Output:
//   Return a CapyFNV1aHashFun
CapyFNV1aHashFun CapyFNV1aHashFunCreate(void) {
  CapyFNV1aHashFun that;
  CapyInherits(that, CapyHashFun, ());
  that.destruct = FNV1aHashFunDestruct;
  that.eval = FNV1aEval;
  that.offset = 0xcbf29ce484222325UL;
  that.prime = 0x100000001b3UL;
  return that;
}

// Allocate memory for a new CapyFNV1aHashFun and create it
// Output:
//   Return a CapyFNV1aHashFun
// Exception:
//   May raise CapyExc_MallocFailed.
CapyFNV1aHashFun* CapyFNV1aHashFunAlloc(void) {
  CapyFNV1aHashFun* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyFNV1aHashFunCreate();
  return that;
}

// Free the memory used by a CapyFNV1aHashFun* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyFNV1aHashFun to free
void CapyFNV1aHashFunFree(CapyFNV1aHashFun** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
