// ------------------------- hierarchicalclustering.h ------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache info@baillehachepascal.dev
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_HIERARCHICALCLUSTERING_H
#define CAPY_HIERARCHICALCLUSTERING_H
#include "externalHeaders.h"
#include "cext.h"
#include "dataset.h"
#include "tree.h"
#include "list.h"

// Description:
// HierarchicalClustering class.

// Clustering of a dataset as a tree where nodes' data are list of rows' index
CapyDecTree(CapyDatasetRowCluster, CapyListSize)

// HierarchicalClustering object
typedef struct CapyHierarchicalClustering {

  // Verbose stream (default: NULL)
  FILE* verboseStream;

  // Destructor
  void (*destruct)(void);

  // Cluster a dataset using the DIANA algorithm
  // Input:
  //   dataset: the dataset to cluster
  // Output:
  //   Return the clustering as a tree of dataset's rows. The clustering
  //   occurs based on input fields converted ot numerical values. The dataset
  //   can have output fields.
  CapyDatasetRowCluster* (*dianaClustering)(CapyDataset const* const dataset);

  // Cluster a dataset using a slightly faster version of the DIANA algorithm
  // Input:
  //   dataset: the dataset to cluster
  // Output:
  //   Return the clustering as a tree of dataset's rows. The clustering
  //   occurs based on input fields converted ot numerical values. The dataset
  //   can have output fields.
  CapyDatasetRowCluster* (*fastDianaClustering)(
    CapyDataset const* const dataset);
} CapyHierarchicalClustering;

// Create a CapyHierarchicalClustering
// Output:
//   Return a CapyHierarchicalClustering
CapyHierarchicalClustering CapyHierarchicalClusteringCreate(void);

// Allocate memory for a new CapyHierarchicalClustering and create it
// Output:
//   Return a CapyHierarchicalClustering
// Exception:
//   May raise CapyExc_MallocFailed.
CapyHierarchicalClustering* CapyHierarchicalClusteringAlloc(void);

// Free the memory used by a CapyHierarchicalClustering* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyHierarchicalClustering to free
void CapyHierarchicalClusteringFree(CapyHierarchicalClustering** const that);
#endif
