// ------------------------------ idxCombination.h -----------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "idxCombination.h"

// Reset the iterator
// Output:
//   Return the first element of the iteration
static size_t** Reset(void) {
  methodOf(CapyIdxCombination);
  loop(i, that->dim) that->indices[i] = that->ranges[i].min;
  that->indices[that->dim] = 0;
  that->idx = 0;
  return &(that->indices);
}

// Check if the iterator is on a valid combination
// Output:
//   Return true if the iterator is on a valid combination, else false
static bool IsActive(void) {
  methodOf(CapyIdxCombination);
  return (that->indices[that->dim] == 0);
}

// Move the iterator to the next combination
// Output:
//   Return the next combination of the iteration
static size_t** Next(void) {
  methodOf(CapyIdxCombination);
  ++(that->idx);
  size_t iDim = 0;
  that->indices[iDim] += 1;
  while(iDim < that->dim && that->indices[iDim] > that->ranges[iDim].max) {
    that->indices[iDim] = that->ranges[iDim].min;
    ++iDim;
    that->indices[iDim] += 1;
  }
  return &(that->indices);
}

// Free the memory used by a CapyIdxCombination
static void Destruct(void) {
  methodOf(CapyIdxCombination);
  free(that->indices);
  loop(iDim, that->dim) $(that->ranges + iDim, destruct)();
  free(that->ranges);
}

// Create a CapyIdxCombination
// Input:
//   dim: dimension of the combination
// Output:
//   Return a CapyIdxCombination
CapyIdxCombination CapyIdxCombinationCreate(size_t const dim) {
  CapyIdxCombination that = {
    .dim = dim,
    .idx = 0,
    .indices = NULL,
    .ranges = NULL,
    .datatype = NULL,
    .reset = Reset,
    .isActive = IsActive,
    .next = Next,
    .destruct = Destruct,
  };
  safeMalloc(that.indices, dim + 1);
  safeMalloc(that.ranges, dim);
  loop(iDim, dim + 1) that.indices[iDim] = 0;
  loop(iDim, dim) that.ranges[iDim] = CapyRangeSizeCreate(0, 1);
  return that;
}

// Allocate memory for a new CapyIdxCombination and create it
// Input:
//   dim: dimension of the combination
// Output:
//   Return a CapyIdxCombination
// Exception:
//   May raise CapyExc_MallocFailed.
CapyIdxCombination* CapyIdxCombinationAlloc(size_t const dim) {
  CapyIdxCombination* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyIdxCombinationCreate(dim);
  return that;
}

// Free the memory used by a CapyIdxCombination* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyIdxCombination to free
void CapyIdxCombinationFree(CapyIdxCombination** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
