// --------------------------------- image.c ----------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "image.h"
#include "imgKernel.h"
#include "random.h"
#include "pngFormat.h"
#include "streamIo.h"

// Predefined dimensions
CapyImgDims const capyImgDims_100x100 = {.width = 100, .height = 100};
CapyImgDims const capyImgDims_800x600 = {.width = 800, .height = 600};
CapyImgDims const capyImgDims_600x800 = {.width = 600, .height = 800};
CapyImgDims const capyImgDims_400x400 = {.width = 400, .height = 400};
CapyImgDims const capyImgDims_800x800 = {.width = 800, .height = 800};

// Range of positions in the image
CapyDefRange(ImgPos, CapyImgPos_t)

// Get the index of a pixel in an image given its 2D position
#define PosToIdx(img, x, y) \
  (size_t)((y) * (CapyImgPos_t)((img)->dims.width) + (x))

// Set of pixels
CapyDefArray(CapyImgPixels, CapyImgPixel)

// Reset the iterator
// Output:
//   Return the first pixel of the iteration
static CapyImgPixel* IterReset(void) {
  methodOf(CapyImgIterator);

  // Reset the index
  that->idx = 0;

  // Reset the current pixel
  that->pixel.idx = 0;
  that->pixel.pos.x = 0;
  that->pixel.pos.y = 0;

  // If the current pixel is in the image, return it
  // Else, return NULL
  if(that->img && $(that, isActive)()) {
    that->pixel.color = that->img->pixels;
    return &(that->pixel);
  }
  else return NULL;
}

// Move the iterator to the previous pixel
// Output:
//   Return the previous pixel of the iteration
static CapyImgPixel* IterPrev(void) {
  methodOf(CapyImgIterator);

  // Decrement the index
  --(that->idx);

  // Update the current pixel
  --(that->pixel.idx);
  --(that->pixel.pos.x);
  if(that->pixel.pos.x < 0) {
    that->pixel.pos.x = 0;
    --(that->pixel.pos.y);
  }
  that->pixel.color = that->img->pixels + that->pixel.idx;

  // If the current pixel is in the image, return it
  // Else, return NULL
  if($(that, isActive)()) return $(that, get)();
  else return NULL;
}

// Move the iterator to the next pixel
// Output:
//   Return the next pixel of the iteration
static CapyImgPixel* IterNext(void) {
  methodOf(CapyImgIterator);

  // Increment the index
  ++(that->idx);

  // Update the current pixel
  ++(that->pixel.idx);
  ++(that->pixel.pos.x);
  if(that->pixel.pos.x == (CapyImgPos_t)(that->img->dims.width)) {
    that->pixel.pos.x = 0;
    ++(that->pixel.pos.y);
  }
  that->pixel.color = that->img->pixels + that->pixel.idx;

  // If the current pixel is in the image, return it
  // Else, return NULL
  if($(that, isActive)()) return $(that, get)();
  else return NULL;
}

// Check if the iterator is on a valid element
// Output:
//   Return true if the iterator is on a valid element, else false
static bool IterIsActive(void) {
  methodOf(CapyImgIterator);

  // The iterator is active if its current pixel is in the image
  return (
    that->pixel.pos.x < (CapyImgPos_t)(that->img->dims.width) &&
    that->pixel.pos.x >= 0 &&
    that->pixel.pos.y < (CapyImgPos_t)(that->img->dims.height) &&
    that->pixel.pos.y >= 0);
}

// Get the current pixel of the iteration
// Output:
//   Return a pointer to the current pixel
static CapyImgPixel* IterGet(void) {
  methodOf(CapyImgIterator);

  // If the iterator is active, return the current pixel
  // Else, return NULL
  if($(that, isActive)()) return &(that->pixel);
  else return NULL;
}

// Set the type of the iterator and reset it
// Input:
//   type: the new type of the iterator
static void IterSetType(CapyImgIteratorType const type) {
  methodOf(CapyImgIterator);

  // Set the new type of the iterator
  that->type = type;

  // Reset the iterator
  $(that, reset)();
}

// Check if coordinates are inside the image
// Input:
//   coord: the coordinates to check
// Output:
//   Return true if the coordinates are inside the image, else false
static bool IsValidCoord(CapyImgPos const* const coord) {
  methodOf(CapyImg);
#if BUILD_MODE == 0
  assert(coord != NULL);
#endif
  return (
    coord->x < (CapyImgPos_t)(that->dims.width) &&
    coord->y < (CapyImgPos_t)(that->dims.height) &&
    coord->x >= 0 &&
    coord->y >= 0);
}

// Free the memory used by an iterator.
static void IteratorDestruct(void) {
  return;
}

// Create an iterator on a CapyImg
// Input:
//   img: the image on which to iterate
//   type: the type of iterator
// Output:
//   Return the iterator
CapyImgIterator CapyImgIteratorCreate(
             CapyImg* const img,
  CapyImgIteratorType const type) {

  // Create the new iterator
  CapyImgIterator that = {
    .idx = 0,
    .img = img,
    .type = type,
    .destruct = IteratorDestruct,
    .reset = IterReset,
    .prev = IterPrev,
    .next = IterNext,
    .isActive = IterIsActive,
    .get = IterGet,
    .setType = IterSetType,
  };

  // Reset the iterator
  $(&that, reset)();

  // Return the new iterator
  return that;
}

// Allocate memory and create an iterator on a CapyImg
// Input:
//   img: the image on which to iterate
//   type: the type of iterator
// Output:
//   Return the iterator
CapyImgIterator* CapyImgIteratorAlloc(
             CapyImg* const img,
  CapyImgIteratorType const type) {
#if BUILD_MODE == 0
  assert(img != NULL);
#endif

  // Declare the new iterator
  CapyImgIterator* that = NULL;

  // Allocate memory
  safeMalloc(that, 1);
  if(!that) return NULL;

  // Create the iterator
  *that = CapyImgIteratorCreate(img, type);

  // Return the new iterator
  return that;
}

// Free the memory used by a pointer to an iterator and reset '*that' to NULL
// Input:
//   that: a pointer to the iterator to free
void CapyImgIteratorFree(CapyImgIterator** const that) {

  // If the iterator is already freed, nothing to do
  if(that == NULL || *that == NULL) return;

  // Destruct the iterator
  $(*that, destruct)();

  // Free memory
  free(*that);
  *that = NULL;
}

// Get the color of a pixel at a given position
// Input:
//   pos: the position of the pixel
// Output:
//   Return the color of the pixel
static CapyColorData* GetColor(CapyImgPos const* const pos) {
  methodOf(CapyImg);
#if BUILD_MODE == 0
  assert(pos != NULL);
#endif
  return that->pixels + PosToIdx(that, pos->x, pos->y);
}

// Set the color of a pixel at a given position
// Input:
//   pos: the position of the pixel
//   color: the color to set
static void SetColor(
     CapyImgPos const* const pos,
  CapyColorData const* const color) {
  methodOf(CapyImg);
#if BUILD_MODE == 0
  assert(pos != NULL);
#endif
  CapyColorData safeColor = capyColorRGBAWhite;
  loop(i, 4) {
    safeColor.RGBA[i] = (
      color->RGBA[i] < 0.0 ? 0.0 :
      color->RGBA[i] > 1.0 ? 1.0 : color->RGBA[i]);
  }
  that->pixels[PosToIdx(that, pos->x, pos->y)] = safeColor;
}

// Initialise the iterator of the image, must be called after the creation
// of the image when it is created with Create(), Alloc() automatically
// initialise the iterator.
static void InitIterator(void) {
  methodOf(CapyImg);

  // Initialise the iterator
  that->iter =
    CapyImgIteratorCreate(that, capyImgIteratorType_LeftToRight_TopToBottom);
}

// Copy another image in the image
// Input:
//   img: the image to copy (must be of same dimensions)
static void Copy(CapyImg const* const img) {
  methodOf(CapyImg);
#if BUILD_MODE == 0
  assert(
    img->dims.width == that->dims.width &&
    img->dims.height == that->dims.height
  );
#endif

  // Copy the pixels
  memcpy(
    that->pixels,
    img->pixels,
    sizeof(CapyColorData) * that->dims.width * that->dims.height);

  // Copy the properties
  that->gamma = img->gamma;
  that->mode = img->mode;
}

// Get the number of pixels
// Output:
//   Return the number of pixels
static size_t GetNbPixels(void) {
  methodOf(CapyImg);
  return that->dims.width * that->dims.height;
}

// Change the gamma encoding of the image. Pixels value are updated.
// Input:
//   gamma: the new gamma value
static void SetGamma(double const gamma) {
  methodOf(CapyImg);

  // Calculate the conversion coefficient
  double coeff = gamma / that->gamma;

  // Loop on the pixels and convert the value
  size_t nbPixels = $(that, getNbPixels)();
  loop(iPixel, nbPixels) loop(i, 3) {
    if(that->pixels[iPixel].RGB[i] > 0.0) {
      that->pixels[iPixel].RGB[i] =
        pow(that->pixels[iPixel].RGB[i], coeff);
    }
  }

  // Update the gamma property
  that->gamma = gamma;
}

// Get the brightest pixel in the image
// Output:
//   Return the brightest (maximum r+g+b) pixel.
static CapyImgPixel GetBrightestPixel(void) {
  methodOf(CapyImg);

  // Variable to memorise the brightest pixel
  CapyImgPixel brightestPixel = {
    .color = NULL, .idx = 0., .pos = {.x = 0, .y = 0}
  };

  // Variable to memorise the maximum brightness
  double maxBrightness = 0.0;

  // Loop on the pixels
  forEach(pixel, that->iter) {

    // Get the brightness
    double brightness = 0.0;
    loop(i, 3) brightness += pixel.color->RGB[i];

    // Update the brightest pixel if necessary
    if(brightness > maxBrightness) {
      maxBrightness = brightness;
      brightestPixel = pixel;
    }
  }

  // Return the brightest pixel
  return brightestPixel;
}

// Get an edge map of the image.
// Input:
//   kernelSize: the size in pixel of the gaussian blur kernel, the
//               higher the less sensitive
//   stdDev: the standard deviation of the gaussian blur (as a guide,
//           kernelSize = 4 * stdDev)
//   iChannel: index of the channel used for edge detection
// Output:
//   Return an image of same dimensions where pixel values
//   represent the 'flatness' of the image. The red channel is the norm of
//   the derivative gaussian blur in x and y direction. The green and blue
//   channels contain the x and y coordinates of the normalised vector
//   indicating the direction of the slope (going up), the alpha channel is
//   set to 1.0.
// Exception:
//   May raise CapyExc_MallocFailed.
static CapyImg* GetEdgeMap(
   CapyImgDims_t const kernelSize,
          double const stdDev,
          size_t const iChannel) {
  methodOf(CapyImg);

  // Create the derivative gaussian blur images
  CapyImg* edge[2];
  CapyImgKernel* kernel = CapyImgKernelAlloc(kernelSize);
  loop(iAxis, (size_t)2) {
    $(kernel, setToDerivativeGaussianBlur)(stdDev, iAxis);
    edge[iAxis] = $(kernel, apply)(that);
  }
  CapyImgKernelFree(&kernel);

  // Create the result image and initialise it to black
  CapyImg* edgeMap = CapyImgClone(that);

  // Calculate the flatness of each pixel as the norm of the derivative
  // in both direction.
  forEach(pixel, edgeMap->iter) {
    if(
      pixel.pos.x < (CapyImgPos_t)(kernelSize) ||
      pixel.pos.x >= (CapyImgPos_t)(that->dims.width - kernelSize) ||
      pixel.pos.y < (CapyImgPos_t)(kernelSize) ||
      pixel.pos.y >= (CapyImgPos_t)(that->dims.height - kernelSize)
    ) {
      *(pixel.color) = capyColorRGBABlack;
    } else {
      double norm = 0.0;
      loop(j, 2) norm += pow(edge[j]->pixels[pixel.idx].RGBA[iChannel], 2.0);
      norm = sqrt(norm * 0.5);
      pixel.color->RGB[0] = norm;
      if(equal(norm, 0.0)) {
        pixel.color->RGB[1] = 0.0;
        pixel.color->RGB[2] = 0.0;
      } else {
        loop(j, 2) {
          pixel.color->RGB[1 + j] =
            edge[j]->pixels[pixel.idx].RGB[iChannel] / norm;
        }
      }
    }
  }

  // Free memory
  loop(i, 2) CapyImgFree(edge + i);

  // Return the result image
  return edgeMap;
}

// Get an edge map of the image based on its first 3 channels.
// Input:
//   kernelSize: the size in pixel of the gaussian blur kernel, the
//               higher the less sensitive
//   stdDev: the standard deviation of the gaussian blur (as a guide,
//           kernelSize = 4 * stdDev)
// Output:
//   Use getEdgeMap() to calculate the edge map for each of the three
//   first channels of the image and combined them into one single edge
//   map by choosing for each pixel the values in the edge map with
//   maximum slope at that pixel.
// Exception:
//   May raise CapyExc_MallocFailed.
static CapyImg* GetEdgeMap3Chan(
   CapyImgDims_t const kernelSize,
          double const stdDev) {
  methodOf(CapyImg);

  // Apply the derivative gaussian blur kernel to the image on all channels
  CapyImg* edgeMaps[3] = {NULL, NULL, NULL};
  loop(iChannel, (size_t)3) {
    edgeMaps[iChannel] =
      $(that, getEdgeMap)(kernelSize, stdDev, iChannel);
  }

  // Combined the three edge maps into one by selecting for each pixel the
  // strongest edges. Keep only the intensity value (set to RGB channels)
  CapyImg* edgeMap = CapyImgClone(that);
  size_t nbPixels = $(that, getNbPixels)();
  CapyColorData* edgeR = edgeMaps[0]->pixels;
  CapyColorData* edgeG = edgeMaps[1]->pixels;
  CapyColorData* edgeB = edgeMaps[2]->pixels;
  loop(iPixel, nbPixels) {
    if(edgeR[iPixel].RGB[0] > edgeB[iPixel].RGB[0]) {
      if(edgeR[iPixel].RGB[0] > edgeG[iPixel].RGB[0]) {
        loop(i, 3) edgeMap->pixels[iPixel].RGB[i] = edgeR[iPixel].RGB[0];
      } else {
        loop(i, 3) edgeMap->pixels[iPixel].RGB[i] = edgeG[iPixel].RGB[0];
      }
    } else if(edgeB[iPixel].RGB[0] > edgeG[iPixel].RGB[0]) {
      loop(i, 3) edgeMap->pixels[iPixel].RGB[i] = edgeB[iPixel].RGB[0];
    } else {
      loop(i, 3) edgeMap->pixels[iPixel].RGB[i] = edgeG[iPixel].RGB[0];
    }
  }

  // Free memory used by the temporary edge maps
  loop(i, 3) CapyImgFree(edgeMaps + i);

  // Return the result
  return edgeMap;
}

// Apply the hysteresis thresholding to an edge image in Canny edge
// detection method
// Input:
//   edge: the edge image to threshold
//   hysteresisLow, hysteresisHigh: the thresholds
static void CapyCannyHysteresis(
  CapyImg* const edge,
    double const hysteresisLow,
    double const hysteresisHigh) {

  // Loop on the pixels
  size_t nbPixels = $(edge, getNbPixels)();
  loop(iPixel, nbPixels) {

    // If that pixel has not been set yet
    if(edge->pixels[iPixel].vals[0] < 0.0) {

      // If it's an edge for sure
      if(edge->pixels[iPixel].vals[0] < -1.0 * hysteresisHigh) {
        loop(i, 3) edge->pixels[iPixel].vals[i] = 1.0;

      // Else, if it's not an edge for sure
      } else if(edge->pixels[iPixel].vals[0] > -1.0 * hysteresisLow) {
        loop(i, 3) edge->pixels[iPixel].vals[i] = 0.0;

      // Else, if its status is unsure
      } else {

        // Search for a neighour which is a sure edge
        bool hasNeighbourEdge = false;
        ldiv_t pos = ldiv((long int)iPixel, edge->dims.width);
        CapyImgPixel pixel = {
          .pos = (CapyImgPos){
            .x = (CapyImgPos_t)pos.rem, .y = (CapyImgPos_t)pos.quot
          },
          .idx = iPixel, .color = edge->pixels + iPixel
        };
        loop(iNeighbour, capyImgNeighbour_last) {
          CapyImgPixel neighbour = $(edge, getNeighbour)(&pixel, iNeighbour);
          if(
            neighbour.color != NULL &&
            (
              neighbour.color->vals[0] < -1.0 * hysteresisHigh ||
              neighbour.color->vals[0] > 0.5
            )
          ) {
            hasNeighbourEdge = true;
          }
        }

        // If it has a neighbour which is an edge for sure
        if(hasNeighbourEdge) {

          // Mark the pixel as an edge for sure
          loop(i, 3) edge->pixels[iPixel].vals[i] = 1.0;

          // Restart the check from the top left neighbour to propagate the
          // edge
          if(iPixel > edge->dims.width) iPixel -= edge->dims.width + 1;
          else iPixel = 0;
        }
      }
    }
  }

  // Loop again on the pixels and remove all the remaining unsure pixels
  loop(iPixel, nbPixels) {
    if(edge->pixels[iPixel].vals[0] < 0.0) {
      loop(i, 3) edge->pixels[iPixel].vals[i] = 0.0;
    }
  }
}

// Get an edge map of the image based on a given channel using the
// Canny edge detection method.
// Input:
//   iChan: the channel to use
//   kernelSize: the size in pixel of the gaussian blur kernel, the
//               higher the less sensitive
//   stdDev: the standard deviation of the gaussian blur (as a guide,
//           kernelSize = 4 * stdDev)
//   hysteresisLow, hysteresisHigh: the thresholds for the hysteresis phase,
//                                  in [0.0, 1.0] and scaled to the maximum
//                                  intensity after local maximum search phase
// Output:
//   Return an image of same dimension as 'that', where white pixels
//   indicates an edge
// Exception:
//   May raise CapyExc_MallocFailed.
static CapyImg* GetCannyEdgeMap(
          size_t const iChan,
   CapyImgDims_t const kernelSize,
          double const stdDev,
          double const hysteresisLow,
          double const hysteresisHigh) {
  methodOf(CapyImg);

  // Variable to memorise the result
  CapyImg* edge = CapyImgClone(that);

  // Initialise the result to black
  size_t nbPixels = $(edge, getNbPixels)();
  loop(iPixel, nbPixels) loop(i, 3) edge->pixels[iPixel].vals[i] = 0.0;

  // Variable to memorise the highest intensity
  double maxIntensity = 0.0;

  // Apply the derivative gaussian blur kernel to the image for that channel
  CapyImg* edgeMap = NULL;
  try {
    edgeMap = $(that, getEdgeMap)(kernelSize, stdDev, iChan);
  } catchDefault {
    CapyImgFree(&edge);
  } endCatch;
  CapyForwardExc();

  // Loop on the pixels
  forEach(pixel, edgeMap->iter) {

    // If there is a gradient for that pixel and channel
    if(pixel.color->vals[0] > 1e-18) {

      // Get the position of the two neighbours in the direction of the
      // gradient
      double pos[2][2];
      loop(iCoord, 2) {
        pos[0][iCoord] =
          pixel.pos.coords[iCoord] + 0.5 +
          edgeMap->pixels[edgeMap->iter.idx].vals[1 + iCoord];
        pos[1][iCoord] =
          pixel.pos.coords[iCoord] + 0.5 -
          edgeMap->pixels[edgeMap->iter.idx].vals[1 + iCoord];
      }

      // A pixel on the border of the image is considered as an edge by
      // default
      if(
        pos[0][0] < 0.0 || pos[1][0] < 0.0 ||
        pos[0][1] < 0.0 || pos[1][1] < 0.0 ||
        pos[0][0] >= that->dims.width - 1 ||
        pos[1][0] >= that->dims.width - 1 ||
        pos[0][1] >= that->dims.height - 1 ||
        pos[1][1] >= that->dims.height - 1
      ) {
        loop(i, 3) edge->pixels[edgeMap->iter.idx].vals[i] = 1.0;

      // Else, the pixel is not on the border of the image
      } else {

        // If the pixel is a local maximum,
        // the pixel is possibly an edge, set its value to the opposite of
        // intensity for the following step
        size_t iPixel[2] = {0, 0};
        loop(i, 2) {
          iPixel[i] =
            (size_t)pos[i][1] * (size_t)(edgeMap->dims.width) +
            (size_t)pos[i][0];
        }
        size_t idx = edgeMap->iter.idx;
        if(
          edgeMap->pixels[idx].vals[0] > edgeMap->pixels[iPixel[0]].vals[0] &&
          edgeMap->pixels[idx].vals[0] > edgeMap->pixels[iPixel[1]].vals[0]
        ) {
          edge->pixels[idx].vals[0] = -1.0 * edgeMap->pixels[idx].vals[0];
          if(maxIntensity < edgeMap->pixels[idx].vals[0]) {
            maxIntensity = edgeMap->pixels[idx].vals[0];
          }
        }
      }
    }
  }

  // Free memory used by the temporary edge maps
  CapyImgFree(&(edgeMap));

  // Apply hysteresis to the result
  CapyCannyHysteresis(
    edge, hysteresisLow * maxIntensity, hysteresisHigh * maxIntensity);

  // Return the result
  return edge;
}

// Get an edge map of the image based on its first 3 channels using the
// Canny edge detection method.
// Input:
//   kernelSize: the size in pixel of the gaussian blur kernel, the
//               higher the less sensitive
//   stdDev: the standard deviation of the gaussian blur (as a guide,
//           kernelSize = 4 * stdDev)
//   hysteresisLow, hysteresisHigh: the thresholds for the hysteresis phase,
//                                  in [0.0, 1.0] and scaled to the maximum
//                                  intensity after local maximum search phase
// Output:
//   Return an image of same dimension as 'that', where white pixels
//   indicates an edge
// Exception:
//   May raise CapyExc_MallocFailed.
static CapyImg* GetCannyEdgeMap3Chan(
   CapyImgDims_t const kernelSize,
          double const stdDev,
          double const hysteresisLow,
          double const hysteresisHigh) {
  methodOf(CapyImg);

  // Variable to memorise the result
  CapyImg* edge = CapyImgClone(that);

  // Initialise the result to black
  size_t nbPixels = $(edge, getNbPixels)();
  loop(iPixel, nbPixels) loop(i, 3) edge->pixels[iPixel].vals[i] = 0.0;

  // Variable to memorise the highest intensity
  double maxIntensity = 0.0;

  // Loop on the channels
  loop(iChannel, (size_t)3) {

    // Apply the derivative gaussian blur kernel to the image for that channel
    CapyImg* edgeMap = NULL;
    try {
      edgeMap = $(that, getEdgeMap)(kernelSize, stdDev, iChannel);
    } catchDefault {
      CapyImgFree(&edge);
    } endCatch;
    CapyForwardExc();

    // Loop on the pixels
    forEach(pixel, edgeMap->iter) {

      // If there is a gradient for that pixel and channel
      if(pixel.color->vals[0] > 1e-18) {

        // Get the position of the two neighbours in the direction of the
        // gradient
        double pos[2][2];
        loop(iCoord, 2) {
          pos[0][iCoord] =
            pixel.pos.coords[iCoord] + 0.5 +
            pixel.color->vals[1 + iCoord];
          pos[1][iCoord] =
            pixel.pos.coords[iCoord] + 0.5 -
            pixel.color->vals[1 + iCoord];
        }

        // A pixel on the border of the image is considered as an edge by
        // default
        if(
          pos[0][0] < 0.0 || pos[1][0] < 0.0 ||
          pos[0][1] < 0.0 || pos[1][1] < 0.0 ||
          pos[0][0] >= that->dims.width - 1||
          pos[1][0] >= that->dims.width - 1 ||
          pos[0][1] >= that->dims.height - 1 ||
          pos[1][1] >= that->dims.height - 1
        ) {
          loop(i, 3) edge->pixels[edgeMap->iter.idx].vals[i] = 1.0;
        }

        // Else, the pixel is not on the border of the image
        else {

          // If the pixel is a local maximum,
          // the pixel is possibly an edge, set it's value to the intensity
          // for the following step
          size_t iPixel[2] = {0, 0};
          loop(i, 2) {
            iPixel[i] =
              (size_t)pos[i][1] * (size_t)(edgeMap->dims.width) +
              (size_t)pos[i][0];
          }
          if(
            pixel.color->vals[0] > edgeMap->pixels[iPixel[0]].vals[0] &&
            pixel.color->vals[0] > edgeMap->pixels[iPixel[1]].vals[0]
          ) {
            size_t idx = edgeMap->iter.idx;
            if(
              edge->pixels[idx].vals[0] > -1.0 * pixel.color->vals[0]
            ) {
              edge->pixels[idx].vals[0] = -1.0 * pixel.color->vals[0];
              if(maxIntensity < pixel.color->vals[0]) {
                maxIntensity = pixel.color->vals[0];
              }
            }
          }
        }
      }
    }

    // Free memory used by the temporary edge maps
    CapyImgFree(&(edgeMap));
  }

  // Apply hysteresis to the result
  CapyCannyHysteresis(
    edge, hysteresisLow * maxIntensity, hysteresisHigh * maxIntensity);

  // Return the result
  return edge;
}

// Get an edge map of the image based on its first 3 channels using the
// local maximum gradient method.
// Input:
//   kernelSize: the size in pixel of the gaussian blur kernel, the
//               higher the less sensitive
//   stdDev: the standard deviation of the gaussian blur (as a guide,
//           kernelSize = 4 * stdDev)
//   threshold: minimum gradient to be marked as an edge, in [0, 1] of the
//              maximum gradient
// Output:
//   Return an image of same dimension as 'that', where white pixels
//   indicates an edge
// Exception:
//   May raise CapyExc_MallocFailed.
static CapyImg* GetLocalMaxGradientEdgeMap3Chan(
   CapyImgDims_t const kernelSize,
          double const stdDev,
          double const threshold) {
  methodOf(CapyImg);

  // Variable to memorise the result
  CapyImg* edge = CapyImgClone(that);

  // Variable to memorise the highest intensity
  double maxIntensity = 0.0;

  // Initialise the result to black
  size_t nbPixels = $(edge, getNbPixels)();
  loop(iPixel, nbPixels) loop(i, 3) edge->pixels[iPixel].vals[i] = 0.0;

  // Loop on the channels
  loop(iChannel, (size_t)3) {

    // Apply the derivative gaussian blur kernel to the image for that channel
    CapyImg* edgeMap = NULL;
    try {
      edgeMap = $(that, getEdgeMap)(kernelSize, stdDev, iChannel);
    } catchDefault {
      CapyImgFree(&edge);
    } endCatch;
    CapyForwardExc();

    // Loop on the pixels
    forEach(pixel, edgeMap->iter) {

      // If there is a gradient for that pixel and channel
      if(pixel.color->vals[0] > 1e-18) {

        // Get the position of the two neighbours in the direction of the
        // gradient
        double pos[2][2];
        loop(iCoord, 2) {
          pos[0][iCoord] =
            pixel.pos.coords[iCoord] + 0.5 +
            edgeMap->pixels[edgeMap->iter.idx].vals[1 + iCoord];
          pos[1][iCoord] =
            pixel.pos.coords[iCoord] + 0.5 -
            edgeMap->pixels[edgeMap->iter.idx].vals[1 + iCoord];
        }

        // A pixel on the border of the image is considered as an edge by
        // default
        if(
          pos[0][0] < 0.0 || pos[1][0] < 0.0 ||
          pos[0][1] < 0.0 || pos[1][1] < 0.0 ||
          pos[0][0] >= that->dims.width - 1 ||
          pos[1][0] >= that->dims.width - 1 ||
          pos[0][1] >= that->dims.height - 1 ||
          pos[1][1] >= that->dims.height - 1
        ) {
          loop(i, 3) edge->pixels[edgeMap->iter.idx].vals[i] = 1.0;
        }

        // Else, the pixel is not on the border of the image
        else {

          // If the pixel is a local maximum,
          // the pixel is an edge
          size_t iPixel[2] = {0, 0};
          loop(i, 2) {
            iPixel[i] =
              (size_t)pos[i][1] * (size_t)(edgeMap->dims.width) +
              (size_t)pos[i][0];
          }
          size_t idx = edgeMap->iter.idx;
          double val = edgeMap->pixels[idx].vals[0];
          if(
            val > edgeMap->pixels[iPixel[0]].vals[0] &&
            val > edgeMap->pixels[iPixel[1]].vals[0]
          ) {
            if(edge->pixels[idx].vals[0] < edgeMap->pixels[idx].vals[0]) {
              edge->pixels[idx].vals[0] = edgeMap->pixels[idx].vals[0];
              if(maxIntensity < edgeMap->pixels[idx].vals[0]) {
                maxIntensity = edgeMap->pixels[idx].vals[0];
              }
            }
          }
        }
      }
    }

    // Free memory used by the temporary edge maps
    CapyImgFree(&(edgeMap));
  }

  // Filter the results based on the thresold
  double minIntensity = maxIntensity * threshold;
  loop(iPixel, nbPixels) {
    if(edge->pixels[iPixel].vals[0] < minIntensity) {
      loop(i, 3) edge->pixels[iPixel].vals[i] = 0.0;
    } else {
      loop(i, 3) edge->pixels[iPixel].vals[i] = 1.0;
    }
  }

  // Return the result
  return edge;
}

// Convert an image to greyscale. RGB channel are replaced with
// (r+g+b)/3
static void FromRGBToGreyScale(void) {
  methodOf(CapyImg);

  // Loop on the pixels
  size_t nbPixels = $(that, getNbPixels)();
  loop(iPixel, nbPixels) {

    // Replace the RGB values with the brightness
    double brightness = 0.0;
    loop(i, 3) brightness += that->pixels[iPixel].RGB[i];
    brightness = brightness / 3.0;
    loop(i, 3) that->pixels[iPixel].RGB[i] = brightness;
  }
}

// Convert a RGB image to HSV
static void FromRGBToHSV(CapyImg* const that) {

  // CapyColor to perform the conversion
  CapyColor cvt = CapyColorCreate(capyColorRGBABlack);

  // Loop on the pixels and convert the RGB values
  size_t nbPixels = $(that, getNbPixels)();
  loop(iPixel, nbPixels) {
    cvt.vals = that->pixels[iPixel];
    that->pixels[iPixel] = $(&cvt, RGB2HSV)();
  }

  // Free memory
  $(&cvt, destruct)();
}

// Convert a RGB image to rgb (normalised RGB)
static void FromRGBTorgb(CapyImg* const that) {

  // CapyColor to perform the conversion
  CapyColor cvt = CapyColorCreate(capyColorRGBABlack);

  // Loop on the pixels and convert the RGB values
  size_t nbPixels = $(that, getNbPixels)();
  loop(iPixel, nbPixels) {
    cvt.vals = that->pixels[iPixel];
    that->pixels[iPixel] = $(&cvt, RGB2rgb)();
  }

  // Free memory
  $(&cvt, destruct)();
}

// Convert a RGB image to l1l2l3
static void FromRGBTol1l2l3(CapyImg* const that) {

  // CapyColor to perform the conversion
  CapyColor cvt = CapyColorCreate(capyColorRGBABlack);

  // Loop on the pixels and convert the RGB values
  size_t nbPixels = $(that, getNbPixels)();
  loop(iPixel, nbPixels) {
    cvt.vals = that->pixels[iPixel];
    that->pixels[iPixel] = $(&cvt, RGB2l1l2l3)();
  }

  // Free memory
  $(&cvt, destruct)();
}

// Convert a RGB image to c1c2c3
static void FromRGBToc1c2c3(CapyImg* const that) {

  // CapyColor to perform the conversion
  CapyColor cvt = CapyColorCreate(capyColorRGBABlack);

  // Loop on the pixels and convert the RGB values
  size_t nbPixels = $(that, getNbPixels)();
  loop(iPixel, nbPixels) {
    cvt.vals = that->pixels[iPixel];
    that->pixels[iPixel] = $(&cvt, RGB2c1c2c3)();
  }

  // Free memory
  $(&cvt, destruct)();
}

// Convert an image from sRGB to L*a*b*
static void FromRGBToLAB(CapyImg* const that) {

  // CapyColor to perform the conversion
  CapyColor cvt = CapyColorCreate(capyColorRGBABlack);

  // Loop on the pixels and convert the RGB values
  size_t nbPixels = $(that, getNbPixels)();
  loop(iPixel, nbPixels) {
    cvt.vals = that->pixels[iPixel];
    cvt.vals = $(&cvt, RGB2XYZ)();
    that->pixels[iPixel] = $(&cvt, XYZ2LAB)();
  }

  // Free memory
  $(&cvt, destruct)();
}

// Convert an image from L*a*b* to sRGB
static void FromLABToRGB(CapyImg* const that) {

  // CapyColor to perform the conversion
  CapyColor cvt = CapyColorCreate(capyColorRGBABlack);

  // Loop on the pixels and convert the RGB values
  size_t nbPixels = $(that, getNbPixels)();
  loop(iPixel, nbPixels) {
    cvt.vals = that->pixels[iPixel];
    cvt.vals = $(&cvt, LAB2XYZ)();
    that->pixels[iPixel] = $(&cvt, XYZ2RGB)();
  }

  // Free memory
  $(&cvt, destruct)();
}

// Convert the image to black and white. The image is supposed to be
// in greyscale, hence the conversion is based on
// 'pixel.color.intensity'. The RGB channels are replaced by 0.0 if
// intensity is less threshold and by 1.0 else.
// Input:
//   threshold: the threshold for the conversion
static void FromGreyscaleToBlackWhite(double threshold) {
  methodOf(CapyImg);

  // Loop on the pixels
  size_t nbPixels = $(that, getNbPixels)();
  loop(iPixel, nbPixels) {

    // Replace the intensity value with the black or white value
    // according to the threshold
    double v = (that->pixels[iPixel].intensity < threshold ? 0.0 : 1.0);
    loop(i, 3) that->pixels[iPixel].RGB[i] = v;
  }
}

// Convert the image to another color space
// Input:
//   colorSpace: the target color space
static void ConvertToColorSpace(CapyColorSpace const colorSpace) {
  methodOf(CapyImg);
  if(that->colorSpace == capyColorSpace_sRGB) {
    if(colorSpace == capyColorSpace_HSV) FromRGBToHSV(that);
    else if(colorSpace == capyColorSpace_rgb) FromRGBTorgb(that);
    else if(colorSpace == capyColorSpace_l1l2l3) FromRGBTol1l2l3(that);
    else if(colorSpace == capyColorSpace_c1c2c3) FromRGBToc1c2c3(that);
    else if(colorSpace == capyColorSpace_LAB) FromRGBToLAB(that);
    else if(colorSpace != capyColorSpace_sRGB) {
      raiseExc(CapyExc_UndefinedExecution);
    }
  } else if(that->colorSpace == capyColorSpace_LAB) {
    if(colorSpace == capyColorSpace_sRGB) FromLABToRGB(that);
    else if(colorSpace != capyColorSpace_LAB) {
      raiseExc(CapyExc_UndefinedExecution);
    }
  } else raiseExc(CapyExc_UndefinedExecution);
  that->colorSpace = colorSpace;
}

// Clean up the values of the pixel to ensure they are in the [0, 1]
// range.
static void CleanupPixelVals(void) {
  methodOf(CapyImg);
  size_t nbPixels = $(that, getNbPixels)();
  loop(i, nbPixels) loop(j, 3) {
    if(that->pixels[i].rgb[j] < 0.0) that->pixels[i].rgb[j] = 0.0;
    if(that->pixels[i].rgb[j] > 1.0) that->pixels[i].rgb[j] = 1.0;
  }
}

// Get the neighbour pixel of a given pixel in a given direction
// Input:
//   pixel: the pixel
// Output:
//   Return a pixel. If it's outside the image, its color property is NULL
//   and other properties are undefined.
static CapyImgPixel GetNeighbour(
  CapyImgPixel const* const pixel,
     CapyImgNeighbour const dir) {
  methodOf(CapyImg);

  // Variable to memorise the result
  CapyImgPixel res = { .color = NULL };

  // Get the position of the neighbour
  CapyImgPos_t deltaPos[8][2] = {
    {0, -1}, {1, 0}, {0, 1}, {-1, 0}, {-1, -1}, {-1, 1}, {1, 1}, {1, -1}
  };
  res.pos = (CapyImgPos){
    .x = pixel->pos.x + deltaPos[dir][0],
    .y = pixel->pos.y + deltaPos[dir][1]
  };

  // If the position of the neighbour is valid, set the index and pointer to
  // the color in the image
  if($(that, isValidCoord)(&(res.pos))) {
    res.idx = PosToIdx(that, res.pos.x, res.pos.y);
    res.color = that->pixels + res.idx;
  }

  // Return the result
  return res;
}

// Set the three first channel values to the value of one of them
// Input:
//   iChan: the channel to be copied to the other channels
static void Set1ChanTo3Chan(size_t const iChan) {
  methodOf(CapyImg);

  // Loop on the pixels
  size_t nbPixels = $(that, getNbPixels)();
  loop(i, nbPixels) {

    // Memorise the value of the channel to be copied
    double val = that->pixels[i].vals[iChan];

    // Set the value of the three channels to that value
    loop(j, 3) that->pixels[i].vals[j] = val;
  }
}

// Normalise the image over the 3 first channels of pixels' color.
static void Normalise(void) {
  methodOf(CapyImg);

  // Variable to memorise the min and max values
  double range[2];
  range[0] = range[1] = that->pixels[0].vals[0];

  // Loop on the pixels' color values
  size_t nbPixel = $(that, getNbPixels)();
  loop(iChan, 3) loop(iPixel, nbPixel) {

    // Update the min and max values
    if(range[0] > that->pixels[iPixel].vals[iChan]) {
      range[0] = that->pixels[iPixel].vals[iChan];
    }
    if(range[1] < that->pixels[iPixel].vals[iChan]) {
      range[1] = that->pixels[iPixel].vals[iChan];
    }
  }

  // Loop on the pixels' color values and normalise the values
  loop(iChan, 3) loop(iPixel, nbPixel) {
    that->pixels[iPixel].vals[iChan] =
      (that->pixels[iPixel].vals[iChan] - range[0]) / (range[1] - range[0]);
  }
}

// Normalise the image per pixels' color channel.
static void Normalise3Chan(void) {
  methodOf(CapyImg);

  // Loop over the channels
  loop(iChan, 3) {

    // Variable to memorise the min and max for that channel
    double range[2];
    range[0] = range[1] = that->pixels[0].vals[iChan];

    // Loop on the pixels
    size_t nbPixel = $(that, getNbPixels)();
    loop(iPixel, nbPixel) {

      // Update the min and values
      if(range[0] > that->pixels[iPixel].vals[iChan]) {
        range[0] = that->pixels[iPixel].vals[iChan];
      }
      if(range[1] < that->pixels[iPixel].vals[iChan]) {
        range[1] = that->pixels[iPixel].vals[iChan];
      }
    }

    // Loop on the pixels and update the value for that channel
    loop(iPixel, nbPixel) {
      that->pixels[iPixel].vals[iChan] =
        (that->pixels[iPixel].vals[iChan] - range[0]) / (range[1] - range[0]);
    }
  }
}

// Get the colors used in the image as a point cloud
// Output:
//   Return a CapyPointCloud of dimension 4 (all color channels are used).
// Exception:
//   May raise CapyExc_MallocFailed.
static CapyPointCloud* GetColorsAsPointCloud(void) {
  methodOf(CapyImg);

  // Create the point cloud
  CapyPointCloud* pointCloud = CapyPointCloudAlloc(4);

  // Add the colors to the point cloud
  size_t nbPixel = that->dims.width * that->dims.height;
  safeMalloc(pointCloud->points, nbPixel);
  if(!(pointCloud->points)) return NULL;
  pointCloud->size = nbPixel;
  loop(iPixel, nbPixel) {
    pointCloud->points[iPixel] = CapyVecCreate(4);
    loop(i, 4) {
      pointCloud->points[iPixel].vals[i] = that->pixels[iPixel].vals[i];
    }
  }

  // Return the point cloud
  return pointCloud;
}

// Check if the image is identical to another
// Input:
//   img: the other image
// Output:
//   Return true if the images are identical (difference of color values
//   within 1/255), else false.
static bool IsSame(CapyImg const* const img) {
  methodOf(CapyImg);
  if(
    that->dims.width != img->dims.width ||
    that->dims.height != img->dims.height ||
    fabs(that->gamma - img->gamma) > 1e-4
  ) {
    return false;
  }
  forEach(pixel, that->iter) {
    loop(i, 4) {
      double diff =
        that->pixels[pixel.idx].vals[i] - img->pixels[pixel.idx].vals[i];
      if(fabs(diff) > 1.0 / 255.0) return false;
    }
  }
  return true;
}

// Save the image to a given path
// Input:
//   path: the path where to save
// Output:
//   The image is saved.
// Exception:
//   May raise CapyExc_UnsupportedFormat.
static void SaveToPath(char const* const path) {
  methodOf(CapyImg);
  size_t len = strlen(path);
  if(strstr(path, ".png") != path + len - 4) {
    raiseExc(CapyExc_UnsupportedFormat);
  }
  CapyPngFormat png = CapyPngFormatCreate();
  CapyStreamIo stream = CapyStreamIoCreate();
  $(&stream, open)(path, "wb");
  $(&png, saveImg)(that, &stream);
  $(&stream, destruct)();
  $(&png, destruct)();
}

// Paste the image into another
// Input:
//   img: the image in which to paste
//   pos: the position in 'img' of the top-left corner of the image to paste
// Output:
//   The image is paste into 'img'. It is clipped as necessary. Both image
//   are assumed to b of compatible color space. If the image color space
//   is rgba, the alpha channel is used to blend the image into 'img', else
//   pixel from the image overwrite those in 'img'.
static void PasteInto(
           CapyImg* const img,
  CapyImgPos const* const pos) {
  methodOf(CapyImg);

  // Create the range of coordinates in img for the clipped area actually
  // pasted
  CapyRangeImgPos rangeX = {
    .min = 0, .max = (CapyImgPos_t)(that->dims.width - 1)
  };
  CapyRangeImgPos rangeY = {
    .min = 0, .max = (CapyImgPos_t)(that->dims.height - 1)
  };
  if(pos->x < 0) rangeX.min = -pos->x;
  if(pos->y < 0) rangeY.min = -pos->y;
  if(rangeX.max + pos->x >= (CapyImgPos_t)(img->dims.width)) {
    rangeX.max += (CapyImgPos_t)(img->dims.width) - 1 - (rangeX.max + pos->x);
  }
  if(rangeY.max + pos->y >= (CapyImgPos_t)(img->dims.height)) {
    rangeY.max += (CapyImgPos_t)(img->dims.height) - 1 - (rangeY.max + pos->y);
  }

  // Loop on the clipped area
  loopRange(posX, rangeX) loopRange(posY, rangeY) {
    CapyImgPos posFrom = {.x = posX, .y = posY};
    CapyImgPos posTo = {.x = posX + pos->x, .y = posY + pos->y};

    // Get the pixel to be pasted
    CapyColorData* pixel = $(that, getColor)(&posFrom);

    // Calculate the blending coefficient according to the image color mode
    double blending = 1.0;
    if(that->mode == capyImgMode_rgba) {
      blending = pixel->RGBA[3];
    }

    // Paste the color
    CapyColorData* pixelImg = $(img, getColor)(&posTo);
    CapyColorData blendedPixel = *pixel;
    loop(i, 3) {
      blendedPixel.vals[i] =
        blending * pixel->vals[i] + (1.0 - blending) * pixelImg->vals[i];
    }
    $(img, setColor)(&posTo, &blendedPixel);
  }
}

// Fill the whole image with a given color
// Input:
//   color: the color
// Output:
//   The whole image is filled with the color
static void FillWithColor(CapyColorData const* const color) {
  methodOf(CapyImg);
  forEach(pixel, that->iter) *(pixel.color) = *color;
}

// Convert the RGB image to black and white using horizontal lines
// representing the local lightness for a given line max thickness
// Input:
//   lineThickness: maximum line thickness
// Output:
//   The image is updated
static void ConvertToBlackWhiteLightnessLine(
  CapyImgDims_t const lineThickness) {
  methodOf(CapyImg);

  // Get the half thickness of the line
  CapyImgDims_t const halfThickness = lineThickness / 2;

  // Gaussian blur the image
  CapyImgKernel* kernel = CapyImgKernelAlloc(halfThickness);
  $(kernel, setToGaussianBlur)(halfThickness);
  CapyImg* blurredImg = $(kernel, apply)(that);
  CapyImgKernelFree(&kernel);

  // Color instance for conversion RGB->HSL
  CapyColor color = CapyColorCreate(capyColorRGBAWhite);

  // Loop on the line in the images
  for (
    CapyImgDims_t y = halfThickness;
    y < that->dims.height;
    y += lineThickness
  ) {

    // Loop on the pixels in the line
    loop(x, that->dims.width) {

      // Calculate the average lightness at the position
      CapyImgPos pos = { .x = (CapyImgPos_t)x, .y = (CapyImgPos_t)y};
      color.vals = *($(blurredImg, getColor)(&pos));
      CapyColorData hsv = $(&color, RGB2HSV)();
      double avgLight = hsv.vals[2];

      // Get the result thickness
      CapyImgDims_t lightThickness = (CapyImgDims_t)(avgLight * lineThickness);

      // Draw the line in the image
      CapyImgDims_t rangeY[2] = {
        (CapyImgDims_t)round((double)(lineThickness - lightThickness) * 0.5),
        (CapyImgDims_t)round((double)(lineThickness + lightThickness) * 0.5)
      };
      loop(dy, lineThickness) {
        CapyColorData pixelColor = capyColorRGBABlack;
        if(rangeY[0] <= dy && dy <= rangeY[1]) pixelColor = capyColorRGBAWhite;
        pos = (CapyImgPos){
          .x = (CapyImgPos_t)x, .y = (CapyImgPos_t)(y - halfThickness + dy)
        };
        if((CapyImgDims_t)(pos.y) < that->dims.height) {
          $(that, setColor)(&pos, &pixelColor);
        }
      }
    }
  }

  // Free memory
  $(&color, destruct)();
  CapyImgFree(&blurredImg);
}

// Rotate the image with the skewing method
// Input:
//   theta: rotation angle in radians
//   bgColor: color of background pixels in the result image
// Output:
//   The image CW is rotated around its center. The skewing method guarantees
//   that each pixel is conserved and translated to a new position. Pixels
//   rotated outside the result image are clipped.
static void Rotate(
                double const theta,
  CapyColorData const* const bgColor) {
  methodOf(CapyImg);

  // Get the center position
  CapyImgPos posCenter = {
    .x = (CapyImgPos_t)(that->dims.width) / 2,
    .y = (CapyImgPos_t)(that->dims.height) / 2,
  };

  // Rotate the image around the center
  $(that, rotateAround)(theta, bgColor, &posCenter);
}

// Rotate the image with the skewing method around a given point
// Input:
//   theta: rotation angle in radians
//   bgColor: color of background pixels in the result image
//   posCenter: position of the center of rotation
// Output:
//   The image CW is rotated around 'pos'. The skewing method guarantees
//   that each pixel is conserved and translated to a new position. Pixels
//   rotated outside the result image are clipped.
static void RotateAround(
                double const theta,
  CapyColorData const* const bgColor,
     CapyImgPos const* const posCenter) {
  methodOf(CapyImg);

  // Allocate memory for the result pixels
  CapyColorData* pixels = NULL;
  safeMalloc(pixels, that->dims.width * that->dims.height);
  if(pixels == NULL) return;
  loop(i, that->dims.width * that->dims.height) pixels[i] = *bgColor;

  // Skew coefficients
  double thetaMod = theta;
  while(thetaMod < -M_PI) thetaMod += 2.0 * M_PI;
  while(thetaMod > M_PI) thetaMod -= 2.0 * M_PI;
  double thetaCorr = thetaMod;
  if(thetaCorr < -M_PI_2) thetaCorr = M_PI + thetaCorr;
  else if(thetaCorr > M_PI_2) thetaCorr = -(M_PI - thetaCorr);
  double skew[2] = {-tan(thetaCorr * 0.5), sin(thetaCorr)};

  // Loop on the pixels
  forEach(pixel, that->iter) {

    // Calculate the coordinates of the rotated pixel
    CapyImgPos_t pos[2] = {pixel.pos.x, pixel.pos.y};
    loop(i, 2) pos[i] -= posCenter->coords[i];
    if(fabs(thetaMod) > M_PI_2) loop(i, 2) pos[i] = -pos[i];
    loop(i, 3) {
      int j = i % 2;
      pos[j] += (CapyImgPos_t)round(skew[j] * (double)(pos[(i + 1) % 2]));
    }
    loop(i, 2) pos[i] += posCenter->coords[i];

    // Copy the pixel to its rotated coordinate if it's inside the image
    if(
      pos[0] >= 0 && (CapyImgDims_t)pos[0] < that->dims.width &&
      pos[1] >= 0 && (CapyImgDims_t)pos[1] < that->dims.height
    ) {
      size_t idx = (size_t)(pos[1] * (CapyImgPos_t)(that->dims.width) + pos[0]);
      pixels[idx] = *(pixel.color);
    }
  }

  // Replace the old pixels with the new pixels
  free(that->pixels);
  that->pixels = pixels;
}

// Check if the image intersects another image based on their alpha channel
// Input:
//   img: the other image
//   pos: the position of the top-left corner of the other image in the image
// Output:
//   Return true if there exists at least one pixel in the image with alpha
//   value greater than 1e-6 and corresponding pixel in the other image with
//   alpha value also greater than 1e-6
static bool IsIntersectingAlpha(
     CapyImg const* const img,
  CapyImgPos const* const pos) {
  methodOf(CapyImg);
  CapyRangeImgPos range[4] = {0};
  range[0] = CapyRangeImgPosCreate(0, (CapyImgPos_t)(that->dims.width));
  range[1] = CapyRangeImgPosCreate(0, (CapyImgPos_t)(that->dims.height));
  range[2] = CapyRangeImgPosCreate(
    pos->x, pos->x + (CapyImgPos_t)(img->dims.width));
  range[3] = CapyRangeImgPosCreate(
    pos->y, pos->y + (CapyImgPos_t)(img->dims.height));
  loop(i, 2) $(range + i, clip)(range + i + 2);
  loop(i, 2) range[i].max -= 1;
  loopRange(x, range[0]) loopRange(y, range[1]) {
    CapyImgPos const posThat = {.x = x, .y = y};
    CapyImgPos const posOther = {.x = x - pos->x, .y = y - pos->y};
    CapyColorData const* const colorThat = $(that, getColor)(&posThat);
    CapyColorData const* const colorOther = $(img, getColor)(&posOther);
    if(colorThat->vals[3] > 1e-6 && colorOther->vals[3] > 1e-6) return true;
  }
  return false;
}

// Free the memory used by a CapyImg
static void Destruct(void) {
  methodOf(CapyImg);
  if(that->pixels != NULL) free(that->pixels);
  $(&(that->iter), destruct)();
}

// Create a CapyImg of given dimensions and mode
// Input:
//   mode: mode of the image
//   dims: dimensions of the image
// Output:
//   Return a CapyImg initialised to rgba opaque white
// Exception:
//   May raise CapyExc_MallocFailed.
CapyImg CapyImgCreate(
  CapyImgMode const mode,
  CapyImgDims const dims) {

  // Create the instance
  CapyImg that = {
    .pixels = NULL,
    .mode = mode,
    .dims = dims,
    .gamma = 1.0,
    .colorSpace = capyColorSpace_sRGB,
    .destruct = Destruct,
    .getColor = GetColor,
    .setColor = SetColor,
    .initIterator = InitIterator,
    .isValidCoord = IsValidCoord,
    .copy = Copy,
    .getNbPixels = GetNbPixels,
    .setGamma = SetGamma,
    .getBrightestPixel = GetBrightestPixel,
    .getEdgeMap = GetEdgeMap,
    .getEdgeMap3Chan = GetEdgeMap3Chan,
    .getCannyEdgeMap3Chan = GetCannyEdgeMap3Chan,
    .getCannyEdgeMap = GetCannyEdgeMap,
    .getLocalMaxGradientEdgeMap3Chan = GetLocalMaxGradientEdgeMap3Chan,
    .fromRGBToGreyScale = FromRGBToGreyScale,
    .fromGreyscaleToBlackWhite = FromGreyscaleToBlackWhite,
    .convertToColorSpace = ConvertToColorSpace,
    .cleanupPixelVals = CleanupPixelVals,
    .getNeighbour = GetNeighbour,
    .set1ChanTo3Chan = Set1ChanTo3Chan,
    .normalise = Normalise,
    .normalise3Chan = Normalise3Chan,
    .getColorsAsPointCloud = GetColorsAsPointCloud,
    .isSame = IsSame,
    .saveToPath = SaveToPath,
    .pasteInto = PasteInto,
    .fillWithColor = FillWithColor,
    .convertToBlackWhiteLightnessLine = ConvertToBlackWhiteLightnessLine,
    .rotate = Rotate,
    .rotateAround = RotateAround,
    .isIntersectingAlpha = IsIntersectingAlpha,
  };
  that.iter =
    CapyImgIteratorCreate(NULL, capyImgIteratorType_LeftToRight_TopToBottom);

  // Allocate memory for the pixels
  safeMalloc(that.pixels, dims.width * dims.height);

  // Set the pixels to black
  if(that.pixels) loop(i, dims.width * dims.height) {
    that.pixels[i] = capyColorRGBABlack;
  }

  // Return the instance
  return that;
}

// Allocate memory for a new CapyImg and create it
// Input:
//   mode: mode of the image
//   dims: dimensions of the image
// Output:
//   Return a CapyImg initialised to rgba opaque white
// Exception:
//   May raise CapyExc_MallocFailed.
CapyImg* CapyImgAlloc(
  CapyImgMode const mode,
  CapyImgDims const dims) {

  // Allocate memory for the instance
  CapyImg* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;

  // Create the instance
  *that = CapyImgCreate(mode, dims);

  // Init the iterator
  $(that, initIterator)();

  // Return the instance
  return that;
}

// Load an image to a given path
// Input:
//   path: the path to the image
// Output:
//   Return a new image.
// Exception:
//   May raise CapyExc_MallocFailed, CapyExc_UnsupportedFormat.
CapyImg* CapyImgLoadFromPath(char const* const path) {
  size_t len = strlen(path);
  if(strstr(path, ".png") != path + len - 4) {
    raiseExc(CapyExc_UnsupportedFormat);
  }
  CapyPngFormat png = CapyPngFormatCreate();
  CapyStreamIo stream = CapyStreamIoCreate();
  $(&stream, open)(path, "rb");
  CapyImg* img = $(&png, loadImg)(&stream);
  $(&stream, destruct)();
  $(&png, destruct)();
  return img;
}

// Allocate memory and create a clone of an image
// Input:
//   img: the image to clone
// Output:
//   Return a clone of the image in argument
// Exception:
//   May raise CapyExc_MallocFailed.
CapyImg* CapyImgClone(CapyImg const* const img) {

  // Create the clone
  CapyImg* that = CapyImgAlloc(img->mode, *(CapyImgDims*)&(img->dims));
  that->gamma = img->gamma;

  // Copy the pixels
  memcpy(
    that->pixels,
    img->pixels,
    sizeof(CapyColorData) * that->dims.width * that->dims.height);

  // Return the clone
  return that;
}

// Free the memory used by a CapyImg* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyImg to free
void CapyImgFree(CapyImg** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Convert from cm to pixels given a dpi
// Input:
//   length: the length in centimeter to convert
//   dpi: the dpi used to convert
// Output:
//   Return the converted length.
double CapyCm2Pixel(
  double const length,
  double const dpi) {
  return length / 2.54 * dpi;
}

// Convert from pixels to cm given a dpi
// Input:
//   length: the length in pixels to convert
//   dpi: the dpi used to convert
// Output:
//   Return the converted length.
double CapyPixel2Cm(
  double const length,
  double const dpi) {
  return length / dpi * 2.54;
}
