// ---------------------------------- image.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_IMAGE_H
#define CAPY_IMAGE_H
#include "externalHeaders.h"
#include "color.h"
#include "list.h"
#include "geometricShape.h"
#include "pointCloud.h"
#include "range.h"

// Description:
// 2D image class.

// Enumeration of image modes
typedef enum CapyImgMode {
  capyImgMode_greyscale,
  capyImgMode_rgb,
  capyImgMode_rgba,
  capyImgMode_hsv
} CapyImgMode;

// Type for the size of an image (choosen to match png_uint_32)
typedef uint32_t CapyImgDims_t;

// Definition of the dimensions of an image
#define CapyImgDims_                               \
  union {                                          \
    CapyImgDims_t vals[2];                         \
    struct __attribute__((packed)) {               \
      CapyImgDims_t width; CapyImgDims_t height;}; \
  }

// Dimensions of an image
typedef CapyImgDims_ CapyImgDims;

// Predefined dimensions
extern CapyImgDims const capyImgDims_100x100;
extern CapyImgDims const capyImgDims_800x600;
extern CapyImgDims const capyImgDims_600x800;
extern CapyImgDims const capyImgDims_400x400;
extern CapyImgDims const capyImgDims_800x800;

// Type to store a pixel coordinate (should be the signed version of
// CapyImgDims_t)
typedef int32_t CapyImgPos_t;

// Range of positions in the image
CapyDecRange(ImgPos, CapyImgPos_t)

// Definition of a position in the image (from left to right and top to bottom)
#define CapyImgPos_                      \
  union {                                \
    CapyImgPos_t coords[2];              \
    struct __attribute__((packed)) {     \
      CapyImgPos_t x; CapyImgPos_t y;};  \
  }

// Position in the image (from left to right and top to bottom)
typedef CapyImgPos_ CapyImgPos;

// Enumeration to identify the neighours of a pixel
typedef enum CapyImgNeighbour {
  capyImgNeighbour_top,
  capyImgNeighbour_right,
  capyImgNeighbour_bottom,
  capyImgNeighbour_left,
  capyImgNeighbour_topLeft,
  capyImgNeighbour_topRight,
  capyImgNeighbour_bottomRight,
  capyImgNeighbour_bottomLeft,
  capyImgNeighbour_last,
} CapyImgNeighbour;

// Container for a pixel position, index and color
typedef struct CapyImgPixel {

  // Position
  CapyImgPos pos;

  // Index in the array of pixels
  size_t idx;

  // Color data
  CapyColorData* color;
} CapyImgPixel;

// Set of pixels
CapyDecArray(CapyImgPixels, CapyImgPixel)

// Enumeration for the types of iterator on a CapyImg
typedef enum CapyImgIteratorType {
  capyImgIteratorType_LeftToRight_TopToBottom,
} CapyImgIteratorType;

// CapyImg object predeclaration
typedef struct CapyImg CapyImg;

// Iterator on the pixels of a CapyImg structure.
typedef struct CapyImgIterator {

  // Index of the current step in the iteration
  size_t idx;

  // Returned data type
  CapyImgPixel datatype;

  // Current pixel of the iteration
  CapyImgPixel pixel;

  // Type of iteration
  CapyImgIteratorType type;
  CapyPad(CapyImgIteratorType, type);

  // Image associated to the iteration
  CapyImg* img;

  // Destructor
  void (*destruct)(void);

  // Reset the iterator
  // Output:
  //   Return the first pixel of the iteration
  CapyImgPixel* (*reset)(void);

  // Move the iterator to the previous pixel
  // Output:
  //   Return the previous pixel of the iteration
  CapyImgPixel* (*prev)(void);

  // Move the iterator to the next pixel
  // Output:
  //   Return the next pixel of the iteration
  CapyImgPixel* (*next)(void);

  // Check if the iterator is on a valid pixel
  // Output:
  //   Return true if the iterator is on a valid pixel, else false
  bool (*isActive)(void);

  // Get the current pixel of the iteration
  // Output:
  //   Return a pointer to the current pixel
  CapyImgPixel* (*get)(void);

  // Set the type of the iterator and reset it
  // Input:
  //   type: the new type of the iterator
  void (*setType)(CapyImgIteratorType const type);
} CapyImgIterator;

// Create an iterator on a CapyImg
// Input:
//   img: the image on which to iterate
//   type: the type of iterator
// Output:
//   Return the iterator
CapyImgIterator CapyImgIteratorCreate(
             CapyImg* const img,
  CapyImgIteratorType const type);

// Allocate memory and create an iterator on a CapyImg
// Input:
//   img: the image on which to iterate
//   type: the type of iterator
// Output:
//   Return the iterator
CapyImgIterator* CapyImgIteratorAlloc(
             CapyImg* const img,
  CapyImgIteratorType const type);

// Free the memory used by a pointer to an iterator and reset '*that' to NULL
// Input:
//   that: a pointer to the iterator to free
void CapyImgIteratorFree(CapyImgIterator** const that);

// CapyImg object
typedef struct CapyImg CapyImg;
struct CapyImg {

  // Array of pixel colors, stored by rows from top to bottom and
  // left to right
  CapyColorData* pixels;

  // Mode of the image
  CapyImgMode mode;

  // Dimensions of the image
  CapyImgDims dims;
  CapyPad(char[4], dims);

  // Iterator on pixels
  CapyImgIterator iter;

  // Display gamma (by default 2.2, equivalent to a standard PC monitor)
  // i.e.: (displayed RGB) = (file RGB)^(display gamma)
  double gamma;

  // Color space to interpret the pixels values (default: sRGB)
  CapyColorSpace colorSpace;
  CapyPad(CapyColorSpace, colorSpace);

  // Destructor
  void (*destruct)(void);

  // Get the color of a pixel at a given position
  // Input:
  //   pos: the position of the pixel
  // Output:
  //   Return the color of the pixel
  CapyColorData* (*getColor)(CapyImgPos const* const pos);

  // Set the color of a pixel at a given position
  // Input:
  //   pos: the position of the pixel
  //   color: the color to set
  void (*setColor)(
       CapyImgPos const* const pos,
    CapyColorData const* const color);

  // Initialise the iterator of the image, must be called after the creation
  // of the image when it is created with Create(), Alloc() automatically
  // initialise the iterator.
  void (*initIterator)(void);

  // Check if coordinates are inside the image
  // Input:
  //   coord: the coordinates to check
  // Output:
  //   Return true if the coordinates are inside the image, else false
  bool (*isValidCoord)(CapyImgPos const* const coord);

  // Copy another image in the image
  // Input:
  //   img: the image to copy (must be of same dimensions)
  void (*copy)(CapyImg const* const img);

  // Get the number of pixels
  // Output:
  //   Return the number of pixels
  size_t (*getNbPixels)(void);

  // Change the gamma encoding of the image. Pixels value are updated.
  // Input:
  //   gamma: the new gamma value
  void (*setGamma)(double const gamma);

  // Get the brightest pixel in the image
  // Output:
  //   Return the brightest (maximum r+g+b) pixel.
  CapyImgPixel (*getBrightestPixel)(void);

  // Get an edge map of the image.
  // Input:
  //   kernelSize: the size in pixel of the gaussian blur kernel, the
  //               higher the less sensitive
  //   stdDev: the standard deviation of the gaussian blur (as a guide,
  //           kernelSize = 4 * stdDev)
  //   iChannel: index of the channel used for edge detection
  // Output:
  //   Return an image of same dimensions where pixel values
  //   represent the 'flatness' of the image. The red channel is the norm of
  //   the derivative gaussian blur in x and y direction. The green and blue
  //   channels contain the x and y coordinates of the normalised vector
  //   indicating the direction of the slope (going up), the alpha channel is
  //   set to 1.0.
  // Exception:
  //   May raise CapyExc_MallocFailed.
  CapyImg* (*getEdgeMap)(
   CapyImgDims_t const kernelSize,
          double const stdDev,
          size_t const iChannel);

  // Get an edge map of the image based on its first 3 channels.
  // Input:
  //   kernelSize: the size in pixel of the gaussian blur kernel, the
  //               higher the less sensitive
  //   stdDev: the standard deviation of the gaussian blur (as a guide,
  //           kernelSize = 4 * stdDev)
  // Output:
  //   Use getEdgeMap() to calculate the edge map for each of the three
  //   first channels of the image and combined them into one single edge
  //   map by choosing for each pixel the values in the edge map with
  //   maximum slope at that pixel.
  // Exception:
  //   May raise CapyExc_MallocFailed.
  CapyImg* (*getEdgeMap3Chan)(
   CapyImgDims_t const kernelSize,
          double const stdDev);

  // Get an edge map of the image based on one given channel using the
  // Canny edge detection method.
  // Input:
  //   iChan: the channel to use
  //   kernelSize: the size in pixel of the gaussian blur kernel, the
  //               higher the less sensitive
  //   stdDev: the standard deviation of the gaussian blur (as a guide,
  //           kernelSize = 4 * stdDev)
  //   hysteresisLow, hysteresisHigh: the thresholds for the hysteresis phase,
  //                                  in [0.0, 1.0] and scaled to the maximum
  //                                  intensity after local maximum search phase
  // Output:
  //   Return an image of same dimension as 'that', where white pixels
  //   indicates an edge
  // Exception:
  //   May raise CapyExc_MallocFailed.
  CapyImg* (*getCannyEdgeMap)(
          size_t const iChan,
   CapyImgDims_t const kernelSize,
          double const stdDev,
          double const hysteresisLow,
          double const hysteresisHigh);

  // Get an edge map of the image based on its first 3 channels using the
  // Canny edge detection method.
  // Input:
  //   kernelSize: the size in pixel of the gaussian blur kernel, the
  //               higher the less sensitive
  //   stdDev: the standard deviation of the gaussian blur (as a guide,
  //           kernelSize = 4 * stdDev)
  //   hysteresisLow, hysteresisHigh: the thresholds for the hysteresis phase,
  //                                  in [0.0, 1.0] and scaled to the maximum
  //                                  intensity after local maximum search phase
  // Output:
  //   Return an image of same dimension as 'that', where white pixels
  //   indicates an edge
  // Exception:
  //   May raise CapyExc_MallocFailed.
  CapyImg* (*getCannyEdgeMap3Chan)(
   CapyImgDims_t const kernelSize,
          double const stdDev,
          double const hysteresisLow,
          double const hysteresisHigh);

  // Get an edge map of the image based on its first 3 channels using the
  // local maximum gradient method.
  // Input:
  //   kernelSize: the size in pixel of the gaussian blur kernel, the
  //               higher the less sensitive
  //   stdDev: the standard deviation of the gaussian blur (as a guide,
  //           kernelSize = 4 * stdDev)
  //   threshold: minimum gradient to be marked as an edge, in [0, 1] of the
  //              maximum gradient
  // Output:
  //   Return an image of same dimension as 'that', where white pixels
  //   indicates an edge
  // Exception:
  //   May raise CapyExc_MallocFailed.
  CapyImg* (*getLocalMaxGradientEdgeMap3Chan)(
   CapyImgDims_t const kernelSize,
          double const stdDev,
          double const threshold);

  // Convert an image to greyscale. RGB channel are replaced with
  // (r+g+b)/3
  void (*fromRGBToGreyScale)(void);

  // Convert the image to black and white. The image is supposed to be
  // in greyscale, hence the conversion is based on
  // 'pixel.color.intensity'. The RGB channels are replaced by 0.0 if
  // intensity is less threshold and by 1.0 else.
  // Input:
  //   threshold: the threshold for the conversion
  void (*fromGreyscaleToBlackWhite)(double threshold);

  // Convert the image to another color space
  // Input:
  //   colorSpace: the target color space
  void (*convertToColorSpace)(CapyColorSpace const colorSpace);

  // Clean up the values of the pixel to ensure they are in the [0, 1]
  // range.
  void (*cleanupPixelVals)(void);

  // Get the neighbour pixel of a given pixel in a given direction
  // Input:
  //   pixel: the pixel
  // Output:
  //   Return a pixel. If it's outside the image, its color property is NULL
  //   and other properties are undefined.
  CapyImgPixel (*getNeighbour)(
    CapyImgPixel const* const pixel,
       CapyImgNeighbour const dir);

  // Set the three first channel values to the value of one of them
  // Input:
  //   iChan: the channel to be copied to the other channels
  void (*set1ChanTo3Chan)(size_t const iChan);

  // Normalise the image over the 3 first channels of pixels' color.
  void (*normalise)(void);

  // Normalise the image per pixels' color channel.
  void (*normalise3Chan)(void);

  // Get the colors used in the image as a point cloud
  // Output:
  //   Return a CapyPointCloud of dimension 4 (all color channels are used).
  // Exception:
  //   May raise CapyExc_MallocFailed.
  CapyPointCloud* (*getColorsAsPointCloud)(void);

  // Check if the image is identical to another
  // Input:
  //   img: the other image
  // Output:
  //   Return true if the images are identical (difference of color values
  //   within 1/255), else false.
  bool (*isSame)(CapyImg const* const img);

  // Save the image to a given path
  // Input:
  //   path: the path where to save
  // Output:
  //   The image is saved.
  // Exception:
  //   May raise CapyExc_UnsupportedFormat.
  void (*saveToPath)(char const* const path);

  // Paste the image into another
  // Input:
  //   img: the image in which to paste
  //   pos: the position in 'img' of the top-left corner of the image to paste
  // Output:
  //   The image is paste into 'img'. It is clipped as necessary. Both image
  //   are assumed to b of compatible color space. If the image color space
  //   is rgba, the alpha channel is used to blend the image into 'img', else
  //   pixel from the image overwrite those in 'img'.
  void (*pasteInto)(
             CapyImg* const img,
    CapyImgPos const* const pos);

  // Fill the whole image with a given color
  // Input:
  //   color: the color
  // Output:
  //   The whole image is filled with the color
  void (*fillWithColor)(CapyColorData const* const color);

  // Convert the RGB image to black and white using horizontal lines
  // representing the local lightness for a given line max thickness
  // Input:
  //   lineThickness: maximum line thickness
  // Output:
  //   The image is updated
  void (*convertToBlackWhiteLightnessLine)(CapyImgDims_t const lineThickness);

  // Rotate the image with the skewing method
  // Input:
  //   theta: rotation angle in radians
  //   bgColor: color of background pixels in the result image
  // Output:
  //   The image CW is rotated around its center. The skewing method guarantees
  //   that each pixel is conserved and translated to a new position. Pixels
  //   rotated outside the result image are clipped.
  void (*rotate)(
                  double const theta,
    CapyColorData const* const bgColor);

  // Rotate the image with the skewing method around a given point
  // Input:
  //   theta: rotation angle in radians
  //   bgColor: color of background pixels in the result image
  //   posCenter: position of the center of rotation
  // Output:
  //   The image CW is rotated around 'pos'. The skewing method guarantees
  //   that each pixel is conserved and translated to a new position. Pixels
  //   rotated outside the result image are clipped.
  void (*rotateAround)(
                  double const theta,
    CapyColorData const* const bgColor,
       CapyImgPos const* const posCenter);

  // Check if the image intersects another image based on their alpha channel
  // Input:
  //   img: the other image
  //   pos: the position of the top-left corner of the other image in the image
  // Output:
  //   Return true if there exists at least one pixel in the image with alpha
  //   value greater than 1e-6 and corresponding pixel in the other image with
  //   alpha value also greater than 1e-6
  bool (*isIntersectingAlpha)(
       CapyImg const* const img,
    CapyImgPos const* const pos);
};

// Create a CapyImg of given dimensions and mode
// Input:
//   mode: mode of the image
//   dims: dimensions of the image
// Output:
//   Return a CapyImg initialised to rgba opaque white
// Exception:
//   May raise CapyExc_MallocFailed.
CapyImg CapyImgCreate(
  CapyImgMode const mode,
  CapyImgDims const dims);

// Shortcuts for CapyImgCreate()
#define CapyImgCreate_Greyscale_800x600() \
  CapyImgCreate(capyImgMode_greyscale, capyImgDims_800x600)
#define CapyImgCreate_Greyscale_600x800() \
  CapyImgCreate(capyImgMode_scale, capyImgDims_600x800)
#define CapyImgCreate_RGB_800x600() \
  CapyImgCreate(capyImgMode_rgb, capyImgDims_800x600)
#define CapyImgCreate_RGB_600x800() \
  CapyImgCreate(capyImgMode_rgb, capyImgDims_600x800)
#define CapyImgCreate_RGBA_800x600() \
  CapyImgCreate(capyImgMode_rgba, capyImgDims_800x600)
#define CapyImgCreate_RGBA_600x800() \
  CapyImgCreate(capyImgMode_rgba, capyImgDims_600x800)
#define CapyImgCreate_HLS_800x600() \
  CapyImgCreate(capyImgMode_hls, capyImgDims_800x600)
#define CapyImgCreate_HLS_600x800() \
  CapyImgCreate(capyImgMode_hls, capyImgDims_600x800)

// Allocate memory for a new CapyImg and create it
// Input:
//   mode: mode of the image
//   dims: dimensions of the image
// Output:
//   Return a CapyImg initialised to rgba opaque white
// Exception:
//   May raise CapyExc_MallocFailed.
CapyImg* CapyImgAlloc(
  CapyImgMode const mode,
  CapyImgDims const dims);

// Load an image to a given path
// Input:
//   path: the path to the image
// Output:
//   Return a new image.
// Exception:
//   May raise CapyExc_MallocFailed, CapyExc_UnsupportedFormat.
CapyImg* CapyImgLoadFromPath(char const* const path);

// Shortcuts for CapyImgAlloc()
#define CapyImgAlloc_Greyscale_800x600() \
  CapyImgAlloc(capyImgMode_greyscale, capyImgDims_800x600)
#define CapyImgAlloc_Greyscale_600x800() \
  CapyImgAlloc(capyImgMode_scale, capyImgDims_600x800)
#define CapyImgAlloc_RGB_800x600() \
  CapyImgAlloc(capyImgMode_rgb, capyImgDims_800x600)
#define CapyImgAlloc_RGB_600x800() \
  CapyImgAlloc(capyImgMode_rgb, capyImgDims_600x800)
#define CapyImgAlloc_RGBA_800x600() \
  CapyImgAlloc(capyImgMode_rgba, capyImgDims_800x600)
#define CapyImgAlloc_RGBA_600x800() \
  CapyImgAlloc(capyImgMode_rgba, capyImgDims_600x800)
#define CapyImgAlloc_HLS_800x600() \
  CapyImgAlloc(capyImgMode_hls, capyImgDims_800x600)
#define CapyImgAlloc_HLS_600x800() \
  CapyImgAlloc(capyImgMode_hls, capyImgDims_600x800)

// Allocate memory and create a clone of an image
// Input:
//   img: the image to clone
// Output:
//   Return a clone of the image in argument
// Exception:
//   May raise CapyExc_MallocFailed.
CapyImg* CapyImgClone(CapyImg const* const img);

// Free the memory used by a CapyImg* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyImg to free
void CapyImgFree(CapyImg** const that);

// Convert from cm to pixels given a dpi
// Input:
//   length: the length in centimeter to convert
//   dpi: the dpi used to convert
// Output:
//   Return the converted length.
double CapyCm2Pixel(
  double const length,
  double const dpi);

// Convert from pixels to cm given a dpi
// Input:
//   length: the length in pixels to convert
//   dpi: the dpi used to convert
// Output:
//   Return the converted length.
double CapyPixel2Cm(
  double const length,
  double const dpi);
#endif
