// ------------------------------ imgKernel.c ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "imgKernel.h"

// Conversion of a relative position to an index in the matrix
#define CAPY_IMGKERNEL_POS_TO_IDX(x, y)                                  \
  (x + (CapyImgPos_t)(that->size) + (CapyImgPos_t)(1 + 2 * that->size) * \
  (y + (CapyImgPos_t)that->size))

// Convolve the kernel on a given image
// Input:
//   img: the image to which the kernel is applied
// Output:
//   Return the  result of the convolution as a newly allocated image
// Exception:
//   May raise CapyExc_MallocFailed.
static CapyImg* Apply(CapyImg const* const img) {
  methodOf(CapyImgKernel);

  // Create a clone of the image
  CapyImg* res = CapyImgClone(img);

  // Loop on the pixels of the image, avoiding the kernel ot get out of
  // the image
  CapyRangeImgPos range[2] = {
    CapyRangeImgPosCreate(
      (CapyImgPos_t)(that->size),
      (CapyImgPos_t)(img->dims.width - that->size - 1)),
    CapyRangeImgPosCreate(
      (CapyImgPos_t)(that->size),
      (CapyImgPos_t)(img->dims.height - that->size - 1))
  };
  loopRange(x, range[0]) loopRange(y, range[1]) {

    // Variables to calculate the result
    CapyImgPos pixel = {{x, y}};
    CapyColorData color = {.RGBA = {0.0, 0.0, 0.0, 0.0}};

    // Loop on the kernel
    loopRange(dx, that->range) loopRange(dy, that->range) {

      // Apply the kernel
      CapyImgPos in = {{x - dx, y - dy}};
      CapyColorData const* inColor = $(img, getColor)(&in);
      loop(i, 4) {
        color.RGBA[i] +=
          inColor->RGBA[i] * that->mat[CAPY_IMGKERNEL_POS_TO_IDX(dx, dy)];
      }
    }

    // Set the color in the result, do not use setColor to avoid the
    // safeguard keeping values in [0, 1]
    CapyColorData* pixelColor = $(res, getColor)(&pixel);
    loop(i, 4) pixelColor->RGBA[i] = color.RGBA[i];
  }

  // Free memory
  loop(i, 2) $(range + i, destruct)();

  // Return the result image
  return res;
}

// Set the kernel to the identity
static void SetToIdentity(void) {
  methodOf(CapyImgKernel);

  // Set all the values to 0.0
  loop(i, that->sizeMat) that->mat[i] = 0.0;

  // Set the value at the center of the kernel to 1.0
  that->mat[that->sizeMat / 2] = 1.0;
}

// Set the kernel to Gaussian blur (the kernel is normalised)
// Input:
//   stdDev: the standard deviation (in pixels) of the blur
// Exception:
//   May raise CapyExc_MallocFailed.
static void SetToGaussianBlur(double const stdDev) {
  methodOf(CapyImgKernel);

  // Create the normal distribution
  double means[2] = {0.0, 0.0};
  double stdDevs[2] = {stdDev, stdDev};
  CapyDistNormal dist =
    CapyDistNormalCreate(2, means, stdDevs);

  // Loop on the relative positions in the kernel and assign the value
  // with the distribution value for that relative position
  double sum = 0.0;
  loopRange(x, that->range) loopRange(y, that->range) {
    double v[2] = {x, y};
    CapyDistEvt evt = {.vec = {.dim = 2, .vals = v}};
    that->mat[CAPY_IMGKERNEL_POS_TO_IDX(x, y)] = $(&dist, getProbability)(&evt);
    sum += that->mat[CAPY_IMGKERNEL_POS_TO_IDX(x, y)];
  }

  // Normalise
  double invSum = 1.0 / sum;
  loop(i, that->sizeMat) that->mat[i] *= invSum;

  // Free memory
  $(&dist, destruct)();
}

// Set the kernel to  derivative Gaussian blur along a given axis
// Input:
//   stdDev: the standard deviation (in pixels) of the blur
//     axis: axis of derivation (0:x, 1:y)
// Exception:
//   May raise CapyExc_MallocFailed.
static void SetToDerivativeGaussianBlur(
  double const stdDev,
  size_t const axis) {
  methodOf(CapyImgKernel);

  // Create the normal distribution
  double means[2] = {0.0, 0.0};
  double stdDevs[2] = {stdDev, stdDev};
  CapyDistNormal dist =
    CapyDistNormalCreate(2, means, stdDevs);

  // Loop on the relative positions in the kernel and assign the value
  // with the distribution value for that relative position
  loopRange(x, that->range) loopRange(y, that->range) {
    double v[2] = {x, y};
    CapyDistEvt evt = {.vec = {.dim = 2, .vals = v}};
    that->mat[CAPY_IMGKERNEL_POS_TO_IDX(x, y)] =
      $(&dist, getProbabilityDerivative)(&evt, axis);
  }

  // Normalise the kernel
  double sum = 0.0;
  loopRange(x, that->range) loopRange(y, that->range) {
    sum += fabs(that->mat[CAPY_IMGKERNEL_POS_TO_IDX(x, y)]);
  }
  double inv = 1.0 / sum;
  loopRange(x, that->range) loopRange(y, that->range) {
    that->mat[CAPY_IMGKERNEL_POS_TO_IDX(x, y)] *= inv;
  }

  // Free memory
  $(&dist, destruct)();
}

// Set the kernel to the sharpen kernel (only center 3x3 is used)
// Exception:
//   May raise CapyExc_MallocFailed.
static void SetToSharpen(void) {
  methodOf(CapyImgKernel);
  loopRange(x, that->range) loopRange(y, that->range) {
    that->mat[CAPY_IMGKERNEL_POS_TO_IDX(x, y)] = 0.0;
  }
  that->mat[CAPY_IMGKERNEL_POS_TO_IDX(0, -1)] = -1.0;
  that->mat[CAPY_IMGKERNEL_POS_TO_IDX(-1, 0)] = -1.0;
  that->mat[CAPY_IMGKERNEL_POS_TO_IDX(0, 0)] = 5.0;
  that->mat[CAPY_IMGKERNEL_POS_TO_IDX(1, 0)] = -1.0;
  that->mat[CAPY_IMGKERNEL_POS_TO_IDX(0, 1)] = -1.0;
}

// Set the kernel to the edge detection kernel (only center 3x3 is used)
// Exception:
//   May raise CapyExc_MallocFailed.
static void SetToEdgeDetection(void) {
  methodOf(CapyImgKernel);
  loopRange(x, that->range) loopRange(y, that->range) {
    that->mat[CAPY_IMGKERNEL_POS_TO_IDX(x, y)] = 0.0;
  }
  that->mat[CAPY_IMGKERNEL_POS_TO_IDX(1, -1)] = -1.0;
  that->mat[CAPY_IMGKERNEL_POS_TO_IDX(1, 0)] = -1.0;
  that->mat[CAPY_IMGKERNEL_POS_TO_IDX(1, 1)] = -1.0;
  that->mat[CAPY_IMGKERNEL_POS_TO_IDX(0, -1)] = -1.0;
  that->mat[CAPY_IMGKERNEL_POS_TO_IDX(0, 0)] = 8.0;
  that->mat[CAPY_IMGKERNEL_POS_TO_IDX(0, 1)] = -1.0;
  that->mat[CAPY_IMGKERNEL_POS_TO_IDX(-1, -1)] = -1.0;
  that->mat[CAPY_IMGKERNEL_POS_TO_IDX(-1, 0)] = -1.0;
  that->mat[CAPY_IMGKERNEL_POS_TO_IDX(-1, 1)] = -1.0;
}

// Free the memory used by a CapyImgKernel
static void Destruct(void) {
  methodOf(CapyImgKernel);
  free(that->mat);
  $(&(that->range), destruct)();
}

// Create a CapyImgKernel
// Input:
//   size: the size of the kernel (the kernel matrix is a
//         (1+2*size)x(1+2*size) matrix)
// Output:
//   Return a CapyImgKernel with identity matrix (in kernel sense)
// Exception:
//   May raise CapyExc_MallocFailed.
CapyImgKernel CapyImgKernelCreate(CapyImgDims_t const size) {
  CapyImgKernel that = {
    .size = size,
    .sizeMat = 1 + 4 * size * (1 + size),
    .range = CapyRangeImgPosCreate(-(CapyImgPos_t)size, (CapyImgPos_t)size),
    .destruct = Destruct,
    .apply = Apply,
    .setToIdentity = SetToIdentity,
    .setToGaussianBlur = SetToGaussianBlur,
    .setToDerivativeGaussianBlur = SetToDerivativeGaussianBlur,
    .setToEdgeDetection = SetToEdgeDetection,
    .setToSharpen = SetToSharpen,
  };
  safeMalloc(that.mat, that.sizeMat);
  loop(i, that.sizeMat) that.mat[i] = 0.0;
  that.mat[that.sizeMat / 2] = 1.0;
  return that;
}

// Allocate memory for a new CapyImgKernel and create it
// Input:
//   size: the size of the kernel (the kernel matrix is a
//         (1+2*size)x(1+2*size) matrix)
// Output:
//   Return a CapyImgKernel with identity matrix (in kernel sense)
// Exception:
//   May raise CapyExc_MallocFailed.
CapyImgKernel* CapyImgKernelAlloc(CapyImgDims_t const size) {
  CapyImgKernel* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyImgKernelCreate(size);
  return that;
}

// Free the memory used by a CapyImgKernel* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyImgKernel to free
void CapyImgKernelFree(CapyImgKernel** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
