// ------------------------------- imgKernel.h ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_IMGKERNEL_H
#define CAPY_IMGKERNEL_H
#include "externalHeaders.h"
#include "image.h"
#include "distribution.h"

// Description:
// Kernel class to perform operations on 2D images.

// ImgKernel object
typedef struct CapyImgKernel {

  // Size of the kernel
  CapyImgDims_t size;
  CapyPad(CapyImgDims_t, size);

  // Kernel matrix (the kernel matrix is a (1+2*size)x(1+2*size)
  // matrix). Values stored by rows.
  double* mat;

  // Size of the kernel matrix
  size_t sizeMat;

  // Range to loop on the kernel values
  CapyRangeImgPos range;

  // Destructor
  void (*destruct)(void);

  // Convolve the kernel on a given image
  // Input:
  //   img: the image to which the kernel is applied
  // Output:
  //   Return the  result of the convolution as a newly allocated image
  // Exception:
  //   May raise CapyExc_MallocFailed.
  CapyImg* (*apply)(CapyImg const* const img);

  // Set the kernel to the identity
  void (*setToIdentity)(void);

  // Set the kernel to Gaussian blur (the kernel is normalised)
  // Input:
  //   stdDev: the standard deviation (in pixels) of the blur
  // Exception:
  //   May raise CapyExc_MallocFailed.
  void (*setToGaussianBlur)(double const stdDev);

  // Set the kernel to  derivative Gaussian blur along a given axis
  // Input:
  //   stdDev: the standard deviation (in pixels) of the blur
  //     axis: axis of derivation (0:x, 1:y)
  // Exception:
  //   May raise CapyExc_MallocFailed.
  void (*setToDerivativeGaussianBlur)(
    double const stdDev,
    size_t const axis);

  // Set the kernel to the sharpen kernel (only center 3x3 is used)
  // Exception:
  //   May raise CapyExc_MallocFailed.
  void (*setToSharpen)(void);

  // Set the kernel to the edge detection kernel (only center 3x3 is used)
  // Exception:
  //   May raise CapyExc_MallocFailed.
  void (*setToEdgeDetection)(void);
} CapyImgKernel;

// Create a CapyImgKernel
// Input:
//   size: the size of the kernel (the kernel matrix is a
//         (1+2*size)x(1+2*size) matrix)
// Output:
//   Return a CapyImgKernel with identity matrix (in kernel sense)
// Exception:
//   May raise CapyExc_MallocFailed.
CapyImgKernel CapyImgKernelCreate(CapyImgDims_t const size);

// Allocate memory for a new CapyImgKernel and create it
// Input:
//   size: the size of the kernel (the kernel matrix is a
//         (1+2*size)x(1+2*size) matrix)
// Output:
//   Return a CapyImgKernel with identity matrix (in kernel sense)
// Exception:
//   May raise CapyExc_MallocFailed.
CapyImgKernel* CapyImgKernelAlloc(CapyImgDims_t const size);

// Free the memory used by a CapyImgKernel* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyImgKernel to free
void CapyImgKernelFree(CapyImgKernel** const that);
#endif
