// ---------------------------- kfoldCrossValid.h ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_KFOLD_CROSS_VALID_H
#define CAPY_KFOLD_CROSS_VALID_H
#include "externalHeaders.h"
#include "cext.h"
#include "predictor.h"
#include "dataset.h"
#include "random.h"

// Description:
// k-fold cross validation class.

// Structure to memorise the result of cross validation on Predictor
typedef struct CapyKfoldCrossValidResPredictor {

  // Number of folds
  size_t k;

  // Result of evaluation training/validation
  union {
    CapyPredictorEvaluation** eval[2];
    struct __attribute__((packed)) {
      CapyPredictorEvaluation **evalTraining, **evalValidation;
    };
  };

  // Min/avg/max accuracy on training/validation (in [0,1], higher is better)
  union {
    double accuracy[2][3];
    struct __attribute__((packed)) {
      struct {double min, avg, max;} accTraining;
      struct {double min, avg, max;} accValidation;
    };
  };

  // Destructor
  void (*destruct)(void);
} CapyKfoldCrossValidResPredictor;

// Create a CapyKfoldCrossValidResPredictor
// Input:
//   k: the number of fold
// Output:
//   Return a CapyKfoldCrossValidResPredictor
CapyKfoldCrossValidResPredictor CapyKfoldCrossValidResPredictorCreate(
  size_t const k);

// KfoldCrossValid object
typedef struct CapyKfoldCrossValid {

  // Number of fold
  size_t k;

  // Flag to memorise the verbose mode (default: false)
  bool verbose;
  CapyPad(bool, verbose);

  // Stream for the output in verbose mode (default: stdout)
  FILE* stream;

  // Seed for the pseudo-random generator (default: 0)
  CapyRandomSeed_t seed;

  // Stream for the splits definition, if not null it is used to create the
  // split during evaluation, else random splits are created (default: NULL).
  // The stream is expected to be in same format as the splits file imported
  // using openmlImport.
  FILE* streamSplit;

  // Accuracy measurement type used (default: mae for numerical predictors
  // and accuracy for categorical predictors)
  CapyPredictorAccuracyMeasure accType;
  CapyPad(CapyPredictorAccuracyMeasure, accType);

  // Destructor
  void (*destruct)(void);

  // Run the k-fold cross validation for a predictor and a dataset
  // Input:
  //   predictor: the predictor
  //      dataset: the dataset
  // Output:
  //   The dataset is split into k folds (the original dataset is not modified),
  //   the predictor is trained on all combination of (k-1) fold and evaluated
  //   on the remaining fold.
  CapyKfoldCrossValidResPredictor (*evalPredictor)(
        CapyPredictor* const predictor,
    CapyDataset const* const dataset);
} CapyKfoldCrossValid;

// Create a CapyKfoldCrossValid
// Input:
//   k: the number of fold
// Output:
//   Return a CapyKfoldCrossValid
CapyKfoldCrossValid CapyKfoldCrossValidCreate(size_t const k);

// Allocate memory for a new CapyKfoldCrossValid and create it
// Input:
//   k: the number of fold
// Exception:
//   May raise CapyExc_MallocFailed.
CapyKfoldCrossValid* CapyKfoldCrossValidAlloc(size_t const k);

// Free the memory used by a CapyKfoldCrossValid* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyKfoldCrossValid to free
void CapyKfoldCrossValidFree(CapyKfoldCrossValid** const that);
#endif
