// ---------------------------------- kmeans.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_KMEANS_H
#define CAPY_KMEANS_H
#include "externalHeaders.h"
#include "cext.h"
#include "pointCloud.h"
#include "image.h"

// Description:
// Class implementing the k-means algorithm.

// Type of initialisation for the k-mean algorithm
typedef enum CapyKMeansInit {
  capyKMeanInit_rand,
  capyKMeanInit_forgy,
  capyKMeanInit_plusplus,
} CapyKMeansInit;

// Cluster of a given point.
typedef struct CapyKMeansClusterOfPoint {

  // Id of the cluster of the point
  size_t id;

  // Distance from the point to the cluster center
  double dist;
} CapyKMeansClusterOfPoint;

// KMean class
typedef struct CapyKMeans {

  // Type of initialiation (default: plusplus)
  CapyKMeansInit typeInit;
  CapyPad(CapyKMeansInit, 0);

  // Clusters center
  CapyPointCloud* clusters;

  // Destructor
  void (*destruct)(void);

  // Cluster a point cloud.
  // Input:
  //   pointCloud: the point cloud to be clustered
  //            k: the number of clusters
  // Output:
  //   Update that->clusters.
  // Exception:
  //   May raise CapyExc_MallocFailed.
  void (*run)(
    CapyPointCloud const* const pointCloud,
                   size_t const k);

  // Get the cluster of a given point
  // Input:
  //   v: the point
  // Output:
  //   Return the cluster id for the given point
  CapyKMeansClusterOfPoint (*getClusterOfPoint)(CapyVec const* const v);

  // Apply the clustering to an image color
  // Input:
  //   img: the image
  // Output:
  //   The image colors are replaced with the color of their cluster)
  void (*applyToImgColor)(CapyImg* const img);
} CapyKMeans;

// Create a CapyKMeans
// Output:
//   Return a CapyKMeans
CapyKMeans CapyKMeansCreate(void);

// Allocate memory for a new CapyKMeans and create it
// Output:
//   Return a CapyKMeans
// Exception:
//   May raise CapyExc_MallocFailed.
CapyKMeans* CapyKMeansAlloc(void);

// Free the memory used by a CapyKMeans* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyKMeans to free
void CapyKMeansFree(CapyKMeans** const that);
#endif
