// ---------------------------- lightray.c ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache info@baillehachepascal.dev
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "lightray.h"

// Cauchy coefficients of the predefined materials
CapyRefractiveCoeff capyRefractiveCoeffs[capyMaterialIdx_nb] = {
  {.vals = {1.0, 0.0}},
  {.vals = {1.000293, 0.0}},
  {.vals = {1.522, 4590.0}},
  {.vals = {1.333, 0.0}},
  {.vals = {0.051585, 0.0}},
  {.vals = {0.27732, 0.0}},
  {.vals = {0.63660, 0.0}},
  {.vals = {2.417, 0.0}},
};

// Estimate the Cauchy coefficients from the reflectivity
// Inputs:
//   reflective: the reflectivity in [0,1]
// Output:
//   Return the estimated Cauchy coefficients (B set to 0.0, relative to vacuum)
CapyRefractiveCoeff CapyEstimateRefractiveCoeffFromReflectiveIndex(
  double const reflective) {
  CapyRefractiveCoeff coeff = { .vals = {1.0, 0.0}};
  double s = sqrt(reflective);
  coeff.vals[0] = (1.0 + s) / (1.0 - s);
  return coeff;
}

// Get the refractive index for given Cauchy coefficients and wavelength
// Inputs:
//   coeff: the Cauchy coefficients
//   lambda: the wavelength in nanometer
// Output:
//   Return the refractive index
double CapyGetRefractiveIndex(
  CapyRefractiveCoeff const coeff,
               double const lambda) {
  double index = coeff.vals[0] + coeff.vals[1] / (lambda * lambda);
  return index;
}

// Get the refraction values for given refractive coefficients, angle of
// incidence and a default wavelength (CAPY_DEFAULT_WAVELENGTH)
// Inputs:
//   coeffFrom: the Cauchy coefficients of the incoming material
//   coeffTo: the Cauchy coefficients of the outgoing material
//   theta: angle of incidence in radians (angle between the incoming ray of
//          light and the normal of the interface of materials)
// Output:
//   Return the refraction values when moving from a material with Cauchy
//   coefficients 'coeffFrom' to a material with Cauchy coefficients 'coeffTo'.
//   In other words, how much light traveling in the 'from' material goes
//   back to the 'from' material after encountering the 'to' material.
CapyRefractionValue CapyGetRefraction(
  CapyRefractiveCoeff const coeffFrom,
  CapyRefractiveCoeff const coeffTo,
               double const theta) {
  return CapyGetRefractionForWavelength(
    coeffFrom, coeffTo, theta, CAPY_DEFAULT_WAVELENGTH);
}

// Get the refraction values for given refractive coefficients, angle of
// incidence and given wavelength
// Inputs:
//   coeffFrom: the Cauchy coefficients of the incoming material
//   coeffTo: the Cauchy coefficients of the outgoing material
//   theta: angle of incidence in radians (angle between the incoming ray of
//          light and the normal of the interface of materials)
//   lambda: the wavelength (in nanometer)
// Output:
//   Return the refractive coeffcients for the given wavelength when moving
//   from a material with Cauchy coefficients 'coeffFrom' to a material with
//   Cauchy coefficients 'coeffTo'
//   In other words, how much light traveling in the 'from' material goes
//   back to the 'from' material after encountering the 'to' material.
CapyRefractionValue CapyGetRefractionForWavelength(
  CapyRefractiveCoeff const coeffFrom,
  CapyRefractiveCoeff const coeffTo,
               double const theta,
               double const lambda) {
  CapyRefractionValue refraction = {0};
  double no = CapyGetRefractiveIndex(coeffFrom, lambda);
  double ni = CapyGetRefractiveIndex(coeffTo, lambda);
  double c = no / ni * sin(theta);
  if(c >= 1.0) {

    // Total internal reflection
    refraction.reflective = 1.0;
    refraction.transmissive = 0.0;
    refraction.theta = 0.0;
  } else {
    refraction.reflective = pow((no - ni) / (no + ni), 2.0);
    refraction.transmissive = 1.0 - refraction.reflective;
    refraction.theta = asin(c);
  }
  return refraction;
}

// Get the critical refractive angle given refractive coefficients and
// wavelength
// Inputs:
//   coeffFrom: the Cauchy coefficients of the incoming material
//   coeffTo: the Cauchy coefficients of the outgoing material
//   lambda: the wavelength (in nanometer)
// Output:
//   Return the critical angle in radians.
double CapyGetCriticalRefractiveAngle(
  CapyRefractiveCoeff const coeffFrom,
  CapyRefractiveCoeff const coeffTo,
               double const lambda) {
  double ni = CapyGetRefractiveIndex(coeffFrom, lambda);
  double no = CapyGetRefractiveIndex(coeffTo, lambda);
  if(no >= ni) {

    // No critical angle
    return M_PI_4;
  } else {
    double thetaCritical = asin(no / ni);
    return thetaCritical;
  }
}

// Free the memory used by a CapyLightRay
static void Destruct(void) {
  methodOf(CapyLightRay);
  CapyVecDestruct(&(that->orig));
  CapyVecDestruct(&(that->dir));
  CapyVecDestruct(&(that->dest));
  that->parent = NULL;
}

// Create a CapyLightRay
// Output:
//   Return a CapyLightRay
CapyLightRay CapyLightRayCreate(void) {
  CapyLightRay that = {
    .orig = CapyVecCreate(3),
    .dir = CapyVecCreate(3),
    .dest = CapyVecCreate(3),
    .parent = NULL,
    .destruct = Destruct,
  };
  return that;
}

// Allocate memory for a new CapyLightRay and create it
// Output:
//   Return a CapyLightRay
// Exception:
//   May raise CapyExc_MallocFailed.
CapyLightRay* CapyLightRayAlloc(void) {
  CapyLightRay* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyLightRayCreate();
  return that;
}

// Free the memory used by a CapyLightRay* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyLightRay to free
void CapyLightRayFree(CapyLightRay** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
