// ---------------------------------- lightray.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache info@baillehachepascal.dev
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_LIGHTRAY_H
#define CAPY_LIGHTRAY_H
#include "externalHeaders.h"
#include "cext.h"
#include "capymath.h"

// Description:
// LightRay class.

// Default wavelength
#define CAPY_DEFAULT_WAVELENGTH 589.29

// Refractive index
typedef struct CapyRefractiveCoeff {

  // Cauchy coefficients (for lambda in nanometers)
  double vals[2];
} CapyRefractiveCoeff;

// Predefined materials index
typedef enum CapyMaterialIdx {
  capyMaterialIdx_vacuum,
  capyMaterialIdx_air,
  capyMaterialIdx_glass,
  capyMaterialIdx_water,
  capyMaterialIdx_silver,
  capyMaterialIdx_gold,
  capyMaterialIdx_copper,
  capyMaterialIdx_diamond,
  capyMaterialIdx_nb,
} CapyMaterialIdx;

// Cauchy coefficients of the predefined materials
extern CapyRefractiveCoeff capyRefractiveCoeffs[capyMaterialIdx_nb];

// Structure to memorise the values associated with refraction
typedef struct CapyRefractionValue {
  union {
    double vals[3];
    struct __attribute__((packed)) { double reflective, transmissive, theta; };
  };
} CapyRefractionValue;

// Get the refractive index for given Cauchy coefficients and wavelength
// Inputs:
//   coeff: the Cauchy coefficients
//   lambda: the wavelength in nanometer
// Output:
//   Return the refractive index
double CapyGetRefractiveIndex(
  CapyRefractiveCoeff const coeff,
               double const lambda);

// Estimate the Cauchy coefficients from the reflectivity
// Inputs:
//   reflectivity: the reflectivity in [0,1]
// Output:
//   Return the estimated Cauchy coefficients (B set to 0.0, relative to vacuum)
CapyRefractiveCoeff CapyEstimateRefractiveCoeffFromReflectiveIndex(
  double const reflective);

// Get the refraction values for given refractive coefficients, angle of
// incidence and a default wavelength (CAPY_DEFAULT_WAVELENGTH)
// Inputs:
//   coeffFrom: the Cauchy coefficients of the incoming material
//   coeffTo: the Cauchy coefficients of the outgoing material
//   theta: angle of incidence in radians (angle between the incoming ray of
//          light and the normal of the interface of materials)
// Output:
//   Return the refraction values when moving from a material with Cauchy
//   coefficients 'coeffFrom' to a material with Cauchy coefficients 'coeffTo'.
//   In other words, how much light traveling in the 'from' material goes
//   back to the 'from' material after encountering the 'to' material.
CapyRefractionValue CapyGetRefraction(
  CapyRefractiveCoeff const coeffFrom,
  CapyRefractiveCoeff const coeffTo,
               double const theta);

// Get the refraction values for given refractive coefficients, angle of
// incidence and given wavelength
// Inputs:
//   coeffFrom: the Cauchy coefficients of the incoming material
//   coeffTo: the Cauchy coefficients of the outgoing material
//   theta: angle of incidence in radians (angle between the incoming ray of
//          light and the normal of the interface of materials)
//   lambda: the wavelength (in nanometer)
// Output:
//   Return the refractive coeffcients for the given wavelength when moving
//   from a material with Cauchy coefficients 'coeffFrom' to a material with
//   Cauchy coefficients 'coeffTo'
//   In other words, how much light traveling in the 'from' material goes
//   back to the 'from' material after encountering the 'to' material.
CapyRefractionValue CapyGetRefractionForWavelength(
  CapyRefractiveCoeff const coeffFrom,
  CapyRefractiveCoeff const coeffTo,
               double const theta,
               double const lambda);

// Get the critical refractive angle given refractive coefficients and
// wavelength
// Inputs:
//   coeffFrom: the Cauchy coefficients of the incoming material
//   coeffTo: the Cauchy coefficients of the outgoing material
//   lambda: the wavelength (in nanometer)
// Output:
//   Return the critical angle in radians.
double CapyGetCriticalRefractiveAngle(
  CapyRefractiveCoeff const coeffFrom,
  CapyRefractiveCoeff const coeffTo,
               double const lambda);

// LightRay object
typedef struct CapyLightRay CapyLightRay;
struct CapyLightRay {

  // Origin (default: 0)
  CapyVec orig;

  // Direction (normed, default: 0)
  CapyVec dir;

  // Destination (default: 0)
  CapyVec dest;

  // Parent ray (default: NULL)
  CapyLightRay* parent;

  // Destructor
  void (*destruct)(void);
};

// Create a CapyLightRay
// Output:
//   Return a CapyLightRay
CapyLightRay CapyLightRayCreate(void);

// Allocate memory for a new CapyLightRay and create it
// Output:
//   Return a CapyLightRay
// Exception:
//   May raise CapyExc_MallocFailed.
CapyLightRay* CapyLightRayAlloc(void);

// Free the memory used by a CapyLightRay* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyLightRay to free
void CapyLightRayFree(CapyLightRay** const that);
#endif
