// ---------------------------------- list.h ----------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_LIST_H
#define CAPY_LIST_H
#include "externalHeaders.h"
#include "cext.h"
#include "array.h"
#include "comparator.h"

// Description:
// Generic list class.

// Enumeration for the types of iterator on generic list
typedef enum CapyListIteratorType {
  capyListIteratorType_forward,
  capyListIteratorType_backward,
} CapyListIteratorType;

// Iterator for generic list structure. Declaration macro for an iterator
// structure named 'name' ## Iterator, associated to a list named 'name'
// containing elements of type 'type'
#define CapyDecListIterator(name, type_)                         \
typedef struct name name;                                        \
typedef struct name ## Iterator {                                \
  size_t idx;                                                    \
  name ## Elem* curElem;                                         \
  name* list;                                                    \
  type_ datatype;                                                \
  CapyPad(type_, 0);                                             \
  CapyListIteratorType type;                                     \
  CapyPad(CapyListIteratorType, 1);                              \
  void (*destruct)(void);                                        \
  type_* (*reset)(void);                                         \
  type_* (*prev)(void);                                          \
  type_* (*next)(void);                                          \
  bool (*isActive)(void);                                        \
  type_* (*get)(void);                                           \
  void (*setType)(CapyListIteratorType type);                    \
  bool (*isFirst)(void);                                         \
  bool (*isLast)(void);                                          \
  void (*toLast)(void);                                          \
} name ## Iterator;                                              \
name ## Iterator name ## Iterator ## Create(                     \
  name* const arr, CapyListIteratorType const type);             \
name ## Iterator* name ## Iterator ## Alloc(                     \
  name* const arr, CapyListIteratorType const type);             \
void name ## Iterator ## Destruct(void);                         \
void name ## Iterator ## Free(name ## Iterator** const that);    \
type_* name ## Iterator ## Reset(void);                          \
type_* name ## Iterator ## Prev(void);                           \
type_* name ## Iterator ## Next(void);                           \
bool name ## Iterator ## IsActive(void);                         \
type_* name ## Iterator ## Get(void);                            \
bool name ## Iterator ## IsFirst(void);                          \
bool name ## Iterator ## IsLast(void);                           \
void name ## Iterator ## ToLast(void);                           \
void name ## Iterator ## SetType(CapyListIteratorType type);

// Create an iterator on a generic list
// Input:
//   list: the generic list on which to iterate
//   type: the type of iterator
// Output:
//   Return the iterator
#define CapyDefListIteratorCreate(name, type_) \
name ## Iterator name ## Iterator ## Create(   \
  name* const list,                            \
  CapyListIteratorType const type) {           \
  name ## Iterator that = {                    \
    .idx = 0,                                  \
    .curElem = NULL,                           \
    .list = list,                              \
    .type = type,                              \
    .destruct = name ## Iterator ## Destruct,  \
    .reset = name ## Iterator ## Reset,        \
    .prev = name ## Iterator ## Prev,          \
    .next = name ## Iterator ## Next,          \
    .isActive = name ## Iterator ## IsActive,  \
    .get = name ## Iterator ## Get,            \
    .setType = name ## Iterator ## SetType,    \
    .isFirst = name ## Iterator ## IsFirst,    \
    .isLast = name ## Iterator ## IsLast,      \
    .toLast = name ## Iterator ## ToLast,      \
  };                                           \
  $(&that, reset)();                           \
  return that;                                 \
}

// Allocate memory and create an iterator on a generic list
// Input:
//   list: the generic list on which to iterate
//   type: the type of iterator
// Output:
//   Return the iterator
#define CapyDefListIteratorAlloc(name, type_)     \
name ## Iterator* name ## Iterator ## Alloc(      \
  name* const list,                               \
  CapyListIteratorType const type) {              \
  name ## Iterator* that = NULL;                  \
  safeMalloc(that, 1);                            \
  if(!that) return NULL;                          \
  *that = name ## Iterator ## Create(list, type); \
  return that;                                    \
}

// Free the memory used by an iterator.
// Input:
//   that: the iterator to free
#define CapyDefListIteratorDestruct(name, type) \
void name ## Iterator ## Destruct(void) {       \
  return;                                       \
}

// Free the memory used by a pointer to an iterator and reset '*that' to NULL
// Input:
//   that: a pointer to the iterator to free
#define CapyDefListIteratorFree(name, type)                    \
void name ## Iterator ## Free(name ## Iterator** const that) { \
  if(*that == NULL) return;                                    \
  $(*that, destruct)();                                        \
  free(*that);                                                 \
  *that = NULL;                                                \
}

// Reset the iterator
// Output:
//   Return the first element of the iteration
#define CapyDefListIteratorReset(name, type_)           \
type_* name ## Iterator ## Reset(void) {                \
  name ## Iterator* that = (name ## Iterator*)capyThat; \
  assert(that->list && "List iterator uninitialised");  \
  that->idx = 0;                                        \
  switch(that->type) {                                  \
    case capyListIteratorType_forward:                  \
      that->curElem = that->list->head;                 \
      break;                                            \
    case capyListIteratorType_backward:                 \
      that->curElem = that->list->tail;                 \
      break;                                            \
  }                                                     \
  if(name ## Iterator ## IsActive())                    \
    return name ## Iterator ## Get();                   \
  else                                                  \
    return NULL;                                        \
}

// Move the iterator to the previous element
// Output:
//   Return the previous element of the iteration
#define CapyDefListIteratorPrev(name, type_)            \
type_* name ## Iterator ## Prev(void) {                 \
  name ## Iterator* that = (name ## Iterator*)capyThat; \
  --(that->idx);                                        \
  switch(that->type) {                                  \
    case capyListIteratorType_forward:                  \
      if(name ## Iterator ## IsActive())                \
        that->curElem = that->curElem->prev;            \
      break;                                            \
    case capyListIteratorType_backward:                 \
      if(name ## Iterator ## IsActive())                \
        that->curElem = that->curElem->next;            \
      break;                                            \
  }                                                     \
  if(name ## Iterator ## IsActive())                    \
    return name ## Iterator ## Get();                   \
  else                                                  \
    return NULL;                                        \
}

// Move the iterator to the next element
// Output:
//   Return the next element of the iteration
#define CapyDefListIteratorNext(name, type_)            \
type_* name ## Iterator ## Next(void) {                 \
  name ## Iterator* that = (name ## Iterator*)capyThat; \
  ++(that->idx);                                        \
  switch(that->type) {                                  \
    case capyListIteratorType_forward:                  \
      if(name ## Iterator ## IsActive())                \
        that->curElem = that->curElem->next;            \
      break;                                            \
    case capyListIteratorType_backward:                 \
      if(name ## Iterator ## IsActive())                \
        that->curElem = that->curElem->prev;            \
      break;                                            \
  }                                                     \
  if(name ## Iterator ## IsActive())                    \
    return name ## Iterator ## Get();                   \
  else                                                  \
    return NULL;                                        \
}

// Check if the iterator is on a valid element
// Output:
//   Return true if the iterator is on a valid element, else false
#define CapyDefListIteratorIsActive(name, type)         \
bool name ## Iterator ## IsActive(void) {               \
  name ## Iterator* that = (name ## Iterator*)capyThat; \
  return (that->curElem != NULL);                       \
}

// Get the current element of the iteration
// Output:
//   Return a pointer to the current element
#define CapyDefListIteratorGet(name, type_)                    \
type_* name ## Iterator ## Get(void) {                         \
  name ## Iterator* that = (name ## Iterator*)capyThat;        \
  return &(that->curElem->data);                               \
}

// Set the type of the iterator and reset it
// Input:
//   type: the new type of the iterator
#define CapyDefListIteratorSetType(name, type_)               \
void name ## Iterator ## SetType(CapyListIteratorType type) { \
  name ## Iterator* that = (name ## Iterator*)capyThat;       \
  that->type = type;                                          \
  name ## Iterator ## Reset();                                \
}

// Return true if the iterator is on the first element of the iteration
// Output:
//   Return true or false
#define CapyDefListIteratorIsFirst(name, type_)                     \
bool name ## Iterator ## IsFirst(void) {                            \
  name ## Iterator* that = (name ## Iterator*)capyThat;             \
  bool ret = false;                                                 \
  switch(that->type) {                                              \
    case capyListIteratorType_forward:                              \
      ret = (that->curElem != NULL && that->curElem->prev == NULL); \
      break;                                                        \
    case capyListIteratorType_backward:                             \
      ret = (that->curElem != NULL && that->curElem->next == NULL); \
      break;                                                        \
  }                                                                 \
  return ret;                                                       \
}

// Return true if the iterator is on the last element of the iteration
// Output:
//   Return true or false
#define CapyDefListIteratorIsLast(name, type_)                      \
bool name ## Iterator ## IsLast(void) {                             \
  name ## Iterator* that = (name ## Iterator*)capyThat;             \
  bool ret = false;                                                 \
  switch(that->type) {                                              \
    case capyListIteratorType_forward:                              \
      ret = (that->curElem != NULL && that->curElem->next == NULL); \
      break;                                                        \
    case capyListIteratorType_backward:                             \
      ret = (that->curElem != NULL && that->curElem->prev == NULL); \
      break;                                                        \
  }                                                                 \
  return ret;                                                       \
}

// Move the iterator to the last element of the iteration
#define CapyDefListIteratorToLast(name, type_)                      \
void name ## Iterator ## ToLast(void) {                             \
  name ## Iterator* that = (name ## Iterator*)capyThat;             \
  switch(that->type) {                                              \
    case capyListIteratorType_forward:                              \
      that->curElem = that->list->tail;                             \
      break;                                                        \
    case capyListIteratorType_backward:                             \
      that->curElem = that->list->head;                             \
      break;                                                        \
  }                                                                 \
}

// Definition macro calling all the submacros at once for an iterator on an
// array structure named 'name', containing elements of type 'type'
#define CapyDefListIterator(name, type)   \
  CapyDefListIteratorCreate(name, type)   \
  CapyDefListIteratorAlloc(name, type)    \
  CapyDefListIteratorDestruct(name, type) \
  CapyDefListIteratorFree(name, type)     \
  CapyDefListIteratorReset(name, type)    \
  CapyDefListIteratorPrev(name, type)     \
  CapyDefListIteratorNext(name, type)     \
  CapyDefListIteratorIsActive(name, type) \
  CapyDefListIteratorGet(name, type)      \
  CapyDefListIteratorIsFirst(name, type)  \
  CapyDefListIteratorIsLast(name, type)   \
  CapyDefListIteratorToLast(name, type)   \
  CapyDefListIteratorSetType(name, type)

// Generic double linked list structure. Declaration macro for a list
// structure named 'name', containing elements of type 'type'
#define CapyDecList(name, type)                    \
typedef struct name ## Elem name ## Elem;          \
typedef struct name ## Elem {                      \
  name ## Elem* prev;                              \
  name ## Elem* next;                              \
  type data;                                       \
  CapyPad(type, 0);                                \
} name ## Elem;                                    \
CapyDecListIterator(name, type)                    \
typedef struct name {                              \
  name ## Elem* head;                              \
  name ## Elem* tail;                              \
  name ## Iterator iter;                           \
  void (*destruct)(void);                          \
  size_t (*getSize)(void);                         \
  void (*push)(type const val);                    \
  void (*add)(type const val);                     \
  type (*pop)(void);                               \
  type (*drop)(void);                              \
  type (*get)(size_t const idx);                   \
  type* (*getPtr)(size_t const idx);               \
  void (*flush)(void);                             \
  type (*getHead)(void);                           \
  type (*getTail)(void);                           \
  type* (*getPtrHead)(void);                       \
  type* (*getPtrTail)(void);                       \
  bool (*isEmpty)(void);                           \
  void (*initIterator)(void);                      \
  void (*remove)(                                  \
      CapyComparator* cmp,                         \
    type const* const elem);                       \
  type* (*find)(                                   \
      CapyComparator* cmp,                         \
    type const* const elem);                       \
  void (*insertSort)(                              \
    CapyComparator* cmp,                           \
         type const val);                          \
} name;                                            \
name name ## Create(void);                         \
name* name ## Alloc(void);                         \
void name ## Destruct(void);                       \
void name ## Free(name** const that);              \
size_t name ## GetSize(void);                      \
void name ## Push(type const val);                 \
void name ## Add(type const val);                  \
type name ## Pop(void);                            \
type name ## Drop(void);                           \
type name ## Get(size_t const idx);                \
type* name ## GetPtr(size_t const idx);            \
void name ## Flush(void);                          \
type name ## GetHead(void);                        \
type name ## GetTail(void);                        \
type* name ## GetPtrHead(void);                    \
type* name ## GetPtrTail(void);                    \
bool name ## IsEmpty(void);                        \
void name ## Remove(                               \
      CapyComparator* cmp,                         \
    type const* const elem);                       \
type* name ## Find(                                \
      CapyComparator* cmp,                         \
    type const* const elem);                       \
void name ## InsertSort(                           \
  CapyComparator* cmp,                             \
       type const val);                            \
void name ## InitIterator(void);

// Definition macro and submacros for a list named 'name', containing elements
// of type 'type'
// Create a list containing elements of type 'type'
// Output:
//   Return a list
#define CapyDefListCreate(name, type)     \
name name ## Create(void) {               \
  return (name){                          \
    .head = NULL,                         \
    .tail = NULL,                         \
    .destruct = name ## Destruct,         \
    .getSize = name ## GetSize,           \
    .push = name ## Push,                 \
    .add = name ## Add,                   \
    .pop = name ## Pop,                   \
    .drop = name ## Drop,                 \
    .get = name ## Get,                   \
    .getPtr = name ## GetPtr,             \
    .flush = name ## Flush,               \
    .getHead = name ## GetHead,           \
    .getTail = name ## GetTail,           \
    .getPtrHead = name ## GetPtrHead,     \
    .getPtrTail = name ## GetPtrTail,     \
    .initIterator = name ## InitIterator, \
    .isEmpty = name ## IsEmpty,           \
    .remove = name ## Remove,             \
    .find = name ## Find,                 \
    .insertSort = name ## InsertSort,     \
  };                                      \
}

// Allocate memory for a new array and create it
// Output:
//   Return an array containing 'size_' elements of type 'type'
// Exception:
//   May raise CapyExc_MallocFailed.
#define CapyDefListAlloc(name, type)     \
name* name ## Alloc(void) {              \
  name* that = NULL;                     \
  safeMalloc(that, 1);                   \
  if(!that) return NULL;                 \
  *that = name ## Create();              \
  that->iter = name ## IteratorCreate(   \
    that, capyListIteratorType_forward); \
  return that;                           \
}

// Free the memory used by a list.
// Input:
//   that: the list to free
#define CapyDefListDestruct(name, type)   \
void name ## Destruct(void) {             \
  name* that = (name*)capyThat;           \
  $(that, flush)();                       \
}

// Free the memory used by a pointer to a list and reset '*that' to NULL
// Input:
//   that: a pointer to the list to free
#define CapyDefListFree(name, type)    \
void name ## Free(name** const that) { \
  if(*that == NULL) return;            \
  $(*that, destruct)();                \
  free(*that);                         \
  *that = NULL;                        \
}

// Get the size of a list
// Output
//   Return the size of the list
#define CapyDefListGetSize(name, type) \
size_t name ## GetSize(void) {         \
  name* that = (name*)capyThat;        \
  name ## Elem* elem = that->head;     \
  size_t size = 0;                     \
  while(elem != NULL) {                \
    ++size;                            \
    elem = elem->next;                 \
  }                                    \
  return size;                         \
}

// Push an element to the head of a list
// Input:
//   val: the element's value
// Exception:
//   May raise CapyExc_MallocFailed.
#define CapyDefListPush(name, type)       \
void name ## Push(type val) {             \
  name* that = (name*)capyThat;           \
  name ## Elem* elem = NULL;              \
  safeMalloc(elem, 1);                    \
  if(!elem) return;                       \
  elem->next = that->head;                \
  elem->prev = NULL;                      \
  elem->data = val;                       \
  if(that->head != NULL)                  \
    that->head->prev = elem;              \
  that->head = elem;                      \
  if(that->tail == NULL)                  \
    that->tail = elem;                    \
}

// Add an element to the tail of a list
// Input:
//   val: the element's value
// Exception:
//   May raise CapyExc_MallocFailed.
#define CapyDefListAdd(name, type)        \
void name ## Add(type val) {              \
  name* that = (name*)capyThat;           \
  name ## Elem* elem = NULL;              \
  safeMalloc(elem, 1);                    \
  if(!elem) return;                       \
  elem->prev = that->tail;                \
  elem->next = NULL;                      \
  elem->data = val;                       \
  if(that->tail != NULL)                  \
    that->tail->next = elem;              \
  that->tail = elem;                      \
  if(that->head == NULL)                  \
    that->head = elem;                    \
}

// Remove an element from the head of a list
// Output:
//   Return the removed element's value
#define CapyDefListPop(name, type) \
type name ## Pop(void) {           \
  name* that = (name*)capyThat;    \
  name ## Elem* elem = that->head; \
  that->head = elem->next;         \
  if(elem->next != NULL)           \
    elem->next->prev = NULL;       \
  if(that->tail == elem)           \
    that->tail = NULL;             \
  type val = elem->data;           \
  free(elem);                      \
  return val;                      \
}

// Remove an element from the tail of a list
// Output:
//   Return the removed element's value
#define CapyDefListDrop(name, type) \
type name ## Drop(void) {           \
  name* that = (name*)capyThat;     \
  name ## Elem* elem = that->tail;  \
  that->tail = elem->prev;          \
  if(elem->prev != NULL)            \
    elem->prev->next = NULL;        \
  if(that->head == elem)            \
    that->head = NULL;              \
  type val = elem->data;            \
  free(elem);                       \
  return val;                       \
}

// Get the value of the element in the list at the given position
// Output:
//   Return the element's value
#define CapyDefListGet(name, type)                    \
type name ## Get(size_t const idx) {                  \
  name* that = (name*)capyThat;                       \
  name ## Elem* elem = that->head;                    \
  for(size_t i = 0; i < idx && elem != NULL; ++i)     \
    elem = elem->next;                                \
  if(elem == NULL) raiseExc(CapyExc_InvalidElemIdx);  \
  return elem->data;                                  \
}

// Get a pointer to the value of the element in the list at the given
// position
// Output:
//   Return the pointer to the element's value
#define CapyDefListGetPtr(name, type)                 \
type* name ## GetPtr(size_t const idx) {              \
  name* that = (name*)capyThat;                       \
  name ## Elem* elem = that->head;                    \
  for(size_t i = 0; i < idx && elem != NULL; ++i)     \
    elem = elem->next;                                \
  if(elem == NULL) raiseExc(CapyExc_InvalidElemIdx);  \
  return &(elem->data);                               \
}

// Empty the list by removing all its elements
#define CapyDefListFlush(name, type)         \
void name ## Flush(void) {                   \
  name* that = (name*)capyThat;              \
  while(that->head != NULL) $(that, pop)();  \
}

// Get the value of the element at the head of the list
// Output:
//   Return the element's value
#define CapyDefListGetHead(name, type) \
type name ## GetHead(void) {           \
  name* that = (name*)capyThat;        \
  return that->head->data;             \
}

// Get the value of the element at the tail of the list
// Output:
//   Return the element's value
#define CapyDefListGetTail(name, type) \
type name ## GetTail(void) {           \
  name* that = (name*)capyThat;        \
  return that->tail->data;             \
}

// Get a pointer to the value of the element at the head of the list
// Output:
//   Return the pointer to the element's value
#define CapyDefListGetPtrHead(name, type) \
type* name ## GetPtrHead(void) {          \
  name* that = (name*)capyThat;           \
  return &(that->head->data);             \
}

// Get a pointer to the value of the element at the tail of the list
// Output:
//   Return the pointer to the element's value
#define CapyDefListGetPtrTail(name, type) \
type* name ## GetPtrTail(void) {          \
  name* that = (name*)capyThat;           \
  return &(that->tail->data);             \
}

// Check if a list is empty
// Output:
//   Return true if the list is empty, else false
#define CapyDefListIsEmpty(name, type)      \
bool name ## IsEmpty(void) {                \
  return (((name*)capyThat)->head == NULL); \
}

// Remove elements from the list
// Input:
//   cmp: comparator to find the element
//   elem: the element to be removed
// Output:
//   All the elements in the list equal (according to the given comparator)
//   to the given element are removed. The iterator is reset.
#define CapyDefListRemove(name, type)                    \
void name ## Remove(                                     \
    CapyComparator* cmp,                                 \
  type const* const elem) {                              \
  name* that = (name*)capyThat;                          \
  name ## Elem* ptr = that->head;                        \
  while(ptr != NULL) {                                   \
    int resCmp = $(cmp, eval)(&(ptr->data), elem);       \
    if(resCmp == 0) {                                    \
      name ## Elem* next = ptr->next;                    \
      if(ptr->prev != NULL) ptr->prev->next = ptr->next; \
      if(ptr->next != NULL) ptr->next->prev = ptr->prev; \
      if(that->head == ptr) that->head = ptr->next;      \
      if(that->tail == ptr) that->tail = ptr->prev;      \
      free(ptr);                                         \
      ptr = next;                                        \
    } else ptr = ptr->next;                              \
  }                                                      \
  $(&(that->iter), reset)();                             \
}

// Find an element in the list
// Input:
//   cmp: comparator to find the element
//   elem: the element to be found
// Output:
//   The first element in the list equal (according to the given comparator)
//   to the given element is returned. If none could be found, return NULL.
#define CapyDefListFind(name, type)                      \
type* name ## Find(                                      \
    CapyComparator* cmp,                                 \
  type const* const elem) {                              \
  name* that = (name*)capyThat;                          \
  name ## Elem* ptr = that->head;                        \
  while(ptr != NULL) {                                   \
    int resCmp = $(cmp, eval)(&(ptr->data), elem);       \
    if(resCmp == 0) return &(ptr->data);                 \
    ptr = ptr->next;                                     \
  }                                                      \
  return NULL;                                           \
}

// Insert an element in the list at a position defined by a comparator
// Input:
//   cmp: comparator to find the element position
//   elem: the element to be added
// Output:
//   Insert the element before the first element in the list equal or after
//   the added element (according to the given comparator).
#define CapyDefListInsertSort(name, type)                \
void name ## InsertSort(                                 \
  CapyComparator* cmp,                                   \
       type const val) {                                 \
  name* that = (name*)capyThat;                          \
  name ## Elem* ptr = that->head;                        \
  while(ptr != NULL) {                                   \
    int resCmp = $(cmp, eval)(&(ptr->data), &val);       \
    if(resCmp >= 0) {                                    \
      if(ptr->prev == NULL) {                            \
        $(that, push)(val);                              \
      } else {                                           \
        name ## Elem* elem = NULL;                       \
        safeMalloc(elem, 1);                             \
        if(!elem) return;                                \
        elem->prev = ptr->prev;                          \
        elem->prev->next = elem;                         \
        ptr->prev = elem;                                \
        elem->next = ptr;                                \
        elem->data = val;                                \
      }                                                  \
      return;                                            \
    }                                                    \
    ptr = ptr->next;                                     \
  }                                                      \
  $(that, add)(val);                                     \
}

// Initialise the iterator of the list, must be called after the creation
// of the list when it is created with Create(), Alloc() automatically
// initialise the iterator.
#define CapyDefListInitIterator(name, type) \
void name ## InitIterator(void) {           \
  name* that = (name*)capyThat;             \
  that->iter = name ## IteratorCreate(      \
    that, capyListIteratorType_forward);    \
}

// Definition macro calling all the submacros at once for a list structure
// named 'name', containing elements of type 'type'
#define CapyDefList(name, type)       \
  CapyDefListCreate(name, type)       \
  CapyDefListAlloc(name, type)        \
  CapyDefListDestruct(name, type)     \
  CapyDefListFree(name, type)         \
  CapyDefListGetSize(name, type)      \
  CapyDefListPush(name, type)         \
  CapyDefListAdd(name, type)          \
  CapyDefListPop(name, type)          \
  CapyDefListDrop(name, type)         \
  CapyDefListGet(name, type)          \
  CapyDefListGetPtr(name, type)       \
  CapyDefListFlush(name, type)        \
  CapyDefListGetHead(name, type)      \
  CapyDefListGetTail(name, type)      \
  CapyDefListGetPtrHead(name, type)   \
  CapyDefListGetPtrTail(name, type)   \
  CapyDefListIsEmpty(name, type)      \
  CapyDefListRemove(name, type)       \
  CapyDefListFind(name, type)         \
  CapyDefListInsertSort(name, type)   \
  CapyDefListInitIterator(name, type) \
  CapyDefListIterator(name, type)

// Declaration of lists for some basic types
CapyDecList(CapyListChar, char)
CapyDecList(CapyListUChar, unsigned char)
CapyDecList(CapyListInt8, int8_t)
CapyDecList(CapyListInt16, int16_t)
CapyDecList(CapyListInt64, int64_t)
CapyDecList(CapyListUInt8, uint8_t)
CapyDecList(CapyListUInt16, uint16_t)
CapyDecList(CapyListUInt64, uint64_t)
CapyDecList(CapyListFloat, float)
CapyDecList(CapyListDouble, double)
CapyDecList(CapyListSize, size_t)
CapyDecList(CapyListPtrChar, char*)
CapyDecList(CapyListPtrUChar, unsigned char*)
CapyDecList(CapyListPtrInt8, int8_t*)
CapyDecList(CapyListPtrInt16, int16_t*)
CapyDecList(CapyListPtrUInt8, uint8_t*)
CapyDecList(CapyListPtrUInt16, uint16_t*)
CapyDecList(CapyListPtrFloat, float*)
CapyDecList(CapyListPtrDouble, double*)
CapyDecList(CapyListArrChar, CapyArrChar)
#endif
