// ---------------------------- lsystem.c ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "lsystem.h"

// Rule for Lindenmayer's algae L-System
static CapyLSysLindenMayerAlgaeState* LindenmayerAlgaeRule(
   size_t const idx,
  size_t* const lenOut) {
  CapyLSysLindenmayerAlgae* that = (CapyLSysLindenmayerAlgae*)capyThat;
  CapyLSysLindenMayerAlgaeState* out = NULL;
  if(that->state[idx].val == 'A') {
    safeMalloc(out, 2);
    assert(out);
    out[0] = (CapyLSysLindenMayerAlgaeState){.val='A'};
    out[1] = (CapyLSysLindenMayerAlgaeState){.val='B'};
    *lenOut = 2;
  } else if(that->state[idx].val == 'B') {
    safeMalloc(out, 1);
    assert(out);
    out[0] = (CapyLSysLindenMayerAlgaeState){.val='A'};
    *lenOut = 1;
  }
  return out;
}

// Lindenmayer's algae L-System definition
CapyDefLSystem(
  CapyLSysLindenmayerAlgae, CapyLSysLindenMayerAlgaeState, \
  LindenmayerAlgaeRule)

// Rule for Koch snowflake L-System
static CapyLSysKochSnowflakeState* KochSnowflakeRule(
   size_t const idx,
  size_t* const lenOut) {
  CapyLSysKochSnowflake* that = (CapyLSysKochSnowflake*)capyThat;
  CapyLSysKochSnowflakeState* out = NULL;
  if(that->state[idx].val == 'F') {
    safeMalloc(out, 10);
    assert(out);
    out[0] = (CapyLSysKochSnowflakeState){.val='D'};
    out[1] = (CapyLSysKochSnowflakeState){.val='F'};
    out[2] = (CapyLSysKochSnowflakeState){.val='+'};
    out[3] = (CapyLSysKochSnowflakeState){.val='F'};
    out[4] = (CapyLSysKochSnowflakeState){.val='-'};
    out[5] = (CapyLSysKochSnowflakeState){.val='-'};
    out[6] = (CapyLSysKochSnowflakeState){.val='F'};
    out[7] = (CapyLSysKochSnowflakeState){.val='+'};
    out[8] = (CapyLSysKochSnowflakeState){.val='F'};
    out[9] = (CapyLSysKochSnowflakeState){.val='U'};
    *lenOut = 10;
  } else {
    safeMalloc(out, 1);
    assert(out);
    out[0] = (CapyLSysKochSnowflakeState){.val=that->state[idx].val};
    *lenOut = 1;
  }
  return out;
}

// Koch snowflake L-System definition
CapyDefLSystem(
  CapyLSysKochSnowflake, CapyLSysKochSnowflakeState, KochSnowflakeRule)

// Rule for fractal binary tree L-System
static CapyLSysFractalBinaryTreeState* FractalBinaryTreeRule(
   size_t const idx,
  size_t* const lenOut) {
  CapyLSysFractalBinaryTree* that = (CapyLSysFractalBinaryTree*)capyThat;
  CapyLSysFractalBinaryTreeState* out = NULL;
  if(that->state[idx].val == '0') {
    safeMalloc(out, 5);
    assert(out);
    out[0] = (CapyLSysFractalBinaryTreeState){.val='1'};
    out[1] = (CapyLSysFractalBinaryTreeState){.val='['};
    out[2] = (CapyLSysFractalBinaryTreeState){.val='0'};
    out[3] = (CapyLSysFractalBinaryTreeState){.val=']'};
    out[4] = (CapyLSysFractalBinaryTreeState){.val='0'};
    *lenOut = 5;
  } else if(that->state[idx].val == '1') {
    safeMalloc(out, 2);
    assert(out);
    out[0] = (CapyLSysFractalBinaryTreeState){.val='1'};
    out[1] = (CapyLSysFractalBinaryTreeState){.val='1'};
    *lenOut = 2;
  } else {
    safeMalloc(out, 1);
    assert(out);
    out[0] = (CapyLSysFractalBinaryTreeState){.val=that->state[idx].val};
    *lenOut = 1;
  }
  return out;
}

// Fractal binary tree L-System definition
CapyDefLSystem(
  CapyLSysFractalBinaryTree, CapyLSysFractalBinaryTreeState,
  FractalBinaryTreeRule)

// Rule for Hilbert curve L-System
static CapyLSysHilbertCurveState* HilbertCurveRule(
   size_t const idx,
  size_t* const lenOut) {
  CapyLSysHilbertCurve* that = (CapyLSysHilbertCurve*)capyThat;
  CapyLSysHilbertCurveState* out = NULL;
  if(that->state[idx].val == 'A') {
    safeMalloc(out, 11);
    assert(out);
    out[0] = (CapyLSysHilbertCurveState){.val='+'};
    out[1] = (CapyLSysHilbertCurveState){.val='B'};
    out[2] = (CapyLSysHilbertCurveState){.val='F'};
    out[3] = (CapyLSysHilbertCurveState){.val='-'};
    out[4] = (CapyLSysHilbertCurveState){.val='A'};
    out[5] = (CapyLSysHilbertCurveState){.val='F'};
    out[6] = (CapyLSysHilbertCurveState){.val='A'};
    out[7] = (CapyLSysHilbertCurveState){.val='-'};
    out[8] = (CapyLSysHilbertCurveState){.val='F'};
    out[9] = (CapyLSysHilbertCurveState){.val='B'};
    out[10] = (CapyLSysHilbertCurveState){.val='+'};
    *lenOut = 11;
  } else if(that->state[idx].val == 'B') {
    safeMalloc(out, 11);
    assert(out);
    out[0] = (CapyLSysHilbertCurveState){.val='-'};
    out[1] = (CapyLSysHilbertCurveState){.val='A'};
    out[2] = (CapyLSysHilbertCurveState){.val='F'};
    out[3] = (CapyLSysHilbertCurveState){.val='+'};
    out[4] = (CapyLSysHilbertCurveState){.val='B'};
    out[5] = (CapyLSysHilbertCurveState){.val='F'};
    out[6] = (CapyLSysHilbertCurveState){.val='B'};
    out[7] = (CapyLSysHilbertCurveState){.val='+'};
    out[8] = (CapyLSysHilbertCurveState){.val='F'};
    out[9] = (CapyLSysHilbertCurveState){.val='A'};
    out[10] = (CapyLSysHilbertCurveState){.val='-'};
    *lenOut = 11;
  } else {
    safeMalloc(out, 1);
    assert(out);
    out[0] = (CapyLSysHilbertCurveState){.val=that->state[idx].val};
    *lenOut = 1;
  }
  return out;
}

// Hilbert curve L-System definition
CapyDefLSystem(
  CapyLSysHilbertCurve, CapyLSysHilbertCurveState, HilbertCurveRule)

// Rule for Heighway dragon curve L-System
static CapyLSysHeighwayDragonCurveState* HeighwayDragonCurveRule(
   size_t const idx,
  size_t* const lenOut) {
  CapyLSysHeighwayDragonCurve* that = (CapyLSysHeighwayDragonCurve*)capyThat;
  CapyLSysHeighwayDragonCurveState* out = NULL;
  if(that->state[idx].val == 'X') {
    char const* const str = "X+YF+";
    size_t const l = strlen(str);
    safeMalloc(out, l);
    assert(out);
    loop(i, l) {
      out[i] = (CapyLSysHeighwayDragonCurveState){.val=str[i]};
    }
    *lenOut = l;
  } else if(that->state[idx].val == 'Y') {
    char const* const str = "-FX-Y";
    size_t const l = strlen(str);
    safeMalloc(out, l);
    assert(out);
    loop(i, l) {
      out[i] = (CapyLSysHeighwayDragonCurveState){.val=str[i]};
    }
    *lenOut = l;
  } else {
    safeMalloc(out, 1);
    assert(out);
    out[0] = (CapyLSysHeighwayDragonCurveState){.val=that->state[idx].val};
    *lenOut = 1;
  }
  return out;
}

// Heighway dragon curve L-System definition
CapyDefLSystem(
  CapyLSysHeighwayDragonCurve,
  CapyLSysHeighwayDragonCurveState,
  HeighwayDragonCurveRule)

// Rule for Gosper curve L-System
static CapyLSysGosperCurveState* GosperCurveRule(
   size_t const idx,
  size_t* const lenOut) {
  CapyLSysGosperCurve* that = (CapyLSysGosperCurve*)capyThat;
  CapyLSysGosperCurveState* out = NULL;
  if(that->state[idx].val == 'A') {
    char const* const str = "sA-B--B+A++AA+B-S";
    size_t const l = strlen(str);
    safeMalloc(out, l);
    assert(out);
    loop(i, l) {
      out[i] = (CapyLSysGosperCurveState){.val=str[i]};
    }
    *lenOut = l;
  } else if(that->state[idx].val == 'B') {
    char const* const str = "s+A-BB--B-A++A+BS";
    size_t const l = strlen(str);
    safeMalloc(out, l);
    assert(out);
    loop(i, l) {
      out[i] = (CapyLSysGosperCurveState){.val=str[i]};
    }
    *lenOut = l;
  } else {
    safeMalloc(out, 1);
    assert(out);
    out[0] = (CapyLSysGosperCurveState){.val=that->state[idx].val};
    *lenOut = 1;
  }
  return out;
}

// Gosper curve L-System definition
CapyDefLSystem(
  CapyLSysGosperCurve, CapyLSysGosperCurveState, GosperCurveRule)
