// ---------------------------------- lsystem.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_LSYSTEM_H
#define CAPY_LSYSTEM_H
#include "externalHeaders.h"
#include "cext.h"

// Description:
// LSystem class.

// Declaration macro for a LSystem object with name N and data type T (must
// have a destruct method, but maybe null).
#define CapyDecLSystem(N, T)        \
typedef struct N {                  \
  T* state;                         \
  size_t len;                       \
  void* params;                     \
  void (*destruct)(void);           \
  void (*step)(void);               \
  T* (*rule)(                       \
    size_t const idx,               \
    size_t* const lenOut);          \
} N;                                \
void N ## Destruct(void);           \
void N ## Step(void);               \
N N ## Create(                      \
      T* const initState,           \
  size_t const len);                \
N* N ## Alloc(                      \
      T* const initState,           \
  size_t const len);                \
void N ## Free(N** const that);

// Definition macro for a LSystem object with name N, data type T (must
// have a destruct method), and rule R. The rule take the index of the current cymbol in the current state of the L-System and return a null terminated
// array of symbols (eventually empty). When calling 'step', the rule
// is applied to each symbol of the current state, and the current state is
// replaced by the result state.
#define CapyDefLSystem(N, T, R)                       \
void N ## Destruct(void) {                            \
  N* that = (N*)capyThat;                             \
  loop(i, that->len) if(that->state[i].destruct) {    \
    $(that->state + i, destruct)();                   \
  }                                                   \
  free(that->state);                                  \
  that->state = NULL;                                 \
  that->len = 0;                                      \
}                                                     \
void N ## Step(void) {                                \
  N* that = (N*)capyThat;                             \
  T* res = NULL;                                      \
  size_t lenRes = 0;                                  \
  loop(idx, that->len) {                              \
    size_t lenOut = 0;                                \
    T* out = $(that, rule)(idx, &lenOut);             \
    if(out != NULL) {                                 \
      safeRealloc(res, lenRes + lenOut);              \
      assert(res != NULL);                            \
      loop(i, lenOut) res[lenRes + i] = out[i];       \
      free(out);                                      \
      lenRes += lenOut;                               \
    }                                                 \
  }                                                   \
  loop(i, that->len) if(that->state[i].destruct) {    \
    $(that->state + i, destruct)();                   \
  }                                                   \
  free(that->state);                                  \
  that->state = res;                                  \
  that->len = lenRes;                                 \
}                                                     \
N N ## Create(                                        \
      T* const initState,                             \
  size_t const len) {                                 \
  N that = {                                          \
    .state = NULL,                                    \
    .len = 0,                                         \
    .params = NULL,                                   \
    .destruct = N ## Destruct,                        \
    .step = N ## Step,                                \
    .rule = R,                                        \
  };                                                  \
  if(len > 0) {                                       \
    safeMalloc(that.state, len);                      \
    assert(that.state);                               \
    loop(i, len) that.state[i] = initState[i];        \
    that.len = len;                                   \
  }                                                   \
  return that;                                        \
}                                                     \
N* N ## Alloc(                                        \
      T* const initState,                             \
  size_t const len) {                                 \
  N* that = NULL;                                     \
  safeMalloc(that, 1);                                \
  if(!that) return NULL;                              \
  *that = N ## Create(initState, len);                \
  return that;                                        \
}                                                     \
void N ## Free(N** const that) {                      \
  if(that == NULL || *that == NULL) return;           \
  $(*that, destruct)();                               \
  free(*that);                                        \
  *that = NULL;                                       \
}

// State for the Lindenmayer algae L-System
typedef struct CapyLSysLindenMayerAlgaeState {
  char val;
  CapyPad(char, val);
  void (*destruct)(void);
} CapyLSysLindenMayerAlgaeState;

// Lindenmayer's algae L-System declaration
CapyDecLSystem(CapyLSysLindenmayerAlgae, CapyLSysLindenMayerAlgaeState)

// State for the Koch snowflake
typedef struct CapyLSysKochSnowflakeState {
  char val;
  CapyPad(char, val);
  void (*destruct)(void);
} CapyLSysKochSnowflakeState;

// Koch snowflake L-System declaration
CapyDecLSystem(CapyLSysKochSnowflake, CapyLSysKochSnowflakeState)

// State for the fractal binary tree
typedef struct CapyLSysFractalBinaryTreeState {
  char val;
  CapyPad(char, val);
  void (*destruct)(void);
} CapyLSysFractalBinaryTreeState;

// Fractal binary tree L-System declaration
CapyDecLSystem(CapyLSysFractalBinaryTree, CapyLSysFractalBinaryTreeState)

// State for the Hilbert curve
typedef struct CapyLSysHilbertCurveState {
  char val;
  CapyPad(char, val);
  void (*destruct)(void);
} CapyLSysHilbertCurveState;

// Hilbert curve L-System declaration
CapyDecLSystem(CapyLSysHilbertCurve, CapyLSysHilbertCurveState)

// State for the Heighway dragon curve
typedef struct CapyLSysHeighwayDragonCurveState {
  char val;
  CapyPad(char, val);
  void (*destruct)(void);
} CapyLSysHeighwayDragonCurveState;

// Heighway dragon curve L-System declaration
CapyDecLSystem(CapyLSysHeighwayDragonCurve, CapyLSysHeighwayDragonCurveState)

// State for the Gosper dragon curve
typedef struct CapyLSysGosperCurveState {
  char val;
  CapyPad(char, val);
  void (*destruct)(void);
} CapyLSysGosperCurveState;

// Gosper curve L-System declaration
CapyDecLSystem(CapyLSysGosperCurve, CapyLSysGosperCurveState)
#endif
