// ------------------------------ mathfun.c ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "mathfun.h"
#include "capymath.h"
#include "dataset.h"

// Delta used to evaluate the derivative of the function
#define CAPY_DERIVATIVE_EPSILON 1e-9
#define CAPY_DERIVATIVE_INV_EPSILON 1e9

// Evaluate the function for a given input.
// Input:
//    in: the input vector
//   out: array of double updated with the result of evaluation
static void Eval(
  double const* const in,
        double* const out) {
  (void)in; (void)out;
  raiseExc(CapyExc_UndefinedExecution);
  assert(false && "MathFun.eval is undefined.");
}

// Evaluate the derivative of the function for a given input and
// dimension
// Input:
//     in: the input vector
//   iDim: the derivation dimension
//    out: array of dimOut double updated with the result of derivation
// Exceptions:
//   May raise CapyExc_MallocFailed
static void EvalDerivative(
  double* const in,
   size_t const iDim,
  double* const out) {
  methodOf(CapyMathFun);
  double const backupVal = in[iDim];
  in[iDim] = in[iDim] + 0.5 * CAPY_DERIVATIVE_EPSILON;
  $(that, eval)(in, out);
  in[iDim] -= CAPY_DERIVATIVE_EPSILON;
  $(that, eval)(in, that->outBuffer);
  loop(iOut, that->dimOut) {
    out[iOut] -= that->outBuffer[iOut];
    out[iOut] *= CAPY_DERIVATIVE_INV_EPSILON;
  }
  in[iDim] = backupVal;
}

// Recursive function for the evaluation of the integration
static void EvalIntegralRec(
            CapyMathFun* const that,
                 double* const in,
  CapyRangeDouble const* const domains,
                 double* const out,
                  size_t const iDim) {
  if(iDim < that->dimIn - 1) {
    double u[2][that->dimOut];
    loop(i, 2) {
      in[iDim] = domains[iDim].vals[i];
      EvalIntegralRec(that, in, domains, u[i], iDim + 1);
    }
    loop(i, that->dimOut) {
      out[i] =
        0.5 * (domains[iDim].max - domains[iDim].min) * (u[0][i] + u[1][i]);
    }
  } else {
    double u[2][that->dimOut];
    loop(i, 2) {
      in[iDim] = domains[iDim].vals[i];
      $(that, eval)(in, u[i]);
    }
    loop(i, that->dimOut) {
      out[i] =
        0.5 * (domains[iDim].max - domains[iDim].min) * (u[0][i] + u[1][i]);
    }
  }
}

// Evaluate the integration of the function over a domain
// Input:
//   domains: the domain over which to integrate, array of that->dimIn ranges
//   out: the result of integration
// Output:
//   'out' is updated. Uses the trapezoidal rule to approximate the integration.
static void EvalIntegral(
  CapyRangeDouble const* const domains,
                 double* const out) {
  methodOf(CapyMathFun);
  double* in = NULL;
  safeMalloc(in, that->dimIn);
  if(in == NULL) return;
  EvalIntegralRec(that, in, domains, out, 0);
  free(in);
}

// Evaluate the jacobian (aka gradient if dimOut==1) of the function for a given
// input
// Input:
//          in: the input vector
//    jacobian: array of dimIn*dimOut double updated with the jacobian,
//              column i is derivative in dimension i, stored by row
// Exceptions:
//   May raise CapyExc_MallocFailed
static void EvalJacobian(
  double* const in,
  double* const jacobian) {
  methodOf(CapyMathFun);
  CapyMat jacobianTransp = {.vals = NULL};
  try {
    jacobianTransp = CapyMatCreate(that->dimOut, that->dimIn);
    loop(i, that->dimIn) {
      $(that, evalDerivative)(
        in, i, jacobianTransp.vals + i * jacobianTransp.nbCol);
    }
    CapyMat j = {
      .nbCol = jacobianTransp.nbRow,
      .nbRow = jacobianTransp.nbCol,
      .vals = jacobian
    };
    CapyMatTransp(&jacobianTransp, &j);
  } endCatch;
  CapyMatDestruct(&jacobianTransp);
  CapyForwardExc();
}

// Evaluate the divergence of the function for a given input
// Input:
//     in: the input vector
// Output:
//   Return the divergence.
// Exceptions:
//   May raise CapyExc_MallocFailed
static double EvalDivergence(double* const in) {
  methodOf(CapyMathFun);
  struct {
    double* out;
    double divergence;
  } volatiles = {NULL, 0.0};
  try {
    safeMalloc(volatiles.out, that->dimOut);
    if(!(volatiles.out)) return 0.0;
    loop(iDim, that->dimIn) {
      $(that, evalDerivative)(in, iDim, volatiles.out);
      volatiles.divergence += volatiles.out[iDim];
    }
  } endCatch;
  free(volatiles.out);
  CapyForwardExc();
  return volatiles.divergence;
}

// Find the solution of f(in)=out using that->nbIterSolveNewtonMethod (by
// default: 10) iterations of the Newton method
// Input:
//    in: initial approximation of the solution, updated with the result
//   out: output values
// Exceptions:
//   May raise CapyExc_MallocFailed
static void SolveByNewtonMethod(
        double* const in,
  double const* const out) {
  methodOf(CapyMathFun);

  // cf https://en.wikipedia.org/wiki/Newton%27s_method#Generalizations
  CapyVec out2 = { .dim = 0, .vals = NULL };
  CapyMat jacobian = { .nbCol = 0, .nbRow = 0, .vals = NULL };
  CapyMat invJacobian = { .nbCol = 0, .nbRow = 0, .vals = NULL };
  try {
    out2 = CapyVecCreate(that->dimOut);
    jacobian = CapyMatCreate(that->dimIn, that->dimOut);
    invJacobian = CapyMatCreate(that->dimOut, that->dimIn);
    loop(iIter, that->nbIterSolveNewtonMethod) {
      $(that, eval)(in, out2.vals);
      loop(i, that->dimOut) out2.vals[i] -= out[i];
      $(that, evalJacobian)(in, jacobian.vals);
      CapyMatInv(&jacobian, &invJacobian);
      loop(i, that->dimIn) loop(j , that->dimOut) {
        in[i] -= invJacobian.vals[i * that->dimOut + j] * out2.vals[j];
      }
    }
  } endCatch;
  CapyVecDestruct(&out2);
  CapyMatDestruct(&jacobian);
  CapyMatDestruct(&invJacobian);
  CapyForwardExc();
}

// Free the memory used by a CapyMathFun
static void Destruct(void) {
  methodOf(CapyMathFun);
  free(that->outBuffer);
  free(that->domains);
}

// Create a CapyMathFun
// Input:
//    dimIn: input dimension
//   dimOut: output dimension
// Output:
//   Return a CapyMathFun
CapyMathFun CapyMathFunCreate(
  size_t const dimIn,
  size_t const dimOut) {
  CapyMathFun that = {
    .dimIn = dimIn,
    .dimOut = dimOut,
    .domains = NULL,
    .nbIterSolveNewtonMethod = 10,
    .destruct = Destruct,
    .eval = Eval,
    .evalDerivative = EvalDerivative,
    .solveByNewtonMethod = SolveByNewtonMethod,
    .evalJacobian = EvalJacobian,
    .evalDivergence = EvalDivergence,
    .evalIntegral = EvalIntegral,
  };
  safeMalloc(that.outBuffer, dimOut);
  safeMalloc(that.domains, dimIn);
  if(that.domains) loop(iIn, dimIn) {
    that.domains[iIn] = CapyRangeDoubleCreate(0.0, 1.0);
  }
  return that;
}

// Evaluate the hyperplane function for a given input.
// Input:
//    in: the input vector
//   out: array of double updated with the result of evaluation
static void EvalHyperplane(
  double const* const in,
        double* const out) {
  methodOf(CapyHyperplane);
  out[0] = 0.0;
  loop(i, that->dimIn) out[0] += in[i] * that->u.vals[i];
  out[0] += that->u.vals[that->dimIn];
}

// Free the memory used by a CapyHyperplane
// Input:
//   that: the CapyHyperplane to free
static void HyperplaneDestruct(void) {
  methodOf(CapyHyperplane);
  $(that, destructCapyMathFun)();
  CapyVecDestruct(&(that->u));
}

// Create a CapyHyperplane
// Input:
//    dimIn: input dimension
// Output:
//   Return a CapyHyperplane.
CapyHyperplane CapyHyperplaneCreate(size_t const dimIn) {
  CapyHyperplane that;
  CapyInherits(that, CapyMathFun, (dimIn, 1));
  that.u = CapyVecCreate(dimIn + 1);
  that.eval = EvalHyperplane;
  that.destruct = HyperplaneDestruct;
  return that;
}

// Allocate memory and create a CapyHyperplane
// Input:
//    dimIn: input dimension
// Output:
//   Return a CapyHyperplane.
CapyHyperplane* CapyHyperplaneAlloc(size_t const dimIn) {
  CapyHyperplane* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyHyperplaneCreate(dimIn);
  return that;
}

// Free the memory used by a CapyHyperplane* and reset '*that' to
// NULL
// Input:
//   that: the CapyHyperplane to free
void CapyHyperplaneFree(CapyHyperplane** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Evaluate the linear combination for a given input.
// Input:
//    in: the input vector
//   out: array of double updated with the result of evaluation
static void LinCombFunEval(
  double const* const in,
        double* const out) {
  methodOf(CapyLinCombFun);

  // Reset the temporary vector with the bias
  CapyVecCopy(&(that->bias), &(that->out));

  // Loop on the combined functions
  loop(iComb, that->nbComb) {

    // Evaluate the function
    $(that->combFuns[iComb], eval)(in, out);

    // Add the result to the combinaison accounting for the coefficient
    loop(iDim, that->dimOut) {
      that->out.vals[iDim] += that->coeff.vals[iComb] * out[iDim];
    }
  }

  // Copy the result of combination to the output in argument
  loop(iDim, that->dimOut) out[iDim] = that->out.vals[iDim];
}

// Calculate the combination coefficients that best fit a dataset
// Input:
//   dataset: the dataset
//   iOutput: the index of the output
// Output:
//   Calculate the best fit coefficients using linear regression. The
//   number of input fields in the dataset must match the input dimension
//   of combined functions. Can be used for dimOut=1 only.
static void LinCombFunLinearRegression(
  CapyDataset const* const dataset,
              size_t const iOutput) {
  methodOf(CapyLinCombFun);

  // Check the input and output dimensions of combined functions
  size_t const nbInput = $(dataset, getNbInput)();
  loop(iComb, that->nbComb) {
    if(
      that->combFuns[iComb]->dimIn != nbInput ||
      that->combFuns[iComb]->dimOut != 1
    ) {
      raiseExc(CapyExc_InvalidParameters);
    }
  }

  // Convert the dataset into a matrix
  CapyMat mDataset = $(dataset, cvtToMatForNumericalPredictor)(iOutput);

  // Execute the linear regression
  $(that, linearRegressionFromMat)(&mDataset);

  // Free memory
  CapyMatDestruct(&mDataset);
}

// Calculate the combination coefficients that best fit a matrix
// Input:
//   mDataset: the matrix
// Output:
//   Calculate the best fit coefficients using linear regression. The
//   number of column must match the input dimension of the combined
//   function plus one. The target is the last column.
static void LinCombFunLinearRegressionFromMat(CapyMat const* const mDataset) {
  methodOf(CapyLinCombFun);

  // Check the input and output dimensions of combined functions
  size_t const nbInput = mDataset->nbCol - 1;
  loop(iComb, that->nbComb) {
    if(
      that->combFuns[iComb]->dimIn != nbInput ||
      that->combFuns[iComb]->dimOut != 1
    ) {
      raiseExc(CapyExc_InvalidParameters);
    }
  }

  // Create a temporary matrix for the regression, same number of rows as
  // the number of row in the dataset, same number of columns as the number
  // of combined functions plus one for the bias.
  CapyMat m = CapyMatCreate(that->nbComb + 1, mDataset->nbRow);

  // Set the temporary matrix values to the ouput of the combined functions
  // for input values from the dataset. Last column is for the bias, hence 1.0.
  loop(iRow, mDataset->nbRow) {
    loop(iComb, that->nbComb) {
      $(that->combFuns[iComb], eval)(
        mDataset->vals + iRow * mDataset->nbCol,
        m.vals + iRow * m.nbCol + iComb);
    }
    m.vals[iRow * m.nbCol + that->nbComb] = 1.0;
  }

  // Create a temporary vector for the output values, size equal to the
  // number of rows in the dataset multiplied by the number of output
  CapyVec u = CapyVecCreate(mDataset->nbRow);

  // Set the temporary vector's values to the dataset output values
  loop(iRow, mDataset->nbRow) {
    u.vals[iRow] = mDataset->vals[iRow * mDataset->nbCol + nbInput];
  }

  // Calculate the pseudo inverse of the temporary matrix
  CapyMat mInv = CapyMatCreate(m.nbRow, m.nbCol);
  CapyMatInv(&m, &mInv);

  // Apply the pseudo inverse to the temporary vector
  CapyVec v = CapyVecCreate(that->nbComb + 1);
  CapyMatProdVec(&mInv, &u, &v);

  // Update the linear combination coefficients and bias with the result
  loop(iComb, that->nbComb) that->coeff.vals[iComb] = v.vals[iComb];
  that->bias.vals[0] = v.vals[that->nbComb];

  // Free memory
  CapyVecDestruct(&v);
  CapyVecDestruct(&u);
  CapyMatDestruct(&mInv);
  CapyMatDestruct(&m);
}

// Free the memory used by a CapyLinCombFun
// Input:
//   that: the CapyLinCombFun to free
static void LinCombFunDestruct(void) {
  methodOf(CapyLinCombFun);
  $(that, destructCapyMathFun)();
  CapyVecDestruct(&(that->coeff));
  CapyVecDestruct(&(that->bias));
  CapyVecDestruct(&(that->out));
  free(that->combFuns);
}

// Create a CapyLinCombFun
// Input:
//    nbComb: number of linearly combined functions
//    dimIn: input dimension
//    dimOut: output dimension
// Output:
//   Return a CapyLinCombFun.
CapyLinCombFun CapyLinCombFunCreate(
  size_t const nbComb,
  size_t const dimIn,
  size_t const dimOut) {
  CapyLinCombFun that = {0};
  CapyInherits(that, CapyMathFun, (dimIn, dimOut));
  that.nbComb = nbComb;
  that.coeff = CapyVecCreate(nbComb);
  that.bias = CapyVecCreate(dimOut);
  that.out = CapyVecCreate(dimOut);
  that.combFuns = NULL;
  that.destruct = LinCombFunDestruct;
  that.eval = LinCombFunEval;
  that.linearRegression = LinCombFunLinearRegression;
  that.linearRegressionFromMat = LinCombFunLinearRegressionFromMat;
  safeMalloc(that.combFuns, nbComb);
  loop(iComb, nbComb) that.combFuns[iComb] = NULL;
  return that;
}

// Allocate memory and create a CapyLinCombFun
// Input:
//    nbComb: number of linearly combined functions
//    dimIn: input dimension
//    dimOut: output dimension
// Output:
//   Return a CapyLinCombFun.
CapyLinCombFun* CapyLinCombFunAlloc(
  size_t const nbComb,
  size_t const dimIn,
  size_t const dimOut) {
  CapyLinCombFun* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyLinCombFunCreate(nbComb, dimIn, dimOut);
  return that;
}

// Free the memory used by a CapyLinCombFun* and reset '*that' to
// NULL
// Input:
//   that: the CapyLinCombFun to free
void CapyLinCombFunFree(CapyLinCombFun** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
