// ------------------------------- mathfun.h -------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_MATHFUN_H
#define CAPY_MATHFUN_H
#include "externalHeaders.h"
#include "cext.h"
#include "range.h"
#include "capymath.h"

// Description:
// Class to manipulate mathematical functions.

// CapyMathFun object definition macro
#define CapyMathFunDef {                   \
  union {                                  \
    size_t dims[2];                        \
    struct __attribute__((packed)) {       \
      size_t dimIn, dimOut;                \
    };                                     \
  };                                       \
  size_t nbIterSolveNewtonMethod;          \
  CapyRangeDouble* domains;                \
  double* outBuffer;                       \
  void (*destruct)(void);                  \
  void (*eval)(                            \
    double const* const in,                \
          double* const out);              \
  void (*evalJacobian)(                    \
    double* const in,                      \
    double* const jacobian);               \
  void (*evalDerivative)(                  \
       double* const in,                   \
        size_t const iDim,                 \
       double* const out);                 \
  void (*evalIntegral)(                    \
    CapyRangeDouble const* const domains,  \
                   double* const out);     \
  double (*evalDivergence)(                \
    double* const in);                     \
  void (*solveByNewtonMethod)(             \
          double* const in,                \
    double const* const out);              \
}

// CapyMathFun object
typedef struct CapyMathFun CapyMathFunDef CapyMathFun;

// Create a CapyMathFun
// Input:
//    dimIn: input dimension
//   dimOut: output dimension
// Output:
//   Return a CapyMathFun. domains set by default to [0, 1]
CapyMathFun CapyMathFunCreate(
  size_t const dimIn,
  size_t const dimOut);

// Hyperplane class
typedef struct CapyHyperplane {

  // Inherit CapyMathFun
  struct CapyMathFunDef;

  // Hyperplane parameters, the hyperplane is defined as the points X such as
  // H(X)=u.[X|1]=0
  CapyVec u;

  // Destructor
  void (*destructCapyMathFun)(void);
} CapyHyperplane;

// Create a CapyHyperplane
// Input:
//    dimIn: input dimension
// Output:
//   Return a CapyHyperplane.
CapyHyperplane CapyHyperplaneCreate(size_t const dimIn);

// Allocate memory and create a CapyHyperplane
// Input:
//    dimIn: input dimension
// Output:
//   Return a CapyHyperplane.
CapyHyperplane* CapyHyperplaneAlloc(size_t const dimIn);

// Free the memory used by a CapyHyperplane* and reset '*that' to
// NULL
// Input:
//   that: the CapyHyperplane to free
void CapyHyperplaneFree(CapyHyperplane** const that);

// Predeclaration
typedef struct CapyDataset CapyDataset;

// Class used to create classes which are linear combination of CapyMathFun
typedef struct CapyLinCombFun {

  // Inherit CapyMathFun
  struct CapyMathFunDef;

  // Number of linearly combined functions
  size_t nbComb;

  // Vector of coefficients for the combination
  CapyVec coeff;

  // Bias coefficient
  CapyVec bias;

  // Temporary vector for calculation
  CapyVec out;

  // References to the combined functions
  CapyMathFun** combFuns;

  // Destructor
  void (*destructCapyMathFun)(void);

  // Calculate the combination coefficients that best fit a dataset
  // Input:
  //   dataset: the dataset
  //   iOutput: the index of the output
  // Output:
  //   Calculate the best fit coefficients using linear regression. The
  //   number of input fields in the dataset must match the input dimension
  //   of combined functions. Can be used for dimOut=1 only.
  void (*linearRegression)(
    CapyDataset const* const dataset,
                size_t const iOutput);

  // Calculate the combination coefficients that best fit a matrix
  // Input:
  //   mDataset: the matrix
  // Output:
  //   Calculate the best fit coefficients using linear regression. The
  //   number of column must match the input dimension of the combined
  //   function plus one. The target is the last column.
  void (*linearRegressionFromMat)(CapyMat const* const mDataset);
} CapyLinCombFun;

// Create a CapyLinCombFun
// Input:
//    nbComb: number of linearly combined functions
//    dimIn: input dimension
//    dimOut: output dimension
// Output:
//   Return a CapyLinCombFun.
CapyLinCombFun CapyLinCombFunCreate(
  size_t const nbComb,
  size_t const dimIn,
  size_t const dimOut);

// Allocate memory and create a CapyLinCombFun
// Input:
//    nbComb: number of linearly combined functions
//    dimIn: input dimension
//    dimOut: output dimension
// Output:
//   Return a CapyLinCombFun.
CapyLinCombFun* CapyLinCombFunAlloc(
  size_t const nbComb,
  size_t const dimIn,
  size_t const dimOut);

// Free the memory used by a CapyLinCombFun* and reset '*that' to
// NULL
// Input:
//   that: the CapyLinCombFun to free
void CapyLinCombFunFree(CapyLinCombFun** const that);
#endif
