// ---------------------------- maze.c ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache info@baillehachepascal.dev
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "maze.h"

// Genetrate the maze using the origin shift algorithm
// Input:
//   rng: the pseudo random generator
// Output:
//   Empty the graph if it is not empty, then create a graph equivalent
//   to a 2D maze where rooms are represented with nodes laying on a 2D
//   grid of maze dimensions, and corridors between rooms are represented
//   with links. The maze is guaranteed to have one single path between each
//   pair of rooms. All room are 1x1 in dimension, and all room in the 2D
//   grid exist. The maze contains no loop and no island.
static void GenerateByOriginShift(CapyRandom* const rng) {
  methodOf(CapyMaze2D);

  // Ensure the graph is empty
  $(&(that->graph), destruct)();
  that->graph = CapyGraphCreate();

  // If the grid is empty, nothing to do
  if(that->width * that->height == 0) return;

  // Create the nodes
  loop(x, that->width) loop(y, that->height) {
    CapyGraphNode node = CapyGraphNodeCreate(y * that->width + x);
    $(&(that->graph), addNode)(node);
  }

  // Initialize the links to have a perfect maze
  loop(x, that->width - 1) loop(y, that->height) {
    size_t idA = y * that->width + x;
    size_t idB = y * that->width + x + 1;
    $(&(that->graph), linkNodes)(idA, idB);
  }
  loop(y, that->height - 1) {
    size_t idA = y * that->width + that->width - 1;
    size_t idB = (y + 1) * that->width + that->width - 1;
    $(&(that->graph), linkNodes)(idA, idB);
  }

  // Initialise the 'origin' node
  size_t idOrigin = that->height * that->width - 1;

  // Number of shifting iteration
  size_t nbShift = that->width * that->height * 50;

  // Loop over the shifting iterations
  loop(iShift, nbShift) {

    // Select a random neighbour (linked or not) of the origin node
    size_t idNeighbours[4] = {0};
    size_t pos[2] = {
      idOrigin - (idOrigin / that->width) * that->width,
      idOrigin / that->width
    };
    int8_t nbPos = 0;
    if(pos[0] > 0) {
      idNeighbours[nbPos] = pos[1] * that->width + pos[0] - 1;
      nbPos += 1;
    }
    if(pos[1] > 0) {
      idNeighbours[nbPos] = (pos[1] - 1) * that->width + pos[0];
      nbPos += 1;
    }
    if(pos[0] < that->width - 1) {
      idNeighbours[nbPos] = pos[1] * that->width + pos[0] + 1;
      nbPos += 1;
    }
    if(pos[1] < that->height - 1) {
      idNeighbours[nbPos] = (pos[1] + 1) * that->width + pos[0];
      nbPos += 1;
    }
    CapyRangeInt8 range = {.min = 0, .max = nbPos - 1};
    int rnd = $(rng, getInt8Range)(&range);
    size_t idNeighbour = idNeighbours[rnd];

    // Link the origin node to the selected neighbour
    $(&(that->graph), linkNodes)(idOrigin, idNeighbour);

    // Remove the link originating from the neighbour
    $(&(that->graph), removeLinksFromNode)(idNeighbour);

    // Set the neighbour as the new origin node
    idOrigin = idNeighbour;
  }
}

// Draw the maze
// Input:
//   img: the image on which to draw
//   pen: the pen to draw the maze
//   dim: the dimension of the maze in the image
//   pos: the position of the top-left corner of maze in the image
// Output:
//   Draw the maze on the image by drawing vertical and horizontal lines
//   between unconnected rooms.
static void Draw(
        CapyImg* const img,
  CapyPen const* const pen,
     CapyImgDims const dims,
      CapyImgPos const pos) {
  methodOf(CapyMaze2D);

  // Draw the border
  CapyQuadrilateral quad = CapyQuadrilateralCreate();
  quad.corners[0].x = pos.x;
  quad.corners[0].y = pos.y;
  quad.corners[1].x = pos.x + (CapyImgPos_t)(dims.width);
  quad.corners[1].y = pos.y;
  quad.corners[2].x = pos.x + (CapyImgPos_t)(dims.width);
  quad.corners[2].y = pos.y + (CapyImgPos_t)(dims.height);
  quad.corners[3].x = pos.x;
  quad.corners[3].y = pos.y + (CapyImgPos_t)(dims.height);
  $(pen, drawQuadrilateral)(&quad, img);

  // If the maze is empty, nothing to do
  if(that->width * that->height == 0) return;

  // Scaled dims of a room
  double dimsRoom[2] = {
    (double)(dims.vals[0]) / (double)(that->width),
    (double)(dims.vals[1]) / (double)(that->height),
  };

  // Loop on the rooms
  loop(x, that->width - 1) loop(y, that->height) {

    // If there is no link with the room to the right, draw a wall
    size_t idA = y * that->width + x;
    size_t idB = y * that->width + x + 1;
    CapyGraphLink* const linkB =
      $(&(that->graph), getLinkBetweenNodes)(idA, idB);
    if(linkB == NULL) {
      double from[2] = {
        pos.x + dimsRoom[0] * (double)(x + 1),
        pos.y + dimsRoom[1] * (double)y
      };
      double to[2] = {
        pos.x + dimsRoom[0] * (double)(x + 1),
        pos.y + dimsRoom[1] * (double)(y + 1)
      };
      $(pen, drawLine)(from, to, img);
    }
  }
  loop(x, that->width) loop(y, that->height - 1) {

    // If there is no link with the room to the bottom, draw a wall
    size_t idA = y * that->width + x;
    size_t idB = (y + 1) * that->width + x;
    CapyGraphLink* const linkB =
      $(&(that->graph), getLinkBetweenNodes)(idA, idB);
    if(linkB == NULL) {
      double from[2] = {
        pos.x + dimsRoom[0] * (double)x,
        pos.y + dimsRoom[1] * (double)(y + 1)
      };
      double to[2] = {
        pos.x + dimsRoom[0] * (double)(x + 1),
        pos.y + dimsRoom[1] * (double)(y + 1)
      };
      $(pen, drawLine)(from, to, img);
    }
  }
}

// Free the memory used by a CapyMaze2D
static void Destruct(void) {
  methodOf(CapyMaze2D);
  $(&(that->graph), destruct)();
}

// Create a CapyMaze2D
//   width: the width of the maze
//   height: the height of the maze
// Output:
//   Return a CapyMaze2D
CapyMaze2D CapyMaze2DCreate(
  size_t const width,
  size_t const height) {
  CapyMaze2D that = {
    .graph = CapyGraphCreate(),
    .width = width,
    .height = height,
    .destruct = Destruct,
    .generateByOriginShift = GenerateByOriginShift,
    .draw = Draw,
  };
  return that;
}

// Allocate memory for a new CapyMaze2D and create it
//   width: the width of the maze
//   height: the height of the maze
// Output:
//   Return a CapyMaze2D
// Exception:
//   May raise CapyExc_MallocFailed.
CapyMaze2D* CapyMaze2DAlloc(
  size_t const width,
  size_t const height) {
  CapyMaze2D* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyMaze2DCreate(width, height);
  return that;
}

// Free the memory used by a CapyMaze2D* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyMaze2D to free
void CapyMaze2DFree(CapyMaze2D** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
