// ---------------------------------- maze.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache info@baillehachepascal.dev
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_MAZE_H
#define CAPY_MAZE_H
#include "externalHeaders.h"
#include "cext.h"
#include "graph.h"
#include "random.h"
#include "image.h"
#include "pen.h"

// Description:
// Maze class.

// Maze object
typedef struct CapyMaze2D {

  // Width of the maze
  size_t width;

  // Height of the maze
  size_t height;

  // Graph representing the maze, nodes are rooms and links are connections
  // between rooms. Node id of room at position (x,y) is equal to (x+y*width)
  CapyGraph graph;

  // Destructor
  void (*destruct)(void);

  // Genetrate the maze using the origin shift algorithm
  // Input:
  //   rng: the pseudo random generator
  // Output:
  //   Empty the graph if it is not empty, then create a graph equivalent
  //   to a 2D maze where rooms are represented with nodes laying on a 2D
  //   grid of maze dimensions, and corridors between rooms are represented
  //   with links. The maze is guaranteed to have one single path between each
  //   pair of rooms. All room are 1x1 in dimension, and all room in the 2D
  //   grid exist. The maze contains no loop and no island.
  void (*generateByOriginShift)(CapyRandom* const rng);

  // Draw the maze
  // Input:
  //   img: the image on which to draw
  //   pen: the pen to draw the maze
  //   dim: the dimension of the maze in the image
  //   pos: the position of the top-left corner of maze in the image
  // Output:
  //   Draw the maze on the image by drawing vertical and horizontal lines
  //   between unconnected rooms.
  void (*draw)(
          CapyImg* const img,
    CapyPen const* const pen,
       CapyImgDims const dims,
        CapyImgPos const pos);
} CapyMaze2D;

// Create a CapyMaze2D
// Input:
//   width: the width of the maze
//   height: the height of the maze
// Output:
//   Return a CapyMaze2D
CapyMaze2D CapyMaze2DCreate(
  size_t const width,
  size_t const height);

// Allocate memory for a new CapyMaze2D and create it
//   width: the width of the maze
//   height: the height of the maze
// Output:
//   Return a CapyMaze2D
// Exception:
//   May raise CapyExc_MallocFailed.
CapyMaze2D* CapyMaze2DAlloc(
  size_t const width,
  size_t const height);

// Free the memory used by a CapyMaze2D* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyMaze2D to free
void CapyMaze2DFree(CapyMaze2D** const that);
#endif
