// -------------------------------- memoryPool.h ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_MEMORY_POOL_H
#define CAPY_MEMORY_POOL_H
#include "externalHeaders.h"
#include "cext.h"

// Description:
// Memory pool generic class.

// Generic memory pool class. Declaration macro for a memory pool of a given
// type. The definition of the type must include CapyMemPoolFields (see below).
#define CapyDecMemPool(N, T)                       \
typedef struct N {                                 \
  T* head;                                         \
  T* tail;                                         \
  size_t sizeMax;                                  \
  size_t size;                                     \
  size_t sizeElem;                                 \
  size_t nbUsed;                                   \
  void (*destruct)(void);                          \
  T* (*alloc)(void const* const initElem);         \
  void (*free)(T** const elem);                    \
  void (*flush)(void);                             \
} N;                                               \
N N ## Create(size_t const sizeElem);              \
N* N ## Alloc(size_t const sizeElem);              \
void N ## Destruct(void);                          \
void N ## Free(N** const that);                    \
T* N ## AllocElem(void const* const initElem);     \
void N ## FreeElem(T** const elem);                \
void N ## Flush(void);

// Macro to add necessary fields to a structure to make it usable with
// CapyMemPool
#define CapyMemPoolFields(T)         \
struct {                             \
  T* prev;                           \
  T* next;                           \
  bool isFree;                       \
  CapyPad(bool, MemPoolFieldIsFree); \
} mempool

// Generic memory pool class. Definition macro for a memory pool of a given
// type.

// Create a memory pool
// Output:
//   Return an empty memory pool
#define CapyDefMemPoolCreate(N, T)                  \
N N ## Create(size_t const sizeElem) {              \
  return (N){                                       \
    .head = NULL,                                   \
    .tail = NULL,                                   \
    .sizeMax = 0,                                   \
    .size = 0,                                      \
    .sizeElem = sizeElem,                           \
    .nbUsed = 0,                                    \
    .destruct = N ## Destruct,                      \
    .alloc = N ## AllocElem,                        \
    .free = N ## FreeElem,                          \
    .flush = N ## Flush,                            \
  };                                                \
}

// Allocate a memory pool
// Output:
//   Return an empty memory pool
#define CapyDefMemPoolAlloc(N, T)                   \
N* N ## Alloc(size_t const sizeElem) {              \
  N* that = NULL;                                   \
  safeMalloc(that, 1);                              \
  assert(that != NULL);                             \
  *that = N ## Create(sizeElem);                    \
  return that;                                      \
}

// Flush and destruct a memory pool
// Input:
//   that: the memory pool to destruct
#define CapyDefMemPoolDestruct(N, T)   \
void N ## Destruct(void) {             \
  N* that = (N*)capyThat;              \
  $(that, flush)();                    \
}

// Flush and free a memory pool
// Input:
//   that: the memory pool to free
#define CapyDefMemPoolFree(N, T)                               \
void N ## Free(N** const that) {                               \
  if(that == NULL || *that == NULL) return;                    \
  $(*that, destruct)();                                        \
  free(*that);                                                 \
  *that = NULL;                                                \
}

// Alloc memory using the memory pool
// Input:
//   initElem: if not null the allocated memory is initialized with initElem,
//             must be of size sizeElem bytes
// Output:
//   Return a 'new' element from the pool. If there was a free one, it is
//   reused, else a new element is allocated and added to the pool before
//   returning it.
#define CapyDefMemPoolAllocElem(N, T)                             \
T* N ## AllocElem(void const* const initElem) {                   \
  N* that = (N*)capyThat;                                         \
  T* ptr = that->head;                                            \
  if(ptr && ptr->mempool.isFree) {                                \
    if(ptr->mempool.next) {                                       \
      that->head = ptr->mempool.next;                             \
      that->head->mempool.prev = NULL;                            \
    }                                                             \
    if(initElem) memcpy(ptr, initElem, that->sizeElem);           \
    if(ptr != that->tail) {                                       \
      ptr->mempool.next = NULL;                                   \
      ptr->mempool.prev = that->tail;                             \
      that->tail->mempool.next = ptr;                             \
      that->tail = ptr;                                           \
    }                                                             \
  } else if(that->sizeMax == 0 || that->size < that->sizeMax) {   \
    char* mem = NULL;                                             \
    safeMalloc(mem, that->sizeElem);                              \
    if(mem == NULL) return NULL;                                  \
    if(initElem) memcpy(mem, initElem, that->sizeElem);           \
    else memset(mem, 0, that->sizeElem);                          \
    ptr = (T*)mem;                                                \
    ++(that->size);                                               \
    ptr->mempool.prev = that->tail;                               \
    ptr->mempool.next = NULL;                                     \
    if(that->tail == NULL) that->head = that->tail = ptr;         \
    else {                                                        \
      that->tail->mempool.next = ptr;                             \
      that->tail = ptr;                                           \
    }                                                             \
  } else return NULL;                                             \
  ptr->mempool.isFree = false;                                    \
  ++(that->nbUsed);                                               \
  return ptr;                                                     \
}

// Free an element allocated using the pool
// Input:
//   elem: pointer to the element to free
// Output:
//   The element is marked as free to reuse in the memory pool. The destructor
//   is applied to the element.
#define CapyDefMemPoolFreeElem(N, T)                               \
void N ## FreeElem(T** const elem) {                               \
  if(elem == NULL || *elem == NULL) return;                        \
  N* that = (N*)capyThat;                                          \
  if((*elem)->destruct) $(*elem, destruct)();                      \
  (*elem)->mempool.isFree = true;                                  \
  --(that->nbUsed);                                                \
  if((*elem) != that->head) {                                      \
    (*elem)->mempool.prev->mempool.next = (*elem)->mempool.next;   \
    if((*elem)->mempool.next) {                                    \
      (*elem)->mempool.next->mempool.prev = (*elem)->mempool.prev; \
    }                                                              \
    if(that->tail == (*elem)) that->tail = (*elem)->mempool.prev;  \
    (*elem)->mempool.prev = NULL;                                  \
    (*elem)->mempool.next = that->head;                            \
    that->head->mempool.prev = (*elem);                            \
    that->head = (*elem);                                          \
  }                                                                \
  *elem = NULL;                                                    \
}

// Flush a memory pool
// Output:
//   All the elements in the pool are freed (their memory is released using the
//   standard 'free' function). The destructor is applied to all remaining
//   element in the pool
#define CapyDefMemPoolFlush(N, T)                        \
void N ## Flush(void) {                                  \
  N* that = (N*)capyThat;                                \
  T* ptr = that->head;                                   \
  while(ptr) {                                           \
    T* next = ptr->mempool.next;                         \
    if(ptr->mempool.isFree == false && ptr->destruct)    \
      $(ptr, destruct)();                                \
    free(ptr);                                           \
    ptr = next;                                          \
  }                                                      \
  that->nbUsed = 0;                                      \
  that->size = 0;                                        \
  that->head = that->tail = NULL;                        \
}

// Definition macro for a memory pool of a given type.
#define CapyDefMemPool(N, T)    \
  CapyDefMemPoolCreate(N, T)    \
  CapyDefMemPoolAlloc(N, T)     \
  CapyDefMemPoolDestruct(N, T)  \
  CapyDefMemPoolFree(N, T)      \
  CapyDefMemPoolAllocElem(N, T) \
  CapyDefMemPoolFreeElem(N, T)  \
  CapyDefMemPoolFlush(N, T)
#endif
