// ---------------------------- minimax.c ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "minimax.h"
#include "list.h"

// Definition of the memory pool of MiniMaxWorld
CapyDefMemPool(CapyMiniMaxWorldMemPool, CapyMiniMaxWorld)

// Destruct a MiniMaxWorld
// Output:
//   The world is destruct
static void WorldDestruct(void) {

  // Nothing to do
}

// Clone a MiniMaxWorld
// Input:
//   mempool: the memory pool used to allocate memory for the clone
// Output:
//   Return a clone of 'that'
static CapyMiniMaxWorld* WorldClone(CapyMiniMaxWorldMemPool* const mempool) {
  (void)mempool;
  raiseExc(CapyExc_UndefinedExecution);
  assert(false && "MiniMaxWorld.clone is undefined.");
  return NULL;
}

// Place holder for the createChilds method
static void WorldCreateChilds(CapyMiniMaxWorldMemPool* const mempool) {
  (void)mempool;
  raiseExc(CapyExc_UndefinedExecution);
  assert(false && "MiniMaxWorld.createChilds is undefined.");
}

// Place holder for the isSame method
static bool WorldIsSame(CapyMiniMaxWorld const* const world) {
  (void)world;
  raiseExc(CapyExc_UndefinedExecution);
  assert(false && "MiniMaxWorld.isSame is undefined.");
}

// Create a MiniMaxWorld and return it.
// Inputs:
//   nbActor: the number of actors
// Output:
//   Return the MiniMaxWorld.
CapyMiniMaxWorld CapyMiniMaxWorldCreate(size_t const nbActor) {
  CapyMiniMaxWorld that = {
    .nbActor = nbActor, .child = NULL, .sibling = NULL, .parent = NULL,
    .sente = 0, .depth = 0, .destruct = WorldDestruct, .clone = WorldClone,
    .createChilds = WorldCreateChilds, .isSame = WorldIsSame,
  };
  loop(i, nbActor) that.values[i] = 0.0;
  return that;
}

// Free a MiniMaxWorld
// Inputs:
//   that: the world to free
//   mempool: the memory pool this world is belonging to
// Output:
//   Free the word in the memory pool.
void CapyMiniMaxWorldFree(
        CapyMiniMaxWorld** const that,
  CapyMiniMaxWorldMemPool* const mempool) {
  if(that == NULL || *that == NULL) return;
  if((*that)->flagAvoid) {
    CapyMiniMaxWorldFree(&((*that)->sibling), mempool);
  } else {
    CapyMiniMaxWorldFree(&((*that)->child), mempool);
    if((*that)->sibling != NULL) {
      CapyMiniMaxWorldFree(&((*that)->sibling), mempool);
    }
    $(mempool, free)(that);
  }
}

// Data for the exploration thread
typedef struct ExploreThreadData {
  CapyMiniMax* minimax;
  size_t currentIter;
} ExploreThreadData;

CapyDecList(WorldToExplore, CapyMiniMaxWorld*)
CapyDefList(WorldToExplore, CapyMiniMaxWorld*)

// Get the score of a world from the point of view of one actor
static double GetScoreWorld(
  CapyMiniMaxWorld const* const that,
                   size_t const iActor) {
  double score = that->values[iActor];
  loop(i, that->nbActor) if(i != iActor) score -= that->values[i];
  return score;
}

// Get the best child
static CapyMiniMaxWorld* GetBestChild(CapyMiniMaxWorld* const that) {
  CapyMiniMaxWorld* bestChild = NULL;
  double bestScore = 0.0;
  CapyMiniMaxWorld* child = that->child;
  while(child != NULL) {
    if(bestChild == NULL) {
      bestChild = child;
      bestScore = GetScoreWorld(child, that->sente);
    } else {
      double score = GetScoreWorld(child, that->sente);
      if(bestScore < score) {
        bestChild = child;
        bestScore = score;
      }
    }
    child = child->sibling;
  }
  return bestChild;
}

// Backpropagation of world values
static void BackPropagation(CapyMiniMaxWorld* const that) {

  // Search the best child from the point of view of the sente
  CapyMiniMaxWorld const* bestChild = GetBestChild(that);

  // Set the world value to the value of its best child
  if(bestChild) loop(i, that->nbActor) that->values[i] = bestChild->values[i];

  // If the world has a parent, continue the backpropagation
  if(that->parent) BackPropagation(that->parent);
}

// Check if a world is pruned
static bool IsPrunedWorld(
  CapyMiniMaxWorld const* const world,
                   double const threshold) {
  if(world->parent == NULL) return false;
  double scoreWorld = GetScoreWorld(world, world->parent->sente);
  CapyMiniMaxWorld const* sibling = world->parent->child;
  while(sibling != NULL) {
    if(sibling != world) {
      double scoreSibling = GetScoreWorld(sibling, world->parent->sente);
      if(scoreSibling - scoreWorld > threshold) return true;
    }
    sibling = sibling->sibling;
  }
  return false;
}

// Main function for the exploration thread
static void* ExploreThreadMain(void* ptr) {

  // Cast the argument
  ExploreThreadData* threadData = ptr;
  CapyMiniMax* minimax = threadData->minimax;

  // List of world to explore
  WorldToExplore worlds = WorldToExploreCreate();
  $(&worlds, add)(minimax->currentWorld);

  // Loop until the main thread asks to stop
  while(minimax->flagStop == false) {
    pthread_mutex_lock(&(minimax->exploreMutex));

    // If the current world has changed, reset the list of world to explore
    if(threadData->currentIter != minimax->nbIter) {
      $(&worlds, flush)();
      threadData->currentIter = minimax->nbIter;
      $(&worlds, add)(minimax->currentWorld);
    }

    // Get the next world to explore
    CapyMiniMaxWorld* world = NULL;
    bool isEmpty = $(&worlds, isEmpty)();
    if(isEmpty == false) world = $(&worlds, pop)();
    if(
      world != NULL &&
      minimax->currentWorld->depth + minimax->maxDepthExplore >= world->depth
    ) {

      // If the world is pruned
      if(IsPrunedWorld(world, minimax->pruningThreshold)) {

        // Free the subtree
        CapyMiniMaxWorldFree(&(world->child), &(minimax->mempool));

      // Else the world is not pruned
      } else {

        // Create the childs of the world
        $(world, createChilds)(&(minimax->mempool));

        // Back-propagation
        BackPropagation(world);

        // Append the childs to the list of words to explore
        CapyMiniMaxWorld* child = world->child;
        while(child != NULL) {
          $(&worlds, add)(child);
          child = child->sibling;
        }
      }
    } else {

      // If there is no more world in the list of worlds to explore,
      // restart the exploration from the current world
      $(&worlds, add)(minimax->currentWorld);
    }
    pthread_mutex_unlock(&(minimax->exploreMutex));
  }

  // Free memory
  $(&worlds, destruct)();
  free(threadData);

  // Exit the thread.
  pthread_exit(NULL);
}

// Start the exploration
static void Start(void) {
  methodOf(CapyMiniMax);

  // If the thread is already running, nothing to do
  if(that->exploreThread != 0) return;

  // Prepare the data
  ExploreThreadData* threadData = NULL;
  safeMalloc(threadData, 1);
  if(threadData == NULL) return;
  threadData->minimax = that;
  threadData->currentIter = that->nbIter;

  // Start the exploration thread
  that->flagStop = false;
  int ret =
    pthread_create(&(that->exploreThread), NULL, ExploreThreadMain, threadData);
  if(ret != 0) {
    raiseExc(CapyExc_ForkFailed);
    return;
  }
}

// Stop the exploration
static void Stop(void) {
  methodOf(CapyMiniMax);
  if(that->exploreThread != 0) {
    that->flagStop = true;
    pthread_join(that->exploreThread, NULL);
  }
  that->exploreThread = 0;
}

// Get the next best world
// Output:
//   Return the next best world.
static CapyMiniMaxWorld* GetNextBest(void) {
  methodOf(CapyMiniMax);
  return GetBestChild(that->currentWorld);
}

// Search a world in the history tree
// Input:
//   world: the searched world
// Output:
//   Return the first found world which is identical (According to
//   CapyMiniMaxWorld.isSame) to the world in argument, or NULL.
static CapyMiniMaxWorld* SearchWorld(
             CapyMiniMax* const that,
        CapyMiniMaxWorld* const world,
  CapyMiniMaxWorld const* const searchedWorld) {
  CapyMiniMaxWorld* foundWorld = NULL;
  bool isSame = $(world, isSame)(searchedWorld);
  if(isSame) foundWorld = world;
  else {
    CapyMiniMaxWorld* child = world->child;
    while(child != NULL && foundWorld == NULL) {
      foundWorld = SearchWorld(that, child, searchedWorld);
      child = child->sibling;
    }
  }
  return foundWorld;
}

// Set the new current world.
// Input:
//   world: the new current world
// Output:
//   Free the current history except for the world in argument and its subtree
//   if it's in the current history, and set the world in argument as the new
//   root of the history.
static void SetCurrentWorld(CapyMiniMaxWorld* const world) {
  methodOf(CapyMiniMax);
  pthread_mutex_lock(&(that->exploreMutex));
  CapyMiniMaxWorld* foundWorld = NULL;
  if(that->currentWorld != NULL) {
    foundWorld = SearchWorld(that, that->currentWorld, world);
  }
  if(foundWorld == NULL) {
    CapyMiniMaxWorldFree(&(that->currentWorld), &(that->mempool));
    foundWorld = $(world, clone)(&(that->mempool));
  } else {
    foundWorld->flagAvoid = true;
    CapyMiniMaxWorldFree(&(that->currentWorld), &(that->mempool));
    foundWorld->flagAvoid = false;
  }
  that->currentWorld = foundWorld;
  that->currentWorld->parent = NULL;
  ++(that->nbIter);
  pthread_mutex_unlock(&(that->exploreMutex));
}

// Free the memory used by a CapyMiniMax
static void Destruct(void) {
  methodOf(CapyMiniMax);
  $(that, stop)();
  CapyMiniMaxWorldFree(&(that->currentWorld), &(that->mempool));
  $(&(that->mempool), destruct)();
  pthread_mutex_destroy(&(that->exploreMutex));
}

// Create a CapyMiniMax
// Input:
//   sizeWorld: size in byte of the world structure
// Output:
//   Return a CapyMiniMax
CapyMiniMax CapyMiniMaxCreate(size_t const sizeWorld) {
  CapyMiniMax that = {
    .currentWorld = NULL,
    .maxDepthExplore = 1,
    .pruningThreshold = 1.0,
    .mempool = CapyMiniMaxWorldMemPoolCreate(sizeWorld),
    .destruct = Destruct,
    .start = Start,
    .stop = Stop,
    .getNextBest = GetNextBest,
    .setCurrentWorld = SetCurrentWorld,
  };
  pthread_mutex_init(&(that.exploreMutex), NULL);
  return that;
}

// Allocate memory for a new CapyMiniMax and create it
// Input:
//   sizeWorld: size in byte of the world structure
// Output:
//   Return a CapyMiniMax
// Exception:
//   May raise CapyExc_MallocFailed.
CapyMiniMax* CapyMiniMaxAlloc(size_t const sizeWorld) {
  CapyMiniMax* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyMiniMaxCreate(sizeWorld);
  return that;
}

// Free the memory used by a CapyMiniMax* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyMiniMax to free
void CapyMiniMaxFree(CapyMiniMax** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
