// ---------------------------------- minimax.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_MINIMAX_H
#define CAPY_MINIMAX_H
#include "externalHeaders.h"
#include "cext.h"
#include "memoryPool.h"

// Description:
// MiniMax class.

// Maximum number of actors in a MiniMaxWorld
#define CAPY_MINIMAX_NB_MAX_ACTOR 4

// Macro to define the MiniMaxWorld structure
#define CapyMiniMaxWorldDef { \
  size_t nbActor; \
  size_t sente; \
  uint64_t depth; \
  CapyMiniMaxWorld* parent; \
  CapyMiniMaxWorld* child; \
  CapyMiniMaxWorld* sibling; \
  bool flagAvoid; \
  CapyPad(bool, flagAvoid); \
  double values[CAPY_MINIMAX_NB_MAX_ACTOR]; \
  CapyMemPoolFields(CapyMiniMaxWorld); \
  void (*destruct)(void); \
  CapyMiniMaxWorld* (*clone)(CapyMiniMaxWorldMemPool* const mempool); \
  void (*createChilds)(CapyMiniMaxWorldMemPool* const mempool); \
  bool (*isSame)(CapyMiniMaxWorld const* const world); \
}

// Predeclaration of the parent structure for MiniMax worlds
typedef struct CapyMiniMaxWorld CapyMiniMaxWorld;

// Predeclaration of the memory pool of MiniMaxWorld
typedef struct CapyMiniMaxWorldMemPool CapyMiniMaxWorldMemPool;

// Definition of the parent structure for MiniMax worlds
struct CapyMiniMaxWorld CapyMiniMaxWorldDef;

// Declaration of a memory pool of MiniMaxWorld
CapyDecMemPool(CapyMiniMaxWorldMemPool, CapyMiniMaxWorld)

// Create a MiniMaxWorld and return it.
// Inputs:
//   nbActor: the number of actors
// Output:
//   Return the MiniMaxWorld.
CapyMiniMaxWorld CapyMiniMaxWorldCreate(size_t const nbActor);

// Free a MiniMaxWorld
// Inputs:
//   that: the world to free
//   mempool: the memory pool this world is belonging to
// Output:
//   Free the word in the memory pool.
void CapyMiniMaxWorldFree(
        CapyMiniMaxWorld** const that,
  CapyMiniMaxWorldMemPool* const mempool);

// MiniMax object
typedef struct CapyMiniMax {

  // Exploration thread.
  pthread_t exploreThread;

  // Mutex to control the exploration thread.
  pthread_mutex_t exploreMutex;

  // Flag to stop the exploration thread
  bool flagStop;
  CapyPad(bool, flagStop);

  // Max depth exploration (default: 1)
  uint64_t maxDepthExplore;

  // Pruning threshold (default: 1.0)
  double pruningThreshold;

  // Current state
  CapyMiniMaxWorld* currentWorld;

  // Memory pool for the worlds
  CapyMiniMaxWorldMemPool mempool;

  // Counter for the number of iteration, incremented each time setCurrentWorld
  // is called
  size_t nbIter;

  // Destructor
  void (*destruct)(void);

  // Start the exploration
  void (*start)(void);

  // Stop the exploration
  void (*stop)(void);

  // Get the next best world
  // Output:
  //   Return a clone of the next best world.
  CapyMiniMaxWorld* (*getNextBest)(void);

  // Set the new current world.
  // Input:
  //   world: the new current world
  // Output:
  //   Free the current history except for the world in argument and its subtree
  //   if it's in the current history, and set the world in argument as the new
  //   root of the history.
  void (*setCurrentWorld)(CapyMiniMaxWorld* const world);
} CapyMiniMax;

// Create a CapyMiniMax
// Input:
//   sizeWorld: size in byte of the world structure
// Output:
//   Return a CapyMiniMax
CapyMiniMax CapyMiniMaxCreate(size_t const sizeWorld);

// Allocate memory for a new CapyMiniMax and create it
// Input:
//   sizeWorld: size in byte of the world structure
// Output:
//   Return a CapyMiniMax
// Exception:
//   May raise CapyExc_MallocFailed.
CapyMiniMax* CapyMiniMaxAlloc(size_t const sizeWorld);

// Free the memory used by a CapyMiniMax* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyMiniMax to free
void CapyMiniMaxFree(CapyMiniMax** const that);
#endif
