// ---------------------------- mirrorballcamera.c ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache info@baillehachepascal.dev
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "mirrorballcamera.h"

// Calculate the coordinates in the mirror ball image for that direction
// vector
// Input:
//   dir: the view direction (normalised, {0,0,1} is toward the center of the
//        mirror ball image
// Output:
//   Return the position in the mirror ball image for the given direction.
//   May return invalid position for invisible direction.
static CapyImgPos FromViewDirectionToImgPos(double const* const dir) {
  methodOf(CapyMirrorBallCamera);
  CapyImgPos pos = {0};
  double const c = -1.0 / (sqrt(2.0 * (1.0 - dir[2])) * sin(that->alpha));
  double const w = (double)(that->img->dims.vals[0] / 2);
  pos.coords[0] = (CapyImgPos_t)(w * (1.0 - c * dir[0]));
  pos.coords[1] = (CapyImgPos_t)(w * (1.0 + c * dir[1]));
  return pos;
}

// Render the view from the mirror ball given the current camera.
// Input:
//   dims: dimensions of the result image
// Output:
//  Return the rendered image.
static CapyImg* Render(CapyImgDims const dims) {
  methodOf(CapyMirrorBallCamera);

  // Allocate memory for the result image
  CapyImg* img = CapyImgAlloc(capyImgMode_rgb, dims);

  // Get the origin position in the result image (set to the center)
  CapyImgPos_t posOrig[2] = {
    (CapyImgPos_t)(dims.width) / 2,
    (CapyImgPos_t)(dims.height) / 2
  };

  // Loop on the pixels of the image
  forEach(pixel, img->iter) {

    // Get the direction vector from the camera for the pixel
    double dir[3];
    $(that->camera, fromImgCoord)(pixel.pos.coords, posOrig, dir);

    // Calculate the coordinates in the mirror ball image for that direction
    // vector
    CapyImgPos posMirrorBall = $(that, fromViewDirectionToImgPos)(dir);

    // If we are not looking behind the mirror ball
    bool const isValidPos = $(that->img, isValidCoord)(&posMirrorBall);
    if(isValidPos) {

      // Set the pixel in the result image to the corresponding pixel in the
      // mirror ball image
      CapyColorData const* color = $(that->img, getColor)(&posMirrorBall);
      $(img, setColor)(&(pixel.pos), color);
    }
  }

  // Return the result image
  return img;
}

// Free the memory used by a CapyMirrorBallCamera
static void Destruct(void) {
  methodOf(CapyMirrorBallCamera);
  CapyImgFree(&(that->img));
  CapyCameraFree(&(that->camera));
}

// Create a CapyMirrorBallCamera
// Input:
//   img: the image of the mirror ball
// Output:
//   Return a CapyMirrorBallCamera
CapyMirrorBallCamera CapyMirrorBallCameraCreate(CapyImg* const img) {
  CapyMirrorBallCamera that = {
    .img = img,
    .camera = CapyCameraAlloc(),
    .alpha = M_PI_2,
    .destruct = Destruct,
    .fromViewDirectionToImgPos = FromViewDirectionToImgPos,
    .render = Render,
  };
  if(that.img->dims.width != that.img->dims.height) {
    raiseExc(CapyExc_InvalidParameters);
  }
  return that;
}

// Allocate memory for a new CapyMirrorBallCamera and create it
// Input:
//   img: the image of the mirror ball
// Output:
//   Return a CapyMirrorBallCamera
// Exception:
//   May raise CapyExc_MallocFailed.
CapyMirrorBallCamera* CapyMirrorBallCameraAlloc(CapyImg* const img) {
  CapyMirrorBallCamera* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyMirrorBallCameraCreate(img);
  return that;
}

// Free the memory used by a CapyMirrorBallCamera* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyMirrorBallCamera to free
void CapyMirrorBallCameraFree(CapyMirrorBallCamera** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
