// ---------------------------------- mirrorballcamera.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache info@baillehachepascal.dev
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_MIRRORBALLCAMERA_H
#define CAPY_MIRRORBALLCAMERA_H
#include "externalHeaders.h"
#include "cext.h"
#include "image.h"
#include "camera.h"

// Description:
// MirrorBallCamera class.

// MirrorBallCamera object
typedef struct CapyMirrorBallCamera {

  // Image of the mirror ball (must be square and the mirror ball must fill
  // the image)
  CapyImg* img;

  // Camera to control the view (front={0,0,1} is considered to correspond to
  // looking toward the center of the mirror ball image)
  CapyCamera* camera;

  // Distortion correction factor (default: pi/2, in ]0, pi/2])
  double alpha;

  // Destructor
  void (*destruct)(void);

  // Calculate the coordinates in the mirror ball image for that direction
  // vector
  // Input:
  //   dir: the view direction (normalised, {0,0,1} is toward the center of the
  //        mirror ball image
  // Output:
  //   Return the position in the mirror ball image for the given direction.
  //   May return invalid position for invisible direction.
  CapyImgPos (*fromViewDirectionToImgPos)(double const* const dir);

  // Render the view from the mirror ball given the current camera.
  // Input:
  //   dims: dimensions of the result image
  // Output:
  //  Return the rendered image.
  CapyImg* (*render)(CapyImgDims const dims);
} CapyMirrorBallCamera;

// Create a CapyMirrorBallCamera
// Input:
//   img: the image of the mirror ball
// Output:
//   Return a CapyMirrorBallCamera
CapyMirrorBallCamera CapyMirrorBallCameraCreate(CapyImg* const img);

// Allocate memory for a new CapyMirrorBallCamera and create it
// Input:
//   img: the image of the mirror ball
// Output:
//   Return a CapyMirrorBallCamera
// Exception:
//   May raise CapyExc_MallocFailed.
CapyMirrorBallCamera* CapyMirrorBallCameraAlloc(CapyImg* const img);

// Free the memory used by a CapyMirrorBallCamera* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyMirrorBallCamera to free
void CapyMirrorBallCameraFree(CapyMirrorBallCamera** const that);
#endif
