// -------------------------- neuralNetwork.c --------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "neuralNetwork.h"

// Copy a CapyNNModel
// Input:
//   that: the model to copy
// Output:
//   Return a copy.
CapyNNModel CapyNNModelCopy(CapyNNModel const* const that) {
  CapyNNModel model = {0};
  model.nbLayer = that->nbLayer;
  safeMalloc(model.layers, that->nbLayer);
  loop(i, that->nbLayer) model.layers[i] = that->layers[i];
  return model;
}

// Destruct a CapyNNModel
// Input:
//   that: the model to destruct
// Output:
//   The model is destructed.
void CapyNNModelDestruct(CapyNNModel* const that) {
  free(that->layers);
}

// Update one node of the neural network
static void EvalNode(
  CapyNeuralNetwork* const nn,
        CapyNNLayer* const layer,
         CapyNNNode* const node) {

  // Reset the node with its bias
  node->valInput = nn->params[node->idxBias];

  // Loop on the links
  loop(iLink, node->nbLink) {
    CapyNNLink* link = node->links + iLink;

    // Calculate the input node value (support for recurrent network)
    double inputNodeVal = 0.0;
    if(nn->layers[link->iLayer].depth < layer->depth) {
      inputNodeVal = nn->layers[link->iLayer].nodes[link->iNode].val;
    } else {
      inputNodeVal = nn->layers[link->iLayer].nodes[link->iNode].prevVal;
    }

    // Add the weighted input for this link
    node->valInput += nn->params[link->idxWeight] * inputNodeVal;
  }

  // Apply the activation function
  if(layer->activation) {
    $(layer->activation, eval)(&(node->valInput), &(node->val));
  } else node->val = node->valInput;
}

// Update one layer of the neural network
static void EvalLayer(
  CapyNeuralNetwork* const nn,
        CapyNNLayer* const layer) {

  // Loop on the nodes and update the previous value
  loop(iNode, layer->nbNode) {
    layer->nodes[iNode].prevVal = layer->nodes[iNode].val;
  }

  // Loop on the nodes and eval them
  loop(iNode, layer->nbNode) {
    EvalNode(nn, layer, layer->nodes + iNode);
  }
}

// Evaluate the neural network
// Input:
//   in: the values of the first layer
//   out: result values (nbOutput array)
// Output:
//   'out' is updated with the value of the last layer after evaluation.
static void Eval(
  double const* const in,
        double* const out) {
  methodOf(CapyNeuralNetwork);

  // Copy the input in the first layer
  loop(i, that->nbInput) {
    that->layers[0].nodes[i].val = in[i];
    that->layers[0].nodes[i].valInput = in[i];
  }

  // Loop on the layers, skipping the first one (which is the input)
  loop(iLayer, that->nbLayer) if(iLayer > 0) {
    EvalLayer(that, that->layers + iLayer);
  }

  // Copy the last layer in the output
  loop(i, that->nbOutput) {
    out[i] = that->layers[that->nbLayer - 1].nodes[i].val;
  }

  // If necessary applies softmax on output
  if(that->flagSoftmaxOutput) {
    double const temperature = 1.0;
    CapyVec u = {.dim = that->nbOutput, .vals = out};
    CapyVecSoftmax(&u, temperature);
  }
}

//  Save the neural network to a stream
//  Input:
//    stream: the stream on which to save
//  Output:
//    The neural network is saved on the stream
static void Save(FILE* const stream) {
  methodOf(CapyNeuralNetwork);
  safeFWrite(stream, sizeof(that->nbInput), &(that->nbInput));
  safeFWrite(stream, sizeof(that->nbOutput), &(that->nbOutput));
  safeFWrite(stream, sizeof(that->nbLayer), &(that->nbLayer));
  loop(i, that->nbLayer) {
    safeFWrite(
      stream, sizeof(that->layers[i].nbNode), &(that->layers[i].nbNode));
    safeFWrite(stream, sizeof(that->layers[i].depth), &(that->layers[i].depth));
    if(that->layers[i].activation == NULL) {
      CapyNNActivationFunType type = capyNNActivationFun_none;
      safeFWrite(stream, sizeof(CapyNNActivationFunType), &type);
    } else {
      safeFWrite(
        stream, sizeof(that->layers[i].activation->type),
        &(that->layers[i].activation->type));
    }
    loop(j, that->layers[i].nbNode) {
      CapyNNNode const* const node = that->layers[i].nodes + j;
      safeFWrite(stream, sizeof(node->idxBias), &(node->idxBias));
      safeFWrite(stream, sizeof(node->nbLink), &(node->nbLink));
      loop(k, node->nbLink) {
        safeFWrite(
          stream, sizeof(node->links[k].iLayer), &(node->links[k].iLayer));
        safeFWrite(
          stream, sizeof(node->links[k].iNode), &(node->links[k].iNode));
        safeFWrite(
          stream, sizeof(node->links[k].idxWeight),
          &(node->links[k].idxWeight));
      }
    }
  }
  safeFWrite(stream, sizeof(that->nbParam), &(that->nbParam));
  loop(i, that->nbParam) {
    safeFWrite(stream, sizeof(that->params[i]), that->params + i);
  }
}

// Evaluate the gradient of a neural network relative to one of its output
// Input:
//   in: the values of the first layer
//   out: result values (nbParam array)
//   iOutput: output used to calculate the gradient
// Output:
//   'out' is updated with the value of the gradient of 'params' relative to
//    'iOutput'.
static void EvalGradient(
  double const* const in,
        double* const out,
         size_t const iOutput) {
  methodOf(CapyNeuralNetwork);

  // Dummy buffer to memorise the result of evaluation
  double* dummy = NULL;
  safeMalloc(dummy, that->nbOutput);
  assert(dummy);

  // Loop on the parameters
  loop(iParam, that->nbParam) {

    // Memorise the original parameter value
    double const origParam = that->params[iParam];

    // Calculate the gradient value for the parameter
    double const epsilon = 1e-3;
    that->params[iParam] += epsilon;
    $(that, eval)(in, dummy);
    out[iParam] = dummy[iOutput];
    that->params[iParam] -= 2.0 * epsilon;
    $(that, eval)(in, dummy);
    out[iParam] -= dummy[iOutput];
    out[iParam] /= 2.0 * epsilon;

    // Reset the original parameter value
    that->params[iParam] = origParam;
  }

  // Free memory
  free(dummy);
}

// Free the memory used by a CapyNeuralNetwork
static void Destruct(void) {
  methodOf(CapyNeuralNetwork);
  loop(iLayer, that->nbLayer) {
    loop(iNode, that->layers[iLayer].nbNode) {
      if(iLayer > 0) free(that->layers[iLayer].nodes[iNode].links);
    }
    free(that->layers[iLayer].nodes);
  }
  free(that->layers);
  free(that->params);
}

// Create a CapyNeuralNetwork
// Input:
//    nbInput: size of the input vector
//      model: the layers model
//   nbOutput: size of the output vector
// Output:
//   Return a CapyNeuralNetwork
CapyNeuralNetwork CapyNeuralNetworkCreateFullyConnected(
              size_t const nbInput,
  CapyNNModel const* const model,
              size_t const nbOutput) {
  CapyNeuralNetwork that = {
    .nbInput = nbInput,
    .nbOutput = nbOutput,
    .nbLayer = model->nbLayer + 2,
    .nbParam = 0,
    .params = NULL,
    .destruct = Destruct,
    .eval = Eval,
    .evalGradient = EvalGradient,
    .save = Save,
  };

  // Allocate memory for the layer
  safeMalloc(that.layers, that.nbLayer);

  // Loop on the layers
  loop(iLayer, that.nbLayer) {

    // If its the input layer
    if(iLayer == 0) {

      // There are as many node as inputs
      that.layers[iLayer].nbNode = nbInput;

      // There is no activation function
      that.layers[iLayer].activation = NULL;

    // Else if its the output layer
    } else if(iLayer == that.nbLayer - 1) {

      // There are as many node as outputs
      that.layers[iLayer].nbNode = nbOutput;

      // There is no activation function
      that.layers[iLayer].activation = NULL;

    // Else it's a hidden layer
    } else {

      // Set the number of nodes and activation function
      that.layers[iLayer].nbNode = model->layers[iLayer - 1].nbNode;
      that.layers[iLayer].activation = model->layers[iLayer - 1].activation;
    }

    // Set the depth of the layer
    that.layers[iLayer].depth = iLayer;

    // Allocate memory for the nodes of the layer
    safeMalloc(that.layers[iLayer].nodes, that.layers[iLayer].nbNode);

    // Loop on the nodes of the layer
    loop(iNode, that.layers[iLayer].nbNode) {

      // Initialise the node values
      that.layers[iLayer].nodes[iNode].val = 0.0;
      that.layers[iLayer].nodes[iNode].valInput = 0.0;
      that.layers[iLayer].nodes[iNode].prevVal = 0.0;
      that.layers[iLayer].nodes[iNode].nbLink = 0;
      that.layers[iLayer].nodes[iNode].idxBias = that.nbParam;
      (that.nbParam)++;

      // If it's not the input layer
      if(iLayer > 0) {

        // Set the number of links to the node as the number of node in the
        // previous layer
        that.layers[iLayer].nodes[iNode].nbLink =
          that.layers[iLayer - 1].nbNode;

        // Allocate memory for the links of the node
        safeMalloc(
          that.layers[iLayer].nodes[iNode].links,
          that.layers[iLayer].nodes[iNode].nbLink);

        // Loop on the link of the node
        loop(iLink, that.layers[iLayer].nodes[iNode].nbLink) {

          // Link to the node in the previous layer
          that.layers[iLayer].nodes[iNode].links[iLink].iLayer = iLayer - 1;
          that.layers[iLayer].nodes[iNode].links[iLink].iNode = iLink;
          that.layers[iLayer].nodes[iNode].links[iLink].idxWeight =
            that.nbParam;
          (that.nbParam)++;
        }
      }
    }
  }

  // Allocate memory for the params and derivatives
  safeMalloc(that.params, that.nbParam);
  loop(i, that.nbParam) that.params[i] = 0.0;

  // Return the CapyNeuralNetwork
  return that;
}

// Allocate memory for a new CapyNeuralNetwork and create it
// Input:
//    nbInput: size of the input vector
//      model: the layers model
//   nbOutput: size of the output vector
// Output:
//   Return a CapyNeuralNetwork
// Exception:
//   May raise CapyExc_MallocFailed.
CapyNeuralNetwork* CapyNeuralNetworkAllocFullyConnected(
              size_t const nbInput,
  CapyNNModel const* const model,
              size_t const nbOutput) {
  CapyNeuralNetwork* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyNeuralNetworkCreateFullyConnected(nbInput, model, nbOutput);
  return that;
}

// Free the memory used by a CapyNeuralNetwork* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyNeuralNetwork to free
void CapyNeuralNetworkFree(CapyNeuralNetwork** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Linear Function
static void Linear(
  double const* const in,
        double* const out) {
  out[0] = in[0];
}

// Destruct a CapyNNActivationLinear
static void ActivationLinearDestruct(void) {
  methodOf(CapyNNActivationLinear);
  $(that, destructCapyMathFun)();
}

// Linear derivative function
static void LinearDerivative(
  double* const in,
   size_t const iDim,
  double* const out) {
  (void)iDim; (void)in;
  out[0] = 1.0;
}

// Export the linear activation function as a C function
// Input:
//   stream: the stream where to export
//   name: the name of the function
// Output:
//   A ready to use C function implementing the activation function is written
//   on the stream. See the comment exported with the function to know how to
//   use the exported function.
static void ActivationLinearExportToCFun(
        FILE* const stream,
  char const* const name) {
  fprintf(
    stream,
    "// Linear activation function\n");
  fprintf(
    stream,
    "double %s(double const x) {\n",
    name);
  fprintf(
    stream,
    "  return x;\n");
  fprintf(
    stream,
    "}\n");
  fprintf(
    stream,
    "\n");
}

// Export the linear activation function as a Javascript function
// Input:
//   stream: the stream where to export
//   name: the name of the function
// Output:
//   A ready to use Javascript function implementing the activation function is
//   written on the stream. See the comment exported with the function to know
//   how to use the exported function.
static void ActivationLinearExportToJavascript(
        FILE* const stream,
  char const* const name) {
  fprintf(
    stream,
    "// Linear activation function\n");
  fprintf(
    stream,
    "function %s(x) {\n",
    name);
  fprintf(
    stream,
    "  return x;\n");
  fprintf(
    stream,
    "}\n");
}

// Create a step activation function
// Output:
//   Return a CapyMathFun implementing a step function
CapyNNActivationLinear CapyNNActivationLinearCreate(void) {
  CapyNNActivationLinear that;
  CapyInherits(that, CapyMathFun, (1, 1));
  that.type = capyNNActivationFun_linear;
  that.destruct = ActivationLinearDestruct;
  that.eval = Linear;
  that.evalDerivative = LinearDerivative;
  that.exportToCFun = ActivationLinearExportToCFun;
  that.exportToJavascript = ActivationLinearExportToJavascript;
  return that;
}

// Allocate memory and create a step activation function
// Output:
//   Return a CapyMathFun implementing a step function
CapyNNActivationLinear* CapyNNActivationLinearAlloc(void) {
  CapyNNActivationLinear* that;
  safeMalloc(that, 1);
  *that = CapyNNActivationLinearCreate();
  assert(that != NULL);
  return that;
}

// Free the memory used by a 
// Output:
//   Return a CapyMathFun implementing a step function
void CapyNNActivationLinearFree(CapyNNActivationLinear** const that) {
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Step Function
static void Step(
  double const* const in,
        double* const out) {
  if(in[0] >= 0.0) out[0] = 1.0;
  else out[0] = 0.0;
}

// Destruct a CapyNNActivationStep
static void ActivationStepDestruct(void) {
  methodOf(CapyNNActivationStep);
  $(that, destructCapyMathFun)();
}

// Step derivative function
static void StepDerivative(
  double* const in,
   size_t const iDim,
  double* const out) {
  (void)iDim; (void)in;
  out[0] = 0.0;
}

// Export the step activation function as a C function
// Input:
//   stream: the stream where to export
//   name: the name of the function
// Output:
//   A ready to use C function implementing the activation function is written
//   on the stream. See the comment exported with the function to know how to
//   use the exported function.
static void ActivationStepExportToCFun(
        FILE* const stream,
  char const* const name) {
  fprintf(
    stream,
    "// Step activation function\n");
  fprintf(
    stream,
    "double %s(double const x) {\n",
    name);
  fprintf(
    stream,
    "  if(x >= 0.0) return 1.0; else return 0.0;\n");
  fprintf(
    stream,
    "}\n");
  fprintf(
    stream,
    "\n");
}

// Export the step activation function as a Javascript function
// Input:
//   stream: the stream where to export
//   name: the name of the function
// Output:
//   A ready to use Javascript function implementing the activation function is
//   written on the stream. See the comment exported with the function to know
//   how to use the exported function.
static void ActivationStepExportToJavascript(
        FILE* const stream,
  char const* const name) {
  fprintf(
    stream,
    "// Step activation function\n");
  fprintf(
    stream,
    "function %s(x) {\n",
    name);
  fprintf(
    stream,
    "  if(x >= 0.0) return 1.0; else return 0.0;\n");
  fprintf(
    stream,
    "}\n");
}

// Create a step activation function
// Output:
//   Return a CapyMathFun implementing a step function
CapyNNActivationStep CapyNNActivationStepCreate(void) {
  CapyNNActivationStep that;
  CapyInherits(that, CapyMathFun, (1, 1));
  that.type = capyNNActivationFun_step;
  that.destruct = ActivationStepDestruct;
  that.eval = Step;
  that.evalDerivative = StepDerivative;
  that.exportToCFun = ActivationStepExportToCFun;
  that.exportToJavascript = ActivationStepExportToJavascript;
  return that;
}

// Allocate memory and create a step activation function
// Output:
//   Return a CapyMathFun implementing a step function
CapyNNActivationStep* CapyNNActivationStepAlloc(void) {
  CapyNNActivationStep* that;
  safeMalloc(that, 1);
  *that = CapyNNActivationStepCreate();
  assert(that != NULL);
  return that;
}

// Free the memory used by a 
// Output:
//   Return a CapyMathFun implementing a step function
void CapyNNActivationStepFree(CapyNNActivationStep** const that) {
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Sigmoid Function
static void Sigmoid(
  double const* const in,
        double* const out) {
  out[0] = 1.0 / (1.0 + exp(-in[0]));
}

// Destruct a CapyNNActivationSigmoid
static void ActivationSigmoidDestruct(void) {
  methodOf(CapyNNActivationSigmoid);
  $(that, destructCapyMathFun)();
}

// Sigmoid derivative function
static void SigmoidDerivative(
  double* const in,
   size_t const iDim,
  double* const out) {
  (void)iDim;
  double x = 1.0 / (1.0 + exp(-in[0]));
  out[0] = x * (1.0 - x);
}

// Export the sigmoid activation function as a C function
// Input:
//   stream: the stream where to export
//   name: the name of the function
// Output:
//   A ready to use C function implementing the activation function is written
//   on the stream. See the comment exported with the function to know how to
//   use the exported function.
static void ActivationSigmoidExportToCFun(
        FILE* const stream,
  char const* const name) {
  fprintf(
    stream,
    "// Sigmoid activation function\n");
  fprintf(
    stream,
    "double %s(double const x) {\n",
    name);
  fprintf(
    stream,
    "  return 1.0 / (1.0 + exp(-x));\n");
  fprintf(
    stream,
    "}\n");
  fprintf(
    stream,
    "\n");
}

// Export the sigmoid activation function as a Javascript function
// Input:
//   stream: the stream where to export
//   name: the name of the function
// Output:
//   A ready to use Javascript function implementing the activation function is
//   written on the stream. See the comment exported with the function to know
//   how to use the exported function.
static void ActivationSigmoidExportToJavascript(
        FILE* const stream,
  char const* const name) {
  fprintf(
    stream,
    "// Sigmoid activation function\n");
  fprintf(
    stream,
    "function %s(x) {\n",
    name);
  fprintf(
    stream,
    "  return 1.0 / (1.0 + Math.exp(-x));\n");
  fprintf(
    stream,
    "}\n");
}

// Create a step activation function
// Output:
//   Return a CapyMathFun implementing a step function
CapyNNActivationSigmoid CapyNNActivationSigmoidCreate(void) {
  CapyNNActivationSigmoid that;
  CapyInherits(that, CapyMathFun, (1, 1));
  that.type = capyNNActivationFun_sigmoid;
  that.destruct = ActivationSigmoidDestruct;
  that.eval = Sigmoid;
  that.evalDerivative = SigmoidDerivative;
  that.exportToCFun = ActivationSigmoidExportToCFun;
  that.exportToJavascript = ActivationSigmoidExportToJavascript;
  return that;
}

// Allocate memory and create a step activation function
// Output:
//   Return a CapyMathFun implementing a step function
CapyNNActivationSigmoid* CapyNNActivationSigmoidAlloc(void) {
  CapyNNActivationSigmoid* that;
  safeMalloc(that, 1);
  *that = CapyNNActivationSigmoidCreate();
  assert(that != NULL);
  return that;
}

// Free the memory used by a 
// Output:
//   Return a CapyMathFun implementing a step function
void CapyNNActivationSigmoidFree(CapyNNActivationSigmoid** const that) {
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Hyperbolic tangent Function
static void HyperTangent(
  double const* const in,
        double* const out) {
  double x = exp(in[0]);
  double y = exp(-in[0]);
  out[0] = (x - y) / (x + y);
}

// Destruct a CapyNNActivationHyperTangent
static void ActivationHyperTangentDestruct(void) {
  methodOf(CapyNNActivationHyperTangent);
  $(that, destructCapyMathFun)();
}

// HyperTangent derivative function
static void HyperTangentDerivative(
  double* const in,
   size_t const iDim,
  double* const out) {
  (void)iDim;
  double x = exp(in[0]);
  double y = exp(-in[0]);
  out[0] = (x - y) / (x + y);
  out[0] = 1.0 - out[0] * out[0];
}

// Export the hypertangent activation function as a C function
// Input:
//   stream: the stream where to export
//   name: the name of the function
// Output:
//   A ready to use C function implementing the activation function is written
//   on the stream. See the comment exported with the function to know how to
//   use the exported function.
static void ActivationHyperTangentExportToCFun(
        FILE* const stream,
  char const* const name) {
  fprintf(
    stream,
    "// HyperTangent activation function\n");
  fprintf(
    stream,
    "double %s(double const x) {\n",
    name);
  fprintf(
    stream,
    "  double y = exp(x * 2.0);\n");
  fprintf(
    stream,
    "  return (y - 1.0) / (y + 1.0);\n");
  fprintf(
    stream,
    "}\n");
  fprintf(
    stream,
    "\n");
}

// Export the hypertangent activation function as a Javascript function
// Input:
//   stream: the stream where to export
//   name: the name of the function
// Output:
//   A ready to use Javascript function implementing the activation function is
//   written on the stream. See the comment exported with the function to know
//   how to use the exported function.
static void ActivationHyperTangentExportToJavascript(
        FILE* const stream,
  char const* const name) {
  fprintf(
    stream,
    "// Hypertangent activation function\n");
  fprintf(
    stream,
    "function %s(x) {\n",
    name);
  fprintf(
    stream,
    "  let y = Math.exp(x * 2.0);\n");
  fprintf(
    stream,
    "  return (y - 1.0) / (y + 1.0);\n");
  fprintf(
    stream,
    "}\n");
}

// Create a hypertangent activation function
// Output:
//   Return a CapyMathFun implementing a step function
CapyNNActivationHyperTangent CapyNNActivationHyperTangentCreate(void) {
  CapyNNActivationHyperTangent that;
  CapyInherits(that, CapyMathFun, (1, 1));
  that.type = capyNNActivationFun_hypertangent;
  that.destruct = ActivationHyperTangentDestruct;
  that.eval = HyperTangent;
  that.evalDerivative = HyperTangentDerivative;
  that.exportToCFun = ActivationHyperTangentExportToCFun;
  that.exportToJavascript = ActivationHyperTangentExportToJavascript;
  return that;
}

// Allocate memory and create a step activation function
// Output:
//   Return a CapyMathFun implementing a step function
CapyNNActivationHyperTangent* CapyNNActivationHyperTangentAlloc(void) {
  CapyNNActivationHyperTangent* that;
  safeMalloc(that, 1);
  *that = CapyNNActivationHyperTangentCreate();
  assert(that != NULL);
  return that;
}

// Free the memory used by a 
// Output:
//   Return a CapyMathFun implementing a step function
void CapyNNActivationHyperTangentFree(
  CapyNNActivationHyperTangent** const that) {
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// ReLU Function
static void ReLU(
  double const* const in,
        double* const out) {
  if(in[0] >= 0.0) out[0] = in[0]; else out[0] = 0.0;
}

// Destruct a CapyNNActivationReLU
static void ActivationReLUDestruct(void) {
  methodOf(CapyNNActivationReLU);
  $(that, destructCapyMathFun)();
}

// ReLU derivative function
static void ReLUDerivative(
  double* const in,
   size_t const iDim,
  double* const out) {
  (void)iDim;
  if(in[0] > 0.0) out[0] = 1.0; else out[0] = 0.0;
}

// Export the relu activation function as a C function
// Input:
//   stream: the stream where to export
//   name: the name of the function
// Output:
//   A ready to use C function implementing the activation function is written
//   on the stream. See the comment exported with the function to know how to
//   use the exported function.
static void ActivationReLUExportToCFun(
        FILE* const stream,
  char const* const name) {
  fprintf(
    stream,
    "// ReLU activation function\n");
  fprintf(
    stream,
    "double %s(double const x) {\n",
    name);
  fprintf(
    stream,
    "  if(x >= 0.0) return x; else return 0.0;\n");
  fprintf(
    stream,
    "}\n");
  fprintf(
    stream,
    "\n");
}

// Export the ReLU activation function as a Javascript function
// Input:
//   stream: the stream where to export
//   name: the name of the function
// Output:
//   A ready to use Javascript function implementing the activation function is
//   written on the stream. See the comment exported with the function to know
//   how to use the exported function.
static void ActivationReLUExportToJavascript(
        FILE* const stream,
  char const* const name) {
  fprintf(
    stream,
    "// ReLU activation function\n");
  fprintf(
    stream,
    "function %s(x) {\n",
    name);
  fprintf(
    stream,
    "  if(x >= 0.0) return x; else return 0.0;\n");
  fprintf(
    stream,
    "}\n");
}

// Create a step activation function
// Output:
//   Return a CapyMathFun implementing a step function
CapyNNActivationReLU CapyNNActivationReLUCreate(void) {
  CapyNNActivationReLU that;
  CapyInherits(that, CapyMathFun, (1, 1));
  that.type = capyNNActivationFun_relu;
  that.destruct = ActivationReLUDestruct;
  that.eval = ReLU;
  that.evalDerivative = ReLUDerivative;
  that.exportToCFun = ActivationReLUExportToCFun;
  that.exportToJavascript = ActivationReLUExportToJavascript;
  return that;
}

// Allocate memory and create a step activation function
// Output:
//   Return a CapyMathFun implementing a step function
CapyNNActivationReLU* CapyNNActivationReLUAlloc(void) {
  CapyNNActivationReLU* that;
  safeMalloc(that, 1);
  *that = CapyNNActivationReLUCreate();
  assert(that != NULL);
  return that;
}

// Free the memory used by a 
// Output:
//   Return a CapyMathFun implementing a step function
void CapyNNActivationReLUFree(CapyNNActivationReLU** const that) {
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// SiLU function
static void SiLU(
  double const* const in,
        double* const out) {
  out[0] = in[0] / (1.0 + exp(-in[0]));
}

// SiLU derivative function
static void SiLUDerivative(
  double* const in,
   size_t const iDim,
  double* const out) {
  (void)iDim;
  double x = exp(-in[0]);
  double y = 1.0 + x;
  out[0] = (y + in[0] * x) / (y * y);
}

// Destruct a CapyNNActivationSiLU
static void ActivationSiLUDestruct(void) {
  methodOf(CapyNNActivationSiLU);
  $(that, destructCapyMathFun)();
}

// Export the silu activation function as a C function
// Input:
//   stream: the stream where to export
//   name: the name of the function
// Output:
//   A ready to use C function implementing the activation function is written
//   on the stream. See the comment exported with the function to know how to
//   use the exported function.
static void ActivationSiLUExportToCFun(
        FILE* const stream,
  char const* const name) {
  fprintf(
    stream,
    "// SiLU activation function\n");
  fprintf(
    stream,
    "double %s(double const x) {\n",
    name);
  fprintf(
    stream,
    "  return x / (1.0 + exp(-x));\n");
  fprintf(
    stream,
    "}\n");
  fprintf(
    stream,
    "\n");
}

// Export the SiLU activation function as a Javascript function
// Input:
//   stream: the stream where to export
//   name: the name of the function
// Output:
//   A ready to use Javascript function implementing the activation function is
//   written on the stream. See the comment exported with the function to know
//   how to use the exported function.
static void ActivationSiLUExportToJavascript(
        FILE* const stream,
  char const* const name) {
  fprintf(
    stream,
    "// SiLU activation function\n");
  fprintf(
    stream,
    "function %s(x) {\n",
    name);
  fprintf(
    stream,
    "  return x / (1.0 + Math.exp(-x));\n");
  fprintf(
    stream,
    "}\n");
}

// Create a step activation function
// Output:
//   Return a CapyMathFun implementing a step function
CapyNNActivationSiLU CapyNNActivationSiLUCreate(void) {
  CapyNNActivationSiLU that;
  CapyInherits(that, CapyMathFun, (1, 1));
  that.type = capyNNActivationFun_silu;
  that.destruct = ActivationSiLUDestruct;
  that.eval = SiLU;
  that.evalDerivative = SiLUDerivative;
  that.exportToCFun = ActivationSiLUExportToCFun;
  that.exportToJavascript = ActivationSiLUExportToJavascript;
  return that;
}

// Allocate memory and create a step activation function
// Output:
//   Return a CapyMathFun implementing a step function
CapyNNActivationSiLU* CapyNNActivationSiLUAlloc(void) {
  CapyNNActivationSiLU* that;
  safeMalloc(that, 1);
  *that = CapyNNActivationSiLUCreate();
  assert(that != NULL);
  return that;
}

// Free the memory used by a 
// Output:
//   Return a CapyMathFun implementing a step function
void CapyNNActivationSiLUFree(CapyNNActivationSiLU** const that) {
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Predefined activation functions
CapyNNActivationFun capyNNActivationFuns[capyNNActivationFun_nb] = {
  [capyNNActivationFun_none] = {.dimIn = 0, .dimOut = 0},
  [capyNNActivationFun_linear] = {
    .dimIn = 1,
    .dimOut = 1,
    .type = capyNNActivationFun_linear,
    .destruct = ActivationLinearDestruct,
    .eval = Linear,
    .evalDerivative = LinearDerivative,
    .exportToCFun = ActivationLinearExportToCFun,
    .exportToJavascript = ActivationLinearExportToJavascript,
  },
  [capyNNActivationFun_step] = {
    .dimIn = 1,
    .dimOut = 1,
    .type = capyNNActivationFun_step,
    .destruct = ActivationStepDestruct,
    .eval = Step,
    .evalDerivative = StepDerivative,
    .exportToCFun = ActivationStepExportToCFun,
    .exportToJavascript = ActivationStepExportToJavascript,
  },
  [capyNNActivationFun_sigmoid] = {
    .dimIn = 1,
    .dimOut = 1,
    .type = capyNNActivationFun_sigmoid,
    .destruct = ActivationSigmoidDestruct,
    .eval = Sigmoid,
    .evalDerivative = SigmoidDerivative,
    .exportToCFun = ActivationSigmoidExportToCFun,
    .exportToJavascript = ActivationSigmoidExportToJavascript,
  },
  [capyNNActivationFun_hypertangent] = {
    .dimIn = 1,
    .dimOut = 1,
    .type = capyNNActivationFun_hypertangent,
    .destruct = ActivationHyperTangentDestruct,
    .eval = HyperTangent,
    .evalDerivative = HyperTangentDerivative,
    .exportToCFun = ActivationHyperTangentExportToCFun,
    .exportToJavascript = ActivationHyperTangentExportToJavascript,
  },
  [capyNNActivationFun_relu] = {
    .dimIn = 1,
    .dimOut = 1,
    .type = capyNNActivationFun_relu,
    .destruct = ActivationReLUDestruct,
    .eval = ReLU,
    .evalDerivative = ReLUDerivative,
    .exportToCFun = ActivationReLUExportToCFun,
    .exportToJavascript = ActivationReLUExportToJavascript,
  },
  [capyNNActivationFun_silu] = {
    .dimIn = 1,
    .dimOut = 1,
    .type = capyNNActivationFun_silu,
    .destruct = ActivationSiLUDestruct,
    .eval = SiLU,
    .evalDerivative = SiLUDerivative,
    .exportToCFun = ActivationSiLUExportToCFun,
    .exportToJavascript = ActivationSiLUExportToJavascript,
  },
};

// Load a CapyNeuralNetwork from a stream
// Input:
//   stream: the stream from which the NN is loaded
//   model: CapyNNModel updated with the model of the loaded NN
// Output:
//   Return a CapyNeuralNetwork
CapyNeuralNetwork* CapyNeuralNetworkLoad(
         FILE* const stream,
  CapyNNModel* const model) {
  CapyNeuralNetwork* that = NULL;
  safeMalloc(that, 1);
  if(that == NULL) return NULL;
  that->destruct = Destruct;
  that->eval = Eval;
  that->save = Save;
  safeFRead(stream, sizeof(that->nbInput), &(that->nbInput));
  safeFRead(stream, sizeof(that->nbOutput), &(that->nbOutput));
  safeFRead(stream, sizeof(that->nbLayer), &(that->nbLayer));
  safeMalloc(that->layers, that->nbLayer);
  model->nbLayer = that->nbLayer - 2;
  safeMalloc(model->layers, that->nbLayer);
  loop(i, that->nbLayer) {
    safeFRead(
      stream, sizeof(that->layers[i].nbNode), &(that->layers[i].nbNode));
    if(i > 0 && i < that->nbLayer - 1) {
      model->layers[i - 1].nbNode = that->layers[i].nbNode;
    }
    safeFRead(stream, sizeof(that->layers[i].depth), &(that->layers[i].depth));
    CapyNNActivationFunType type;
    safeFRead(stream, sizeof(CapyNNActivationFunType), &type);
    if(type == capyNNActivationFun_none) {
      that->layers[i].activation = NULL;
    } else {
      that->layers[i].activation = capyNNActivationFuns + type;
    }
    if(i > 0 && i < that->nbLayer - 1) {
      model->layers[i - 1].activation = that->layers[i].activation;
    }
    safeMalloc(that->layers[i].nodes, that->layers[i].nbNode);
    loop(j, that->layers[i].nbNode) {
      CapyNNNode* const node = that->layers[i].nodes + j;
      safeFRead(stream, sizeof(node->idxBias), &(node->idxBias));
      safeFRead(stream, sizeof(node->nbLink), &(node->nbLink));
      safeMalloc(node->links, node->nbLink);
      loop(k, node->nbLink) {
        safeFRead(
          stream, sizeof(node->links[k].iLayer), &(node->links[k].iLayer));
        safeFRead(
          stream, sizeof(node->links[k].iNode), &(node->links[k].iNode));
        safeFRead(
          stream, sizeof(node->links[k].idxWeight),
          &(node->links[k].idxWeight));
      }
    }
  }
  safeFRead(stream, sizeof(that->nbParam), &(that->nbParam));
  safeMalloc(that->params, that->nbParam);
  loop(i, that->nbParam) {
    safeFRead(stream, sizeof(that->params[i]), that->params + i);
  }
  return that;
}
