// ------------------------------ neuralNetwork.h -----------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_NEURALNETWORK_H
#define CAPY_NEURALNETWORK_H
#include "externalHeaders.h"
#include "cext.h"
#include "mathfun.h"

// Description:
// Class implementing a neural network.

// NNLink structure
typedef struct CapyNNLink {

  // Index of the input layer
  size_t iLayer;

  // Index of the input node in the layer
  size_t iNode;

  // Index of the weight of the link in the array of params
  size_t idxWeight;
} CapyNNLink;

// NNNode structure
typedef struct CapyNNNode {

  // Value of the node before applying the activation function (i.e. w.x+b)
  double valInput;

  // Value of the node after applying the activation function (i.e. A(w.x+b))
  double val;

  // Previous value of the node
  double prevVal;

  // Variable to memorise and reuse the propagating derivation during training
  double propagatingDeriv;

  // Number of input links
  size_t nbLink;

  // Input links
  CapyNNLink* links;

  // Index of the bias of the node in the array of params
  size_t idxBias;
} CapyNNNode;

// Types of activation function
typedef enum CapyNNActivationFunType {
  capyNNActivationFun_none,
  capyNNActivationFun_linear,
  capyNNActivationFun_step,
  capyNNActivationFun_sigmoid,
  capyNNActivationFun_hypertangent,
  capyNNActivationFun_relu,
  capyNNActivationFun_silu,
  capyNNActivationFun_nb,
} CapyNNActivationFunType;

// Activation function class definition macro
#define CapyNNActivationFunDef struct {   \
  struct CapyMathFunDef;                  \
  CapyNNActivationFunType type;           \
  CapyPad(CapyNNActivationFunType, type); \
  void (*destructCapyMathFun)(void);      \
  void (*exportToCFun)(                   \
          FILE* const stream,             \
    char const* const name);              \
  void (*exportToJavascript)(             \
          FILE* const stream,             \
    char const* const name);              \
}

// Activation function class
typedef CapyNNActivationFunDef CapyNNActivationFun;

// Macro of the definition of one layer
// Number of nodes
// size_t nbNode;
// Reference to the activation function for the nodes of this layer
// CapyMathFun* activation;
#define CapyNNLayerDef_              \
  struct {                           \
    size_t nbNode;                   \
    CapyNNActivationFun* activation; \
  }

// NNLayer structure
typedef struct CapyNNLayer {

  // Layer definition
  CapyNNLayerDef_;

  // Depth of the layer
  size_t depth;

  // Nodes
  CapyNNNode* nodes;
} CapyNNLayer;

// Definition of one layer
typedef CapyNNLayerDef_ CapyNNLayerDef;

// Neural network model
typedef struct CapyNNModel {

  // Number of layers
  size_t nbLayer;

  // Number of node and activation function per layer
  CapyNNLayerDef* layers;
} CapyNNModel;

// Copy a CapyNNModel
// Input:
//   that: the model to copy
// Output:
//   Return a copy.
CapyNNModel CapyNNModelCopy(CapyNNModel const* const that);

// Destruct a CapyNNModel
// Input:
//   that: the model to destruct
// Output:
//   The model is destructed.
void CapyNNModelDestruct(CapyNNModel* const that);

// NeuralNetwork object
typedef struct CapyNeuralNetwork {

  // Number of input
  size_t nbInput;

  // Number of output
  size_t nbOutput;

  // Number of layers
  size_t nbLayer;

  // Layers (including input and output layers)
  CapyNNLayer* layers;

  // Total number of params (weights and biases) in the model
  size_t nbParam;

  // Array of all weights and biases, ordered by layer and node
  double* params;

  // Flag to automatically apply sigmoid function on output of evaluation
  // (default: false)
  bool flagSoftmaxOutput;
  CapyPad(bool, flagSoftmaxOutput);

  // Destructor
  void (*destruct)(void);

  // Evaluate the neural network
  // Input:
  //   in: the values of the first layer
  //   out: result values (nbOutput array)
  // Output:
  //   'out' is updated with the value of the last layer after evaluation.
  void (*eval)(
    double const* const in,
          double* const out);

  //  Save the neural network to a stream
  //  Input:
  //    stream: the stream on which to save
  //  Output:
  //    The neural network is saved on the stream
  void (*save)(FILE* const stream);

  // Evaluate the gradient of a neural network relative to one of its output
  // Input:
  //   in: the values of the first layer
  //   out: result values (nbParam array)
  //   iOutput: output used to calculate the gradient
  // Output:
  //   'out' is updated with the value of the gradient of 'params' relative to
  //    'iOutput'.
  void (*evalGradient)(
    double const* const in,
          double* const out,
           size_t const iOutput);
} CapyNeuralNetwork;

// Create a CapyNeuralNetwork
// Input:
//    nbInput: size of the input vector
//      model: the layers model
//   nbOutput: size of the output vector
// Output:
//   Return a CapyNeuralNetwork
CapyNeuralNetwork CapyNeuralNetworkCreateFullyConnected(
              size_t const nbInput,
  CapyNNModel const* const model,
              size_t const nbOutput);

// Allocate memory for a new CapyNeuralNetwork and create it
// Input:
//    nbInput: size of the input vector
//      model: the layers model
//   nbOutput: size of the output vector
// Output:
//   Return a CapyNeuralNetwork
// Exception:
//   May raise CapyExc_MallocFailed.
CapyNeuralNetwork* CapyNeuralNetworkAllocFullyConnected(
              size_t const nbInput,
  CapyNNModel const* const model,
              size_t const nbOutput);

// Free the memory used by a CapyNeuralNetwork* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyNeuralNetwork to free
void CapyNeuralNetworkFree(CapyNeuralNetwork** const that);

// Load a CapyNeuralNetwork from a stream
// Input:
//   stream: the stream from which the NN is loaded
//   model: CapyNNModel updated with the model of the loaded NN
// Output:
//   Return a CapyNeuralNetwork
CapyNeuralNetwork* CapyNeuralNetworkLoad(
         FILE* const stream,
  CapyNNModel* const model);

// Linear activation function class
typedef struct CapyNNActivationLinear {
  CapyNNActivationFunDef;
} CapyNNActivationLinear;

// Create a step activation function
// Output:
//   Return a CapyMathFun implementing a step function
CapyNNActivationLinear CapyNNActivationLinearCreate(void);

// Allocate memory and create a step activation function
// Output:
//   Return a CapyMathFun implementing a step function
CapyNNActivationLinear* CapyNNActivationLinearAlloc(void);

// Free the memory used by a 
// Output:
//   Return a CapyMathFun implementing a step function
void CapyNNActivationLinearFree(CapyNNActivationLinear** const that);

// Step activation function class
typedef struct CapyNNActivationStep {
  CapyNNActivationFunDef;
} CapyNNActivationStep;

// Create a step activation function
// Output:
//   Return a CapyMathFun implementing a step function
CapyNNActivationStep CapyNNActivationStepCreate(void);

// Allocate memory and create a step activation function
// Output:
//   Return a CapyMathFun implementing a step function
CapyNNActivationStep* CapyNNActivationStepAlloc(void);

// Free the memory used by a 
// Output:
//   Return a CapyMathFun implementing a step function
void CapyNNActivationStepFree(CapyNNActivationStep** const that);

// Sigmoid activation function class
typedef struct CapyNNActivationSigmoid {
  CapyNNActivationFunDef;
} CapyNNActivationSigmoid;

// Create a step activation function
// Output:
//   Return a CapyMathFun implementing a step function
CapyNNActivationSigmoid CapyNNActivationSigmoidCreate(void);

// Allocate memory and create a step activation function
// Output:
//   Return a CapyMathFun implementing a step function
CapyNNActivationSigmoid* CapyNNActivationSigmoidAlloc(void);

// Free the memory used by a 
// Output:
//   Return a CapyMathFun implementing a step function
void CapyNNActivationSigmoidFree(CapyNNActivationSigmoid** const that);

// HyperTangent activation function class
typedef struct CapyNNActivationHyperTangent {
  CapyNNActivationFunDef;
} CapyNNActivationHyperTangent;

// Create a step activation function
// Output:
//   Return a CapyMathFun implementing a step function
CapyNNActivationHyperTangent CapyNNActivationHyperTangentCreate(void);

// Allocate memory and create a step activation function
// Output:
//   Return a CapyMathFun implementing a step function
CapyNNActivationHyperTangent* CapyNNActivationHyperTangentAlloc(void);

// Free the memory used by a 
// Output:
//   Return a CapyMathFun implementing a step function
void CapyNNActivationHyperTangentFree(
  CapyNNActivationHyperTangent** const that);

// ReLU activation function class
typedef struct CapyNNActivationReLU {
  CapyNNActivationFunDef;
} CapyNNActivationReLU;

// Create a step activation function
// Output:
//   Return a CapyMathFun implementing a step function
CapyNNActivationReLU CapyNNActivationReLUCreate(void);

// Allocate memory and create a step activation function
// Output:
//   Return a CapyMathFun implementing a step function
CapyNNActivationReLU* CapyNNActivationReLUAlloc(void);

// Free the memory used by a 
// Output:
//   Return a CapyMathFun implementing a step function
void CapyNNActivationReLUFree(CapyNNActivationReLU** const that);

// SiLU activation function class
typedef struct CapyNNActivationSiLU {
  CapyNNActivationFunDef;
} CapyNNActivationSiLU;

// Create a step activation function
// Output:
//   Return a CapyMathFun implementing a step function
CapyNNActivationSiLU CapyNNActivationSiLUCreate(void);

// Allocate memory and create a step activation function
// Output:
//   Return a CapyMathFun implementing a step function
CapyNNActivationSiLU* CapyNNActivationSiLUAlloc(void);

// Free the memory used by a 
// Output:
//   Return a CapyMathFun implementing a step function
void CapyNNActivationSiLUFree(CapyNNActivationSiLU** const that);
#endif
