// ------------------------------ nnPredictor.c ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "nnPredictor.h"
#include "chrono.h"
#include "random.h"
#include "slidingAverage.h"
#include "distribution.h"

// Loss function class for the gradient descent method
typedef struct LossFun {

  // Inherits CapyMathFun
  struct CapyMathFunDef;

  // Reference to the neural network
  CapyNeuralNetwork* nn;

  // Reference to the dataset
  CapyDataset const* dataset;

  // Reference to the dataset converted to a matrix
  CapyMat const* mat;

  // Total loss value
  double loss;

  // Average over the batch of the loss for each output nodes
  double* avgLossNodes;

  // Input of the loss function for each output nodes
  double* lossInputs;

  // Target values of the loss function for each output nodes
  double* lossTargets;

  // Array of all derivative of the cost wrt the weights and biases, ordered
  // by layer and node as params
  double* derivatives;

  // Array of all average of derivative of the cost wrt the weights and
  // biases, ordered by layer and node as params
  double* avgDerivatives;

  // Index of the first row in the current batch
  size_t iFirstRowBatch;

  // Batch size
  size_t batchSize;

  // Type of predictor
  CapyPredictorType type;
  CapyPad(CapyPredictorType, type);

  // Type of loss function (default: capyNNPredictorLossType_mse)
  CapyNNPredictorLossType lossType;
  CapyPad(CapyNNPredictorLossType, lossType);

  // Threshold for Huber loss (default: 1.0)
  double huberThreshold;

  // Variable to store the result of evaluation during training
  double* nnOut;

  // Destructor
  void (*destructCapyMathFun)(void);

  // Update the all the derivative values prior to executing the gradient
  // descent step
  // Input:
  //   in: the current parameters
  // Output:
  //   that.derivatives is updated
  void (*updateDerivatives)(double const* const in);
} LossFun;

static void LossFunDestruct(void) {
  methodOf(LossFun);
  $(that, destructCapyMathFun)();
  free(that->avgLossNodes);
  free(that->lossInputs);
  free(that->lossTargets);
  free(that->derivatives);
  free(that->avgDerivatives);
  free(that->nnOut);
}

static void EvalSample(
  LossFun* const that,
    size_t const iSample,
   double* const nnOut) {
  double const* row = that->mat->vals + iSample * that->mat->nbCol;
  $(that->nn, eval)(row, nnOut);
  double softmax[that->nn->nbOutput];
  if(that->lossType == capyNNPredictorLossType_cross_entropy) {
    CapyVec u = {.dim = that->nn->nbOutput, .vals = nnOut};
    CapyVec v = {.dim = that->nn->nbOutput, .vals = softmax};
    CapyVecCopy(&u, &v);
    CapyVecSoftmax(&v, 1.0);
  }
  loop(iOutput, that->nn->nbOutput) {
    double tgt = 0.0;
    if(that->type == capyPredictorType_categorical) {
      tgt = fabs(row[that->mat->nbCol - 1] - (double)iOutput) < .1 ? 1.0 : 0.0;
    } else {
      tgt = row[that->nn->nbInput + iOutput];
    }
    if(that->lossType == capyNNPredictorLossType_mse) {

      // For MSE
      // Loss=(nodeVal - expectedVal)^2
      double deltaTgt = nnOut[iOutput] - tgt;
      that->lossInputs[iOutput] = deltaTgt;
      that->lossTargets[iOutput] = tgt;
      that->avgLossNodes[iOutput] += deltaTgt * deltaTgt;
    } else if(that->lossType == capyNNPredictorLossType_mae) {

      // For MAE
      // Loss=|nodeVal - expectedVal|
      double deltaTgt = nnOut[iOutput] - tgt;
      that->lossInputs[iOutput] = deltaTgt;
      that->lossTargets[iOutput] = tgt;
      that->avgLossNodes[iOutput] += fabs(deltaTgt);
    } else if(that->lossType == capyNNPredictorLossType_huber) {

      // For Huber
      // If |nodeVal - expectedVal| > huberThreshold
      // Loss=huberThreshold*(|nodeVal - expectedVal| - 0.5 * huberThreshold)
      // else
      // Loss=0.5*(nodeVal - expectedVal)^2
      double deltaTgt = nnOut[iOutput] - tgt;
      that->lossInputs[iOutput] = deltaTgt;
      that->lossTargets[iOutput] = tgt;
      if(fabs(deltaTgt) > that->huberThreshold) {
        that->avgLossNodes[iOutput] +=
          that->huberThreshold * (fabs(deltaTgt) - 0.5 * that->huberThreshold);
      } else {
        that->avgLossNodes[iOutput] += 0.5 * deltaTgt * deltaTgt;
      }
    } else if(that->lossType == capyNNPredictorLossType_cross_entropy) {

      // For cross entropy
      // Loss=-sum_i(log(softmax(nodeVal_i))*expectedVal_i)
      that->lossInputs[iOutput] = softmax[iOutput];
      that->lossTargets[iOutput] = tgt;
      that->avgLossNodes[iOutput] +=
        -(log((nnOut[iOutput] < 1e-6 ? 1e-6 : nnOut[iOutput])) * tgt);
    } else raiseExc(CapyExc_UndefinedExecution);
  }
}

// The eval method is actually never used because we use only the derivative
// of the loss function and they are calculated not by differential method using
// eval but by backpropagation using UpdateDerivatives
static void Eval(
  double const* const in,
        double* const out) {
  methodOf(LossFun);

  // Copy the input in the neural network parameters
  memcpy(that->nn->params, in, sizeof(double) * that->nn->nbParam);

  // Reset the average loss per output node
  loop(iOutput, that->nn->nbOutput) that->avgLossNodes[iOutput] = 0.0;

  // Loop on the samples in the dataset
  loop(iSample, that->batchSize) {

    // Evaluate the output of the neural network
    size_t iSampleMod =
      (that->iFirstRowBatch + iSample) % that->mat->nbRow;
    EvalSample(that, iSampleMod, that->nnOut);
  }

  // Average the loss
  that->loss = 0.0;
  loop(iOutput, that->nn->nbOutput) {
    that->avgLossNodes[iOutput] /= (double)(that->batchSize);
    that->loss += that->avgLossNodes[iOutput];
  }

  // Update the result
  out[0] = that->loss;
}

static void UpdateDerivativeOutputLayer(
            LossFun* const that,
  CapyNNLayer const* const layer) {

  // Loop on the nodes of the output layer
  loop(iNode, layer->nbNode) {
    CapyNNNode* node = layer->nodes + iNode;

    // Update derivative of loss function
    // for MSE:
    // dLoss/dBias = 2 * (nodeVal - expectedVal)
    // for MAE:
    // dLoss/dBias = -1 if (nodeVal - expectedVal) < 0, else 1
    // for Huber:
    // if |nodeVal - expectedVal| > huberThreshold
    // dLoss/dBias = -huberThreshold if (nodeVal - expectedVal) < 0,
    //   else huberThreshold
    // else
    // dLoss/dBias = (nodeVal - expectedVal)
    // For cross entropy
    // dLoss/dBias = softmax(nodeVal) - expectedVal
    // If there was an activation function to the output nodes their
    // derivative should be added here (cf the code for the hidden layer)
    if(that->lossType == capyNNPredictorLossType_mse) {
      that->derivatives[node->idxBias] = 2.0 * that->lossInputs[iNode];
    } else if(that->lossType == capyNNPredictorLossType_mae) {
      if(that->lossInputs[iNode] < 0.0) {
        that->derivatives[node->idxBias] = -1.0;
      } else {
        that->derivatives[node->idxBias] = 1.0;
      }
    } else if(that->lossType == capyNNPredictorLossType_huber) {
      if(fabs(that->lossInputs[iNode]) > that->huberThreshold) {
        if(that->lossInputs[iNode] < 0.0) {
          that->derivatives[node->idxBias] = -that->huberThreshold;
        } else {
          that->derivatives[node->idxBias] = that->huberThreshold;
        }
      } else {
        that->derivatives[node->idxBias] = that->lossInputs[iNode];
      }
    } else if(that->lossType == capyNNPredictorLossType_cross_entropy) {
      that->derivatives[node->idxBias] =
        that->lossInputs[iNode] - that->lossTargets[iNode];
    } else raiseExc(CapyExc_UndefinedExecution);

    // Loop on the input links of that node
    loop(iLink, node->nbLink) {
      CapyNNLink* link = node->links + iLink;

      // Calculate the input node value (support for recurrent network)
      double inputNodeVal = 0.0;
      CapyNNLayer* inputLayer = that->nn->layers + link->iLayer;
      CapyNNNode* inputNode = inputLayer->nodes + link->iNode;
      if(link->iLayer < layer->depth) inputNodeVal = inputNode->val;
      else inputNodeVal = inputNode->prevVal;

      // Update dLoss/dWeight = dLoss/dBias * inputNodeVal
      that->derivatives[link->idxWeight] =
        that->derivatives[node->idxBias] * inputNodeVal;

      // Update the propagating derivative for the input node
      inputNode->propagatingDeriv +=
        that->derivatives[node->idxBias] * that->nn->params[link->idxWeight];
    }
  }
}

static void UpdateDerivativeHiddenLayer(
            LossFun* const that,
  CapyNNLayer const* const layer) {

  // Loop on the nodes of the layer
  loop(iNode, layer->nbNode) {
    CapyNNNode* node = layer->nodes + iNode;

    // Update dLoss/dBias = propagatingDerivative *
    // derivativeActivationFunction(nodeInputValue) * 1
    double derivActivation = 1.0;
    if(layer->activation != NULL) {
      $(layer->activation, evalDerivative)(
        &(node->valInput), 0, &derivActivation);
    }
    that->derivatives[node->idxBias] = node->propagatingDeriv * derivActivation;

    // Loop on the input links of that node
    loop(iLink, node->nbLink) {
      CapyNNLink* link = node->links + iLink;

      // Calculate the input node value (support for recurrent network)
      double inputNodeVal = 0.0;
      CapyNNLayer* inputLayer = that->nn->layers + link->iLayer;
      CapyNNNode* inputNode = inputLayer->nodes + link->iNode;
      if(link->iLayer < layer->depth) inputNodeVal = inputNode->val;
      else inputNodeVal = inputNode->prevVal;

      // Update dLoss/dWeight = propagatingDerivative *
      // derivativeActivationFunction(nodeInputValue) * inputNodeVal
      that->derivatives[link->idxWeight] =
        that->derivatives[node->idxBias] * inputNodeVal;

      // Update the propagating derivative for the input node
      inputNode->propagatingDeriv +=
        that->derivatives[node->idxBias] * that->nn->params[link->idxWeight];
    }
  }
}

static void UpdateDerivatives(double const* const in) {
  methodOf(LossFun);

  // Copy the input in the neural network parameters
  memcpy(that->nn->params, in, sizeof(double) * that->nn->nbParam);

  // Reset the average loss per output node
  loop(iOutput, that->nn->nbOutput) that->avgLossNodes[iOutput] = 0.0;

  // Reset the average of derivatives
  loop(i, that->nn->nbParam) that->avgDerivatives[i] = 0.0;

  // Loop on the samples in the dataset
  loop(iBatchSample, that->batchSize) {

    // Reset the loss input per output nodes
    loop(iOutput, that->nn->nbOutput) that->lossInputs[iOutput] = 0.0;

    // Reset the derivatives for this sample
    loop(i, that->nn->nbParam) that->derivatives[i] = 0.0;

    // Reset the node's propagating derivative
    loop(iLayer, that->nn->nbLayer) {
      CapyNNLayer* layer = that->nn->layers + iLayer;
      loop(iNode, layer->nbNode) {
        CapyNNNode* node = layer->nodes + iNode;
        node->propagatingDeriv = 0.0;
      }
    }

    // Evaluate the output of the neural network and update the average loss
    // per output
    size_t iSample =
      (that->iFirstRowBatch + iBatchSample) % that->mat->nbRow;
    EvalSample(that, iSample, that->nnOut);

    // Loop on the layers backward (this allow to commonalise portion of
    // the derivative calculations, cf node.propagatingDeriv). jskip the first
    // layers because it the input layer so there is nothing to do there
    loop(jLayer, that->nn->nbLayer - 1) {
      size_t iLayer = that->nn->nbLayer - jLayer - 1;
      CapyNNLayer* layer = that->nn->layers + iLayer;

      // Update the derivatives of the nodes of the layer
      if(jLayer == 0) UpdateDerivativeOutputLayer(that, layer);
      else UpdateDerivativeHiddenLayer(that, layer);
    }

    // Update the average of derivatives
    loop(i, that->nn->nbParam) that->avgDerivatives[i] += that->derivatives[i];
  }

  // Average the loss per output node and calculate the overall loss
  that->loss = 0.0;
  loop(iOutput, that->nn->nbOutput) {
    that->avgLossNodes[iOutput] /= (double)(that->batchSize);
    that->loss += that->avgLossNodes[iOutput];
  }

  // Calculate the average of the derivatives
  loop(i, that->nn->nbParam) {
    that->avgDerivatives[i] /= (double)(that->batchSize);
  }
}

static void EvalDerivative(
  double* const in,
   size_t const iDim,
  double* const out) {
  (void)in;
  methodOf(LossFun);

  // Simply return the average derivatives which has been precomputed in
  // UpdateDerivatives
  out[0] = that->avgDerivatives[iDim];
}

static LossFun LossFunCreate(
  CapyNeuralNetwork* const nn,
  CapyDataset const* const dataset,
      CapyMat const* const mat,
   CapyPredictorType const type) {
  LossFun that;
  CapyInherits(that, CapyMathFun, (nn->nbParam, 1));
  that.destruct = LossFunDestruct;
  that.loss = 0.0;
  that.type = type;
  safeMalloc(that.derivatives, nn->nbParam);
  safeMalloc(that.avgDerivatives, nn->nbParam);
  loop(i, nn->nbParam) {
    that.derivatives[i] = 0.0;
    that.avgDerivatives[i] = 0.0;
  }
  safeMalloc(that.avgLossNodes, nn->nbOutput);
  loop(i, nn->nbOutput) that.avgLossNodes[i] = 0.0;
  safeMalloc(that.lossInputs, nn->nbOutput);
  loop(i, nn->nbOutput) that.lossInputs[i] = 0.0;
  safeMalloc(that.lossTargets, nn->nbOutput);
  loop(i, nn->nbOutput) that.lossTargets[i] = 0.0;
  that.nn = nn;
  that.dataset = dataset;
  that.mat = mat;
  that.nnOut = NULL;
  safeMalloc(that.nnOut, nn->nbOutput);
  that.eval = Eval;
  that.evalDerivative = EvalDerivative;
  that.updateDerivatives = UpdateDerivatives;
  return that;
}

// Initialise the parameters value using the Xavier method
// Input:
//   params: the array of parameters
// Ouput:
//   The array of parameters is updated
static void InitParamsXavier(double* const params) {
  methodOf(CapyNNPredictor);
  CapyRandom rnd = CapyRandomCreate(that->seed);
  double means[1] = {0.0};
  double stdDevs[1] = {0.0};
  CapyDistNormal dist = CapyDistNormalCreate(1, means, stdDevs);
  loop(iLayer, that->nn->nbLayer) {
    CapyNNLayer* layer = that->nn->layers + iLayer;
    dist.stdDev[0] = 1.0 / (double)(layer->nbNode);
    loop(iNode, layer->nbNode) {
      CapyNNNode* node = layer->nodes + iNode;
      params[node->idxBias] = 0.0;
      loop(iLink, node->nbLink) {
        CapyNNLink* link = node->links + iLink;
        CapyDistEvt evt = $(&rnd, getDistEvt)((CapyDist*)&dist);
        params[link->idxWeight] = evt.vec.vals[0];
      }
    }
  }
  $(&dist, destruct)();
  $(&rnd, destruct)();
}

// Train the predictor on a dataset
// Input:
//   dataset: the dataset
// Output:
//   The predictor is trained.
// Exception:
//   May raise CapyExc_UnsupportedFormat
static void Train(CapyDataset const* const dataset) {
  methodOf(CapyNNPredictor);

  // Ensure the predicted field type matches the predictor type
  size_t idxOutput = $(dataset, getIdxOutputField)(that->iOutput);
  if(
    that->type == capyPredictorType_categorical &&
    dataset->fields[idxOutput].type != capyDatasetFieldType_cat
  ) {
    raiseExc(CapyExc_UnsupportedFormat);
    return;
  }

  // Ensure the loss type matches the predicor type
  if(
    that->lossType == capyNNPredictorLossType_cross_entropy &&
    that->type != capyPredictorType_categorical
  ) {
    raiseExc(CapyExc_InvalidParameters);
    return;
  }

  // Allocate memory for the scaling ranges
  loop(i, that->nbInput) $(that->scalingFrom + i, destruct)();
  free(that->scalingFrom);
  that->nbInput = $(dataset, getNbInput)();
  safeMalloc(that->scalingFrom, that->nbInput);
  loop(i, that->nbInput) that->scalingFrom[i] = CapyRangeDoubleCreate(0.0, 1.0);

  // Get the number of output of the neural network. As we use hot encoding it
  // is equal to the number of categories for the predicted field
  size_t nbOutput = 1;
  if(that->type == capyPredictorType_categorical) {
    nbOutput = dataset->fields[idxOutput].nbCategoryVal;
  }

  // Delete an eventual previously trained neural network
  CapyNeuralNetworkFree(&(that->nn));

  // Chrono to measure training time
  CapyChrono chrono = CapyChronoCreate();

  // Create the fully connected neural network
  that->nn = CapyNeuralNetworkAllocFullyConnected(
    that->nbInput, &(that->nnModel), nbOutput);

  // Convert the dataset into a matrix usable for training
  CapyMat mat = $(that, cvtDatasetToMat)(dataset);

  // Preprocess the input features in the training data
  $(that, scaleTrainingInputFeatures)(&mat, dataset);

  // Create the loss function instance
  LossFun loss = LossFunCreate(that->nn, dataset, &mat, that->type);
  loss.lossType = that->lossType;
  loss.huberThreshold = that->huberThreshold;

  // Init the row index for the current batch and batch size
  loss.iFirstRowBatch = 0;
  loss.batchSize =
    (dataset->nbRow < that->batchSize ? dataset->nbRow : that->batchSize);

  // Create the gradient descent
  CapyGradientDescent gradientDescent =
    CapyGradientDescentCreate((CapyMathFun*)&loss, that->nn->params);
  gradientDescent.learnRate = that->learnRate;

  // Set the type of gradient descent
  $(&gradientDescent, setType)(that->gradientDescentType);
  if(that->gradientDescentType == capyGradientDescent_momentum) {
    gradientDescent.momentum = that->momentum;
  } else if(that->gradientDescentType == capyGradientDescent_adam) {
    loop(i, 2) gradientDescent.decayRates[i] = that->decayRates[i];
  }

  // Initialise the parameters using the Xavier method
  // TODO: should allow for other methods too
  $(that, initParams)(gradientDescent.in.vals);

  // Sliding average to calculate the loss average over the batches
  CapySlidingAverage avgLoss =
    CapySlidingAverageCreate(dataset->nbRow / loss.batchSize);

  // Initialise the derivative value of the cost wrt each parameters
  $(&loss, updateDerivatives)(gradientDescent.in.vals);

  // Variable to memorise the current best params
  double* bestParams = NULL;
  safeMalloc(bestParams, that->nn->nbParam);
  if(bestParams == NULL) return;
  $(&avgLoss, update)(loss.loss);
  that->bestLoss = avgLoss.val;
  memcpy(
    bestParams,
    gradientDescent.in.vals,
    sizeof(double) * that->nn->nbParam);

  // Loop on the training step until the available time or number of iteration
  // is consumed or the gradient descent becomes null
  $(&chrono, start)();
  double elapsedTime = 0.0;
  that->nbIterTrain = 0;
  do {

    // Step the gradient descent
    $(&gradientDescent, step)();

    // Update the derivative value of the cost wrt each parameters
    $(&loss, updateDerivatives)(gradientDescent.in.vals);

    // Update the best loss
    $(&avgLoss, update)(loss.loss);
    if(that->bestLoss > avgLoss.val) {
      that->bestLoss = avgLoss.val;
      memcpy(
        bestParams,
        gradientDescent.in.vals,
        sizeof(double) * that->nn->nbParam);
    }

    // Increment the number of iteration
    ++(that->nbIterTrain);

    // Update the index of the first row
    loss.iFirstRowBatch =
      (loss.iFirstRowBatch + that->batchSize) % dataset->nbRow;

    // Update the elapsed time
    $(&chrono, stop)();
    elapsedTime = $(&chrono, getElapsedTime)(capyChrono_second);

    // Update the gradient norm
    that->gradientNorm = CapyVecGetNorm(&(gradientDescent.gradient));

    // If in verbose mode print some infos
    if(that->verbose) {
      fprintf(
        stderr,
        " %.0lfs/%.0lfs #%lu: best loss %lf, loss %lf, gradient %lf       \r",
        elapsedTime, that->timeTraining, that->nbIterTrain,
        that->bestLoss, avgLoss.val, that->gradientNorm);
    }
  } while(
    that->bestLoss > that->lossEpsilon &&
    that->gradientNorm > that->gradientNormEpsilon &&
    (
      (that->timeTraining > 0 && elapsedTime < that->timeTraining) ||
      (that->nbIterTrainMax > 0 && that->nbIterTrain < that->nbIterTrainMax)
    )
  );

  // If in verbose mode print the end line
  if(that->verbose) {
    printf(
      " %.0lfs/%.0lfs #%lu: best loss %lf, loss %lf, gradient %lf       \n",
      elapsedTime, that->timeTraining, that->nbIterTrain,
      that->bestLoss, avgLoss.val, that->gradientNorm);
  }

  // Copy the best params in the neural network
  memcpy(that->nn->params, bestParams, sizeof(double) * that->nn->nbParam);

  // Free memory
  free(bestParams);
  $(&gradientDescent, destruct)();
  $(&loss, destruct)();
  CapyMatDestruct(&mat);
  $(&chrono, destruct)();
  $(&avgLoss, destruct)();
}

// Classify an input
// Input:
//   inp: the input
// Output:
//   Return the result of prediction
static CapyPredictorPrediction Predict(CapyVec const* const inp) {
  methodOf(CapyNNPredictor);

  // Variable to memorise the result
  CapyPredictorPrediction pred;

  // Apply input features scaling
  CapyVec scaledInp = CapyVecCreate(inp->dim);
  loop(i, inp->dim) scaledInp.vals[i] = inp->vals[i];
  $(that, scaleInputFeatures)(&scaledInp);

  // Evaluate the output
  double out[that->nn->nbOutput];
  $(that->nn, eval)(scaledInp.vals, out);

  // Update the result according to the type of predictor. 
  if(that->type == capyPredictorType_categorical) {

    // For categorical predictor the output is hot encoded so the predicted
    // category is the argmax. The confidence is the value of the predicted
    // category in the normalised output.
    pred.category = iMax_double(out, that->nn->nbOutput);
    CapyVec v = {.dim = that->nn->nbOutput, .vals = out};
    CapyVecNormalise(&v);
    pred.confidence = out[pred.category];
  } else if(that->type == capyPredictorType_numerical) {
    pred.val = out[0];
    pred.confidence = 0.0;
  } else raiseExc(CapyExc_UndefinedExecution);

  // Free memory
  CapyVecDestruct(&scaledInp);

  // Return the result
  return pred;
}

// Convert a CapyDataset into a CapyMat usable by the predictor
// Input:
//   dataset: the dataset to be converted
// Output:
//   Return a matrix formatted as necessary
static CapyMat CvtDatasetToMat(CapyDataset const* const dataset) {
  methodOf(CapyNNPredictor);
  CapyMat mat = {0};
  if(that->type == capyPredictorType_categorical) {
    mat = $(dataset, cvtToMatForSingleCatPredictor)(that->iOutput);
  } else if(that->type == capyPredictorType_numerical) {
    mat = $(dataset, cvtToMatForNumericalPredictor)(that->iOutput);
  } else raiseExc(CapyExc_UndefinedExecution);
  return mat;
}

// Export the predictor as a C function
// Input:
//   stream: the stream where to export
//   name: the name of the function
//   dataset: the training dataset
// Output:
//   A ready to use C function implementing the predictor is written on the
//   stream. See the comment exported with the function to know how to use
//   the exported function.
static void ExportToCFun(
               FILE* const stream,
         char const* const name,
  CapyDataset const* const dataset) {
  methodOf(CapyNNPredictor);

  // Write the include-s
  fprintf(stream, "#include <stdlib.h>\n");
  fprintf(stream, "#include <stdio.h>\n");
  fprintf(stream, "#include <string.h>\n");
  fprintf(stream, "#include <math.h>\n");
  fprintf(stream, "\n");

  // Export the activation functions
  loop(iLayer, that->nn->nbLayer) {
    CapyNNLayer* layer = that->nn->layers + iLayer;
    if(layer->activation != NULL) {
      char* nameActivation = NULL;
      safeSPrintf(&nameActivation, "%sLayer%03dActivation", name, iLayer);
      $(layer->activation, exportToCFun)(
        stream, nameActivation);
      free(nameActivation);
    }
  }

  // Write the prediction function's comment block
  fprintf(stream, "// Neural network prediction\n");
  fprintf(stream, "// Input:\n");
  size_t nbInput = that->nn->nbInput;
  fprintf(
    stream,
    "// u: the predicted input, array of %lu double values as follow:\n",
    nbInput);
  loop(iInput, nbInput) {
    size_t iFieldInput = $(dataset, getIdxInputField)(iInput);
    fprintf(
      stream,
      "// u[%lu]: [%s], ",
      iInput,
      dataset->fields[iFieldInput].label);
    if(dataset->fields[iFieldInput].type == capyDatasetFieldType_num) {
      fprintf(
        stream,
        "trained on [%.9lf, %.9lf]\n",
        dataset->fields[iFieldInput].range.min,
        dataset->fields[iFieldInput].range.max);
    } else {
      fprintf(stream, "encoded as ");
      unsigned long n = dataset->fields[iFieldInput].nbCategoryVal - 1;
      if(n < 1) n = 1;
      loop(iVal, dataset->fields[iFieldInput].nbCategoryVal) {
        fprintf(
          stream,
          "[%s]=%lu%s",
          dataset->fields[iFieldInput].categoryVals[iVal], iVal,
          iVal < (dataset->fields[iFieldInput].nbCategoryVal - 1) ? ", " : "");
      }
      fprintf(stream, "\n");
    }
  }
  size_t nbOutput = that->nn->nbOutput;
  size_t iFieldOutput =
    $(dataset, getIdxOutputField)(that->iOutput);
  if(that->type == capyPredictorType_categorical) {
    fprintf(
      stream,
      "// v: the result of prediction, array of %lu double values in [-1,1]\n"
      "//    representing the confidence (higher is more confident) that\n"
      "//    [%s] is:\n",
      nbOutput, dataset->fields[iFieldOutput].label);
    loop(iOutput, nbOutput) {
      char const* lblPredVal =
        dataset->fields[iFieldOutput].categoryVals[iOutput];
      fprintf(
        stream,
        "// v[%lu]: [%s]\n",
        iOutput, lblPredVal);
    }
    fprintf(stream, "// Output:\n");
    fprintf(
      stream,
      "// Return argmax(v), the index of the predicted value\n");
  } else if(that->type == capyPredictorType_numerical) {
    fprintf(stream, "// Output:\n");
    fprintf(
      stream,
      "// Return the predicted value\n");
  } else raiseExc(CapyExc_UndefinedExecution);

  // Write the prediction function's header
  if(that->type == capyPredictorType_categorical) {
    fprintf(
      stream,
      "int %s(double const* const u, double* const v) {\n",
      name);
  } else if(that->type == capyPredictorType_numerical) {
    fprintf(
      stream,
      "double %s(double const* const u) {\n",
      name);
  } else raiseExc(CapyExc_UndefinedExecution);

  // Write the prediction algorithm
  if(that->type == capyPredictorType_numerical) {
    fprintf(stream, "  double v[1] = {0.0};\n");
  }

  // Scale the input features
  $(that, exportScaleInputToCFun)(stream);

  // Declare the nodes
  loop(iLayer, that->nn->nbLayer) {
    if(iLayer != 0 && iLayer != that->nn->nbLayer - 1) {
      fprintf(
        stream,
        "  static double node%lu[%lu] = {0};\n",
        iLayer, that->nn->layers[iLayer].nbNode);
    }
  }
  loop(iLayer, that->nn->nbLayer) if(iLayer > 0) {
    CapyNNLayer* layer = that->nn->layers + iLayer;
    loop(iNode, layer->nbNode) {
      CapyNNNode* node = layer->nodes + iNode;
      if(iLayer != that->nn->nbLayer - 1) fprintf(stream, "  node%lu", iLayer);
      else fprintf(stream, "  v");
      fprintf(stream, "[%lu] = ", iNode);
      if(layer->activation != NULL) {
        fprintf(stream, "%sLayer%03luActivation(", name, iLayer);
      }
      fprintf(stream, "(double)%a + ", that->nn->params[node->idxBias]);
      loop(iLink, node->nbLink) {
        CapyNNLink* link = node->links + iLink;
        if(iLink > 0) fprintf(stream, " + ");
        fprintf(stream, "(double)%a * ", that->nn->params[link->idxWeight]);
        if(link->iLayer == 0) fprintf(stream, "w");
        else if(link->iLayer == that->nn->nbLayer - 1) fprintf(stream, "v");
        else fprintf(stream, "node%lu", link->iLayer);
        fprintf(stream, "[%lu]", link->iNode);
      }
      if(layer->activation != NULL) fprintf(stream, ")");
      fprintf(stream, ";\n");
    }
  }
  if(that->type == capyPredictorType_categorical) {
    fprintf(stream, "  double norm = 0.0;\n");
    fprintf(
      stream,
      "  for(int i = 0; i < %lu; ++i) norm += v[i] * v[i];\n", nbOutput);
    fprintf(stream, "  norm = 1.0 / sqrt(norm);\n");
    fprintf(stream, "  for(int i = 0; i < %lu; ++i) v[i] *= norm;\n", nbOutput);
    fprintf(stream, "  int pred = 0;\n");
    fprintf(
      stream,
      "  for(int i = 1; i < %lu; ++i) if(v[pred] < v[i]) pred = i;\n",
      nbOutput);
    fprintf(stream, "  return pred;\n");
  } else if(that->type == capyPredictorType_numerical) {
    fprintf(stream, "  return v[0];\n");
  } else raiseExc(CapyExc_UndefinedExecution);
  fprintf(stream, "}\n");
  fprintf(stream, "\n");

  // Write the driver function
  fprintf(stream, "// Driver function\n");
  fprintf(stream, "int main(int argc, char** argv) {\n");
  fprintf(stream, "  if(argc != %lu) {\n", nbInput + 1);
  fprintf(stream, "    printf(\"Expect %lu arguments\\n\");\n", nbInput);
  fprintf(stream, "    return 1;\n");
  fprintf(stream, "  }\n");
  fprintf(stream, "  double u[%lu] = {0};\n", nbInput);
  fprintf(stream, "  double v[%lu] = {0};\n", nbOutput);
  loop(iInput, nbInput) {
    size_t iFieldInput = $(dataset, getIdxInputField)(iInput);
    if(dataset->fields[iFieldInput].type == capyDatasetFieldType_num) {
      fprintf(stream, "  u[%lu] = atof(argv[%lu]);\n", iInput, iInput + 1);
    } else {
      loop(iVal, dataset->fields[iFieldInput].nbCategoryVal) {
        fprintf(
          stream,
          "  if(strcmp(argv[%lu], \"%s\") == 0) u[%lu] = %lu.0;\n",
          iInput + 1,
          dataset->fields[iFieldInput].categoryVals[iVal],
          iInput, iVal);
      }
    }
  }
  fprintf(stream, "\n");
  if(that->type == capyPredictorType_categorical) {
    fprintf(stream, "  int pred = %s(u, v);\n", name);
    fprintf(stream, "  char* predVal[%lu] = {\n", nbOutput);
    loop(iOutput, nbOutput) {
      char const* lblPredVal =
        dataset->fields[iFieldOutput].categoryVals[iOutput];
      fprintf(
        stream,
        "    \"%s\",\n",
        lblPredVal);
    }
    fprintf(stream, "  };\n");
    fprintf(stream, "  printf(\"%%s\\n\", predVal[pred]);\n");
  } else if(that->type == capyPredictorType_numerical) {
    fprintf(stream, "  v[0] = %s(u);\n", name);
    fprintf(stream, "  printf(\"%%lf\\n\", v[0]);\n");
  } else raiseExc(CapyExc_UndefinedExecution);
  fprintf(stream, "  return 0;\n");
  fprintf(stream, "}\n");
  fprintf(stream, "\n");
}

// Export the predictor as a HTML web app
// Input:
//   stream: the stream where to export
//   title: the title of the web app
//   dataset: the training dataset
//   expectedAccuracy: the expected accuracy of the predictor (in [0,1])
// Output:
//   A ready to use web app implementing the predictor is written on the
//   stream.
static void ExportToHtml(
               FILE* const stream,
         char const* const title,
  CapyDataset const* const dataset,
              double const expectedAccuracy) {
  methodOf(CapyNNPredictor);

  // Write the head of the page (up to <body>)
  fprintf(stream, "<!DOCTYPE html>\n");
  fprintf(stream, "<html>\n");

  // Write the body
  $(that, exportBodyToHtml)(stream, title, dataset, expectedAccuracy);

  // Write the javascript
  fprintf(stream, "<script>\n");
  fprintf(
    stream,
    "function range(size, startAt = 0) { "
    "return [...Array(size).keys()].map(i => i + startAt);}\n");
  fprintf(
    stream,
    "function elem(id) { return document.getElementById(id); }\n");
  loop(iLayer, that->nn->nbLayer) {
    CapyNNLayer* layer = that->nn->layers + iLayer;
    if(layer->activation != NULL) {
      char* nameActivation = NULL;
      safeSPrintf(&nameActivation, "layer%03dActivation", iLayer);
      $(layer->activation, exportToJavascript)(
        stream, nameActivation);
      free(nameActivation);
    }
  }
  fprintf(stream, "function predict() {\n");
  fprintf(stream, "  let u = getInput();\n");
  fprintf(stream, "  let v = [];\n");
  loop(iLayer, that->nn->nbLayer) {
    if(iLayer != 0 && iLayer != that->nn->nbLayer - 1) {
      fprintf(stream, "  let node%lu = [];\n", iLayer);
    }
  }
  loop(iLayer, that->nn->nbLayer) if(iLayer > 0) {
    CapyNNLayer* layer = that->nn->layers + iLayer;
    loop(iNode, layer->nbNode) {
      CapyNNNode* node = layer->nodes + iNode;
      if(iLayer != that->nn->nbLayer - 1) fprintf(stream, "  node%lu", iLayer);
      else fprintf(stream, "  v");
      fprintf(stream, "[%lu] = ", iNode);
      if(layer->activation != NULL) {
        fprintf(stream, "layer%03luActivation(", iLayer);
      }
      fprintf(stream, "%.9lf + ", that->nn->params[node->idxBias]);
      loop(iLink, node->nbLink) {
        CapyNNLink* link = node->links + iLink;
        if(iLink > 0) fprintf(stream, " + ");
        fprintf(stream, "%.9lf * ", that->nn->params[link->idxWeight]);
        if(link->iLayer == 0) fprintf(stream, "u");
        else if(link->iLayer == that->nn->nbLayer - 1) fprintf(stream, "v");
        else fprintf(stream, "node%lu", link->iLayer);
        fprintf(stream, "[%lu]", link->iNode);
      }
      if(layer->activation != NULL) fprintf(stream, ")");
      fprintf(stream, ";\n");
    }
  }
  if(that->type == capyPredictorType_categorical) {
    fprintf(stream, "  let norm = 0.0;\n");
    size_t nbValOutput =
      $(dataset, getNbValOutputField)(that->iOutput);
    fprintf(
      stream,
      "  for(let i = 0; i < %lu; i = i + 1) norm += v[i] * v[i];\n",
      nbValOutput);
    fprintf(stream, "  norm = 1.0 / Math.sqrt(norm);\n");
    fprintf(
      stream,
      "  for(let i = 0; i < %lu; i = i + 1) v[i] *= norm;\n", nbValOutput);
    fprintf(stream, "  let pred = 0;\n");
    fprintf(
      stream,
      "  for(let i = 1; i < %lu; i = i + 1) if(v[pred] < v[i]) pred = i;\n",
      nbValOutput);
    fprintf(
      stream,
      "  elem(\"spanConfidence\").innerHTML = Math.abs(v[pred]).toFixed(2);\n");
    fprintf(
      stream,
      "  for(let i = 0; i < %lu; i = i + 1) "
      "elem(\"pred\" + i).checked = false;\n",
      nbValOutput);
    fprintf(stream, "  elem(\"pred\" + pred).checked = true;\n");
  } else if(that->type == capyPredictorType_numerical) {
    fprintf(
      stream,
      "  elem(\"pred\").value = v[0];\n");
  } else raiseExc(CapyExc_UndefinedExecution);
  fprintf(stream, "}\n");
  fprintf(stream, "</script>\n");

  // Write the end of the page
  fprintf(stream, "</html>\n");
}

// Free the memory used by a CapyNNPredictor
static void Destruct(void) {
  methodOf(CapyNNPredictor);
  $(that, destructCapyPredictor)();
  CapyNNModelDestruct(&(that->nnModel));
  CapyNeuralNetworkFree(&(that->nn));
}

// Clone a CapyNNPredictor
// Output:
//   Return a clone of the predictor.
static void* Clone(void) {
  methodOf(CapyNNPredictor);
  CapyNNPredictor* clone = CapyNNPredictorAlloc(&(that->nnModel), that->type);
  clone->nnModel = CapyNNModelCopy(&(that->nnModel));
  clone->verbose = that->verbose;
  clone->timeTraining = that->timeTraining;
  clone->nbIterTrain = that->nbIterTrain;
  clone->nbIterTrainMax = that->nbIterTrainMax;
  clone->learnRate = that->learnRate;
  clone->batchSize = that->batchSize;
  clone->momentum = that->momentum;
  clone->seed = that->seed;
  clone->bestLoss = that->bestLoss;
  clone->gradientNorm = that->gradientNorm;
  clone->gradientNormEpsilon = that->gradientNormEpsilon;
  clone->lossEpsilon = that->lossEpsilon;
  return clone;
}

//  Save the predictor to a stream
//  Input:
//    stream: the stream on which to save
//  Output:
//    The predictor data are saved on the stream
static void Save(FILE* const stream) {
  methodOf(CapyNNPredictor);
  safeFWrite(stream, sizeof(that->iOutput), &(that->iOutput));
  safeFWrite(stream, sizeof(that->type), &(that->type));
  safeFWrite(stream, sizeof(that->featureScaling), &(that->featureScaling));
  safeFWrite(stream, sizeof(that->nbInput), &(that->nbInput));
  loop(i, that->nbInput) {
    safeFWrite(
      stream, sizeof(that->scalingFrom[i].min), &(that->scalingFrom[i].min));
    safeFWrite(
      stream, sizeof(that->scalingFrom[i].max), &(that->scalingFrom[i].max));
  }
  safeFWrite(stream, sizeof(that->scalingTo.min), &(that->scalingTo.min));
  safeFWrite(stream, sizeof(that->scalingTo.max), &(that->scalingTo.max));
  $(that->nn, save)(stream);
}


// Create a CapyNNPredictor
// Input:
//   model: the neural network model
//   type: type of predictor
// Output:
//   Return a CapyNNPredictor
CapyNNPredictor CapyNNPredictorCreate(
  CapyNNModel const* const model,
   CapyPredictorType const type) {
  CapyNNPredictor that;
  CapyInherits(that, CapyPredictor, (type));
  that.nnModel = CapyNNModelCopy(model);
  that.nn = NULL;
  that.verbose = false;
  that.timeTraining = 60;
  that.nbIterTrain = 0;
  that.nbIterTrainMax = 0;
  that.learnRate = 0.1;
  that.batchSize = 100;
  that.momentum = 0.1;
  that.seed = 0;
  that.bestLoss = 0.0;
  that.gradientNorm = 0.0;
  that.gradientNormEpsilon = 1e-6;
  that.lossEpsilon = 1e-6;
  that.gradientDescentType = capyGradientDescent_adam;
  that.decayRates[0] = 0.9;
  that.decayRates[1] = 0.999;
  that.lossType = capyNNPredictorLossType_mse;
  that.huberThreshold = 1.0;
  that.destruct = Destruct;
  that.clone = Clone;
  that.train = Train;
  that.predict = Predict;
  that.cvtDatasetToMat = CvtDatasetToMat;
  that.exportToCFun = ExportToCFun;
  that.exportBodyToHtml = that.exportToHtml;
  that.exportToHtml = ExportToHtml;
  that.initParams = InitParamsXavier;
  that.save = Save;
  return that;
}

// Allocate memory for a new CapyNNPredictor and create it
// Input:
//   model: the neural network model
//   type: type of predictor
// Output:
//   Return a CapyNNPredictor
// Exception:
//   May raise CapyExc_MallocFailed.
CapyNNPredictor* CapyNNPredictorAlloc(
  CapyNNModel const* const model,
   CapyPredictorType const type) {
  CapyNNPredictor* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyNNPredictorCreate(model, type);
  return that;
}

// Free the memory used by a CapyNNPredictor* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyNNPredictor to free
void CapyNNPredictorFree(CapyNNPredictor** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Load a CapyNNPredictor from a stream
// Input:
//   stream: the stream from which the predictor is loaded
// Output:
//   Return a CapyNNPredictor
CapyNNPredictor* CapyNNPredictorLoad(FILE* const stream) {
  CapyNNPredictor* that = NULL;
  safeMalloc(that, 1);
  if(that == NULL) return NULL;
  size_t iOutput = 0;
  safeFRead(stream, sizeof(iOutput), &iOutput);
  CapyPredictorType type = capyPredictorType_categorical;
  safeFRead(stream, sizeof(type), &type);
  CapyPredictorFeatureScaling scaling = capyPredictorFeatureScaling_none;
  safeFRead(stream, sizeof(scaling), &scaling);
  size_t nbInput = 0;
  safeFRead(stream, sizeof(nbInput), &nbInput);
  CapyRangeDouble* scalingFrom = NULL;
  safeMalloc(scalingFrom, nbInput);
  if(scalingFrom == NULL) {
    free(that);
    return NULL;
  }
  loop(i, nbInput) scalingFrom[i] = CapyRangeDoubleCreate(0, 1);
  loop(i, nbInput) {
    safeFRead(
      stream, sizeof(scalingFrom[i].min), &(scalingFrom[i].min));
    safeFRead(
      stream, sizeof(scalingFrom[i].max), &(scalingFrom[i].max));
  }
  CapyRangeDouble scalingTo = CapyRangeDoubleCreate(0, 1);
  safeFRead(stream, sizeof(scalingTo.min), &(scalingTo.min));
  safeFRead(stream, sizeof(scalingTo.max), &(scalingTo.max));
  that->nn = CapyNeuralNetworkLoad(stream, &(that->nnModel));
  CapyInherits(*that, CapyPredictor, (type));
  that->iOutput = iOutput;
  that->type = type;
  that->verbose = false;
  that->timeTraining = 60;
  that->nbIterTrain = 0;
  that->nbIterTrainMax = 0;
  that->learnRate = 0.1;
  that->batchSize = 100;
  that->momentum = 0.1;
  that->seed = 0;
  that->bestLoss = 0.0;
  that->gradientNorm = 0.0;
  that->gradientNormEpsilon = 1e-6;
  that->lossEpsilon = 1e-6;
  that->featureScaling = scaling;
  that->nbInput = nbInput;
  that->scalingFrom = scalingFrom;
  that->scalingTo = scalingTo;
  that->destruct = Destruct;
  that->clone = Clone;
  that->train = Train;
  that->predict = Predict;
  that->cvtDatasetToMat = CvtDatasetToMat;
  that->exportToCFun = ExportToCFun;
  that->exportBodyToHtml = that->exportToHtml;
  that->exportToHtml = ExportToHtml;
  that->initParams = InitParamsXavier;
  that->save = Save;
  return that;
}
