// ------------------------------ nnPredictor.h -----------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_NNPREDICTOR_H
#define CAPY_NNPREDICTOR_H
#include "externalHeaders.h"
#include "cext.h"
#include "predictor.h"
#include "neuralNetwork.h"
#include "dataset.h"
#include "random.h"
#include "gradientDescent.h"

// Description:
// Class implementing a predictor based on a fully connected neural network.

// Type of loss function used for training
typedef enum {
  capyNNPredictorLossType_mse,
  capyNNPredictorLossType_mae,
  capyNNPredictorLossType_huber,
  capyNNPredictorLossType_cross_entropy,
  capyNNPredictorLossType_nb,
} CapyNNPredictorLossType;

// NNPredictor object
typedef struct CapyNNPredictor {

  // Inherits CapyPredictor
  CapyPredictorDef;

  // Neural network model
  CapyNNModel nnModel;

  // Neural network
  CapyNeuralNetwork* nn;

  // Verbose mode (default: false)
  bool verbose;
  CapyPad(bool, 1);

  // Time available for training (in second, default: 60, no time limit if 0)
  double timeTraining;

  // Number of iteration available for training (default: 0, no limit if 0)
  size_t nbIterTrainMax;

  // Batch size for training (default: 100)
  size_t batchSize;

  // Counter for iteration during training
  size_t nbIterTrain;

  // Step size for the gradient descent (default: 0.1)
  double learnRate;

  // Momentum for the gradient descent (default: 0.1)
  double momentum;

  // Seed for the pseudo random generator (default: 0)
  CapyRandomSeed_t seed;

  // Best loss during training
  double bestLoss;

  // Current gradient norm during training
  double gradientNorm;

  // Threshold to stop the training if the gradient is null (default: 1e-6)
  double gradientNormEpsilon;

  // Threshold to stop the training if the loss is null (default: 1e-6)
  double lossEpsilon;

  // Type of gradient descent (default: adam)
  CapyGradientDescentType gradientDescentType;
  CapyPad(CapyGradientDescentType, gradientDescentType);

  // Threshold for Huber loss (default: 1.0)
  double huberThreshold;

  // Decay rates for adam gradient descent (default: [0.9, 0.999]
  double decayRates[2];

  // Type of loss function (default: capyNNPredictorLossType_mse)
  CapyNNPredictorLossType lossType;
  CapyPad(CapyNNPredictorLossType, lossType);

  // Destructor
  void (*destructCapyPredictor)(void);

  // Initialise the parameters value
  // Input:
  //   params: the array of parameters
  // Ouput:
  //   The array of parameters is updated
  void (*initParams)(double* const params);

  // Function herited from the parent to export the body to HTML. 
  // Input:
  //   stream: the stream where to export
  //   title: the title of the web app
  //   dataset: the training dataset
  //   expectedAccuracy: the expected accuracy of the predictor (in [0,1])
  // Output:
  //   The <head> and <body> part of a ready to use web app implementing the
  //   predictor is written on the stream. The web app can be completed by
  //   calling exportToHtml on the predictor to write the <script> part.
  void (*exportBodyToHtml)(
                 FILE* const stream,
           char const* const title,
    CapyDataset const* const dataset,
                double const expectedAccuracy);
} CapyNNPredictor;

// Create a CapyNNPredictor
// Input:
//   model: the neural network model
//   type: type of predictor
// Output:
//   Return a CapyNNPredictor
CapyNNPredictor CapyNNPredictorCreate(
  CapyNNModel const* const model,
   CapyPredictorType const type);

// Allocate memory for a new CapyNNPredictor and create it
// Input:
//   model: the neural network model
//   type: type of predictor
// Output:
//   Return a CapyNNPredictor
// Exception:
//   May raise CapyExc_MallocFailed.
CapyNNPredictor* CapyNNPredictorAlloc(
  CapyNNModel const* const model,
   CapyPredictorType const type);

// Free the memory used by a CapyNNPredictor* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyNNPredictor to free
void CapyNNPredictorFree(CapyNNPredictor** const that);

// Load a CapyNNPredictor from a stream
// Input:
//   stream: the stream from which the predictor is loaded
// Output:
//   Return a CapyNNPredictor
CapyNNPredictor* CapyNNPredictorLoad(FILE* const stream);
#endif
