// --------------------------------- noise.c ----------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "noise.h"

// Evaluate the noise for a given input.
// Input:
//    in: the input vector
//   out: array of double updated with the result of evaluation
static void Eval(
  double const* const in,
        double* const out) {
  (void)in; (void)out;
  raiseExc(CapyExc_UndefinedExecution);
  assert(false && "Noise.eval is undefined.");
}

// Initialise the noise.
// Output:
//   The noise is initialise with new random values
static void Init(void) {
  raiseExc(CapyExc_UndefinedExecution);
  assert(false && "Noise.init is undefined.");
}

// Free the memory used by a CapyNoise
static void Destruct(void) {
  methodOf(CapyNoise);
  $(that, destructCapyMathFun)();
  $(&(that->rng), destruct)();
}

// Create a CapyNoise
// Input:
//     seed: seed of the noise
//    dimIn: number of inputs
//   dimOut: number of outputs
// Output:
//   Return a CapyNoise
CapyNoise CapyNoiseCreate(
  CapyRandomSeed_t const seed,
            size_t const dimIn,
            size_t const dimOut) {

  // Create the Noise
  CapyNoise that = {
    .rng = CapyRandomCreate(seed),
  };
  CapyInherits(that, CapyMathFun, (dimIn, dimOut));
  that.destruct = Destruct;
  that.eval = Eval;
  that.init = Init;
  return that;
}

// Allocate memory for a new CapyNoise and create it
// Input:
//     seed: seed of the noise
//    dimIn: number of inputs
//   dimOut: number of outputs
// Output:
//   Return a CapyNoise
// Exception:
//   May raise CapyExc_MallocFailed.
CapyNoise* CapyNoiseAlloc(
  CapyRandomSeed_t const seed,
            size_t const dimIn,
            size_t const dimOut) {

  // Allocate memory and create the new CapyNoise
  CapyNoise* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  CapyNoise b = CapyNoiseCreate(seed, dimIn, dimOut);
  memcpy(that, &b, sizeof(CapyNoise));

  // Return the new CapyNoise
  return that;
}

// Free the memory used by a CapyNoise* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyNoise to free
void CapyNoiseFree(CapyNoise** const that) {
  if(*that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
