// --------------------------------- noise.h ----------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_NOISE_H
#define CAPY_NOISE_H
#include "externalHeaders.h"
#include "cext.h"
#include "mathfun.h"
#include "random.h"

// Description:
// Noise generator parent class.

// CapyNoise object definition macro
#define CapyNoiseDef struct {        \
  struct CapyMathFunDef;             \
  CapyRandom rng;                    \
  void (*destructCapyMathFun)(void); \
  void (*init)(void);                \
}

// CapyNoise object
typedef CapyNoiseDef CapyNoise;

// Create a CapyNoise
// Input:
//     seed: seed of the noise
//    dimIn: number of inputs
//   dimOut: number of outputs
// Output:
//   Return a CapyNoise
CapyNoise CapyNoiseCreate(
  CapyRandomSeed_t const seed,
            size_t const dimIn,
            size_t const dimOut);

// Allocate memory for a new CapyNoise and create it
// Input:
//     seed: seed of the noise
//    dimIn: number of inputs
//   dimOut: number of outputs
// Output:
//   Return a CapyNoise
// Exception:
//   May raise CapyExc_MallocFailed.
CapyNoise* CapyNoiseAlloc(
  CapyRandomSeed_t const seed,
            size_t const dimIn,
            size_t const dimOut);

// Free the memory used by a CapyNoise* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyNoise to free
void CapyNoiseFree(CapyNoise** const that);
#endif
