// ---------------------------- pathfinder.c ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "pathfinder.h"

// Free the memory used by a CapyPath
static void DestructPath(void) {
  methodOf(CapyPath);
  if(that->points) free(that->points);
  that->nbStep = 0;
}

// Create a CapyPath
// Output:
//   Return a CapyPath
CapyPath CapyPathCreate(void) {
  CapyPath that = {
    .nbStep = 0,
    .points = NULL,
    .sumWeight = 0.0,
    .destruct = DestructPath,
  };
  return that;
}

// Structure for the set during A* algorithm
typedef struct CapyAStarNode CapyAStarNode;
struct CapyAStarNode {
  size_t iPoint;
  double fScore;
  CapyAStarNode* next;
};

// Search the path in a CapyPointCloud using the A* algorithm
// Inputs:
//   pointCloud: the searched point cloud
//   iPointFrom: the start point of the path
//   iPointTo: the end point of the path
// Output:
//   Return a CapyPath, eventually with nbStep=0 if no path could be found.
//   The returned path minimised the links' weight. Uses
//   CapyPointCloud.estimateWeightPath() as a heuristic.
static CapyPath SearchPointCloud(
  CapyPointCloud const* const pointCloud,
                 size_t const iPointFrom,
                 size_t const iPointTo) {
  CapyPath path = CapyPathCreate();
  CapyAStarNode* openSet = NULL;
  safeMalloc(openSet, 1);
  assert(openSet != NULL);
  openSet->iPoint = iPointFrom;
  openSet->fScore =
    $(pointCloud, estimateWeightPath)(openSet->iPoint, iPointTo);
  openSet->next = NULL;
  double* gScore = NULL;
  safeMalloc(gScore, pointCloud->size);
  assert(gScore != NULL);
  loop(i, pointCloud->size) gScore[i] = -1.0;
  size_t* cameFrom = NULL;
  safeMalloc(cameFrom, pointCloud->size);
  assert(cameFrom != NULL);
  gScore[iPointFrom] = 0.0;
  while(openSet != NULL) {
    if(openSet->iPoint == iPointTo) {
      size_t iPoint = iPointTo;
      while(iPoint != iPointFrom) {
        path.nbStep++;
        iPoint = cameFrom[iPoint];
      }
      safeMalloc(path.points, path.nbStep + 1);
      assert(path.points != NULL);
      iPoint = iPointTo;
      size_t* point = path.points + path.nbStep;
      while(iPoint != iPointFrom) {
        *point = iPoint;
        iPoint = cameFrom[iPoint];
      }
      path.sumWeight = gScore[iPointTo];
      *point = iPointFrom;
      while(openSet != NULL) {
        CapyAStarNode* p = openSet;
        openSet = openSet->next;
        free(p);
      }
    } else {
      size_t iCurPoint = openSet->iPoint;
      CapyAStarNode* p = openSet;
      openSet = openSet->next;
      free(p);
      loop(iLink, pointCloud->nbLink) {
        if(pointCloud->links[iLink].iPoints[0] == iCurPoint) {
          size_t iNeighbour = pointCloud->links[iLink].iPoints[1];
          double score = gScore[iCurPoint] + pointCloud->links[iLink].weight;
          if(gScore[iNeighbour] < 0.0 || score < gScore[iNeighbour]) {
            cameFrom[iNeighbour] = iCurPoint;
            gScore[iNeighbour] = score;
            double fScore =
              gScore[iNeighbour] +
              $(pointCloud, estimateWeightPath)(iNeighbour, iPointTo);
            bool flagFound = false;
            CapyAStarNode* prev = NULL;
            p = openSet;
            while(flagFound == false && p != NULL) {
              if(p->fScore < fScore) prev = p;
              if(p->iPoint == iNeighbour) flagFound = true;
              else p = p->next;
            }
            if(flagFound == false) {
              CapyAStarNode* newNode = NULL;
              safeMalloc(newNode, 1);
              assert(newNode != NULL);
              if(prev != NULL) {
                newNode->next = prev->next;
                prev->next = newNode;
              } else {
                newNode->next = openSet;
                openSet = newNode;
              }
              newNode->iPoint = iNeighbour;
              newNode->fScore = fScore;
            }
          }
        }
      }
    }
  }
  free(gScore);
  free(cameFrom);
  return path;
}

// Free the memory used by a CapyPathFinder
static void Destruct(void) {
  return;
}

// Create a CapyPathFinder
// Output:
//   Return a CapyPathFinder
CapyPathFinder CapyPathFinderCreate(void) {
  CapyPathFinder that = {
    .destruct = Destruct,
    .searchPointCloud = SearchPointCloud,
  };
  return that;
}

// Allocate memory for a new CapyPathFinder and create it
// Output:
//   Return a CapyPathFinder
// Exception:
//   May raise CapyExc_MallocFailed.
CapyPathFinder* CapyPathFinderAlloc(void) {
  CapyPathFinder* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyPathFinderCreate();
  return that;
}

// Free the memory used by a CapyPathFinder* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyPathFinder to free
void CapyPathFinderFree(CapyPathFinder** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
