// ---------------------------- pathfinder.h ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_PATHFINDER_H
#define CAPY_PATHFINDER_H
#include "externalHeaders.h"
#include "cext.h"
#include "pointCloud.h"

// Description:
// PathFinder class.

// Structure to memorise a path
typedef struct CapyPath {

  // Number of steps in the path
  size_t nbStep;

  // Indices of the points (array of nbStep+1 elements) forming the path from
  // start to end
  size_t* points;

  // Sum of traversed link's weight
  double sumWeight;

  // Destructor
  void (*destruct)(void);
} CapyPath;

// Create a CapyPath
// Output:
//   Return a CapyPath
CapyPath CapyPathCreate(void);

// PathFinder object
typedef struct CapyPathFinder {

  // Destructor
  void (*destruct)(void);

  // Search the path in a CapyPointCloud using the A* algorithm
  // Inputs:
  //   pointCloud: the searched point cloud
  //   iPointFrom: the start point of the path
  //   iPointTo: the end point of the path
  // Output:
  //   Return a CapyPath, eventually with nbStep=0 if no path could be found.
  //   The returned path minimised the links' weight. Uses
  //   CapyPointCloud.estimateWeightPath() as a heuristic.
  CapyPath (*searchPointCloud)(
    CapyPointCloud const* const pointCloud,
                   size_t const iPointFrom,
                   size_t const iPointTo);
} CapyPathFinder;

// Create a CapyPathFinder
// Output:
//   Return a CapyPathFinder
CapyPathFinder CapyPathFinderCreate(void);

// Allocate memory for a new CapyPathFinder and create it
// Output:
//   Return a CapyPathFinder
// Exception:
//   May raise CapyExc_MallocFailed.
CapyPathFinder* CapyPathFinderAlloc(void);

// Free the memory used by a CapyPathFinder* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyPathFinder to free
void CapyPathFinderFree(CapyPathFinder** const that);
#endif
