// ------------------------------ pen.c ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "pen.h"

// Draw a point on a CapyImage.
// Input:
//   pos: the coordinate in pixel of the point
//   img: the image on which to draw
static void DrawPoint(
  double const* const pos,
       CapyImg* const img) {
  methodOf(CapyPen);

  // Variable to memorize intermediate values for optimization
  double squaredSize = that->size * that->size;
  double invSquaredSize = 1.0 / squaredSize;
  double c = 0.05 / that->size;

  // Loop over the area covered by the pen
  CapyRangeInt64 range = {
    .min = -(int64_t)(that->size), .max = (int64_t)(that->size)
  };
  loopRange(dx, range) loopRange(dy, range) {
    double d = (double)(dx * dx + dy * dy);
    if(d <= squaredSize) {

      // Convert to pixel coordinates
      CapyImgPos posPx = {
        .x = (CapyImgPos_t)pos[0] + (CapyImgPos_t)dx,
        .y = (CapyImgPos_t)pos[1] + (CapyImgPos_t)dy
      };

      // If the pixel is in the image
      if($(img, isValidCoord)(&posPx)) {

        // Get the current color
        CapyColorData color = *$(img, getColor)(&posPx);

        // Calculate the updated pixel color
        double strength = 1.0;
        if(that->hardness != capyPenHardness_none) {
          strength = (double)(that->hardness) * c;
          strength *= 1.0 - d * invSquaredSize;
        }
        strength *= color.RGBA[3];
        loop(i, 3) {
          color.RGB[i] =
            (1.0 - strength) * color.RGB[i] + strength * that->color.RGB[i];
        }
        color.RGBA[3] += that->color.RGBA[3];
        if(color.RGBA[3] > 1.0) color.RGBA[3] = 1.0;

        // Set the color in the image
        $(img, setColor)(&posPx, &color);
      }
    }
  }
}

// Draw a line on a CapyImage.
// Input:
//   from: the coordinate in pixel of the start point
//   to: the coordinate in pixel of the end point
//   img: the image on which to draw
static void DrawLine(
  double const* const from,
  double const* const to,
       CapyImg* const img) {
  methodOf(CapyPen);

  // Create the Bezier of the line
  CapyBezier line = CapyBezierCreate(1, 1, 2);
  $(&line, setCtrlById)(0, from);
  $(&line, setCtrlById)(1, to);

  // Draw the Bezier
  $(that, drawBezier)(&line, img);

  // Free memory
  $(&line, destruct)();
}

// Predeclaration of the recursive function used to draw Bezier curves and
// splines.
static void DrawBezierRec(
       CapyPen* const that,
   CapyMathFun* const curve,
       CapyImg* const img,
  double const* const tLowHigh,
        double* const posLow,
        double* const posHigh);

// Draw a CapyBezierSpline curve on a CapyImage. The curve is expected to
// have one input (in [0, spline.nbSegment]) and return two outputs (in pixel
// coordinates)
// Input:
//   spline: the spline to draw
//      img: the image on which to draw
static void DrawBezierSpline(
  CapyBezierSpline* const spline,
           CapyImg* const img) {
  methodOf(CapyPen);

  // Get the coordinates of the curve's extremities
  double posLowHigh[2][spline->dimOut];
  loop(i, 2) {
    double in = spline->domains[0].vals[i];
    $(spline, eval)(&in, posLowHigh[i]);
  }

  // Draw the start point
  $(that, drawPoint)(posLowHigh[0], img);

  // Start the recursion for the point in between points
  DrawBezierRec(
    that, (CapyMathFun*)spline, img, spline->domains[0].vals,
    posLowHigh[0], posLowHigh[1]);
}

// Draw a CapyBezier curve on a CapyImage. The curve is expected to
// have one input and return two outputs (in pixel coordinates)
// Input:
//   curve: the curve to draw
//   img: the image on which to draw
static void DrawBezier(
  CapyBezier* const curve,
     CapyImg* const img) {
  methodOf(CapyPen);

  // Get the coordinates of the curve's extremities
  double posLowHigh[2][curve->dimOut];
  loop(i, 2) {
    double in = curve->domains[0].vals[i];
    $(curve, eval)(&in, posLowHigh[i]);
  }

  // Draw the start point
  $(that, drawPoint)(posLowHigh[0], img);

  // Start the recursion for the points in between points
  DrawBezierRec(
    that, (CapyMathFun*)curve, img, curve->domains[0].vals,
    posLowHigh[0], posLowHigh[1]);
}

static void DrawBezierRec(
       CapyPen* const that,
   CapyMathFun* const curve,
       CapyImg* const img,
  double const* const tLowHigh,
        double* const posLow,
        double* const posHigh) {

  // End recursion if extrimities are neighbours
  int64_t d[2] = {0, 0};
  loop(i, 2) d[i] = (int64_t)posLow[i] - (int64_t)posHigh[i];
  if(labs(d[0]) <= 1 && labs(d[1]) <= 1 && tLowHigh[1] - tLowHigh[0] < 0.99) {
    return;
  }
  if(equald(tLowHigh[0], tLowHigh[1])) return;

  // Calculate the median position
  double tMedian = 0.5 * (tLowHigh[0] + tLowHigh[1]);
  double posMedian[curve->dimOut];
  $(curve, eval)(&tMedian, posMedian);

  // Check if the median has reached the extremities
  bool checkLow = (
    (CapyImgPos_t)posLow[0] == (CapyImgPos_t)posMedian[0] &&
    (CapyImgPos_t)posLow[1] == (CapyImgPos_t)posMedian[1]);
  bool checkHigh = (
    (CapyImgPos_t)posHigh[0] == (CapyImgPos_t)posMedian[0] &&
    (CapyImgPos_t)posHigh[1] == (CapyImgPos_t)posMedian[1]);

  // If the median is not at the lower extremity, continue the recursion
  // between the lower extremity and the median
  if(!checkLow) {
    double nextTLowHigh[2] = {tLowHigh[0], tMedian};
    double nextPosLow[2] = {posLow[0], posLow[1]};
    double nextPosHigh[2] = {posMedian[0], posMedian[1]};
    DrawBezierRec(that, curve, img, nextTLowHigh, nextPosLow, nextPosHigh);
  }

  // If the median is different from both extremity, draw it
  if(!checkLow && !checkHigh) $(that, drawPoint)(posMedian, img);

  // If the median is not at the higher extremity, continue the recursion
  // between the higher extremity and the median
  if(!checkHigh) {
    double nextTLowHigh[2] = {tMedian, tLowHigh[1]};
    double nextPosLow[2] = {posMedian[0], posMedian[1]};
    double nextPosHigh[2] = {posHigh[0], posHigh[1]};
    DrawBezierRec(that, curve, img, nextTLowHigh, nextPosLow, nextPosHigh);
  }
}

// Draw a quadrilateral on a CapyImage.
// Input:
//   quad: the quadrialteral to draw
//   img: the image on which to draw
static void DrawQuadrilateral(
  CapyQuadrilateral const* const quad,
                  CapyImg* const img) {
  methodOf(CapyPen);
  $(that, drawGeometry2D)((CapyGeometry2D const*)quad, img);
}

// Draw a rectangle on a CapyImage.
// Input:
//   rect: the rectangle to draw
//   img: the image on which to draw
static void DrawRectangle(
  CapyRectangle const* const rect,
              CapyImg* const img) {
  methodOf(CapyPen);
  $(that, drawGeometry2D)((CapyGeometry2D const*)rect, img);
}

// Draw a filled rectangle on a CapyImage.
// Input:
//   rect: the rectangle to draw
//   img: the image on which to draw
static void DrawFilledRectangle(
  CapyRectangle const* const rect,
              CapyImg* const img) {
  methodOf(CapyPen);
  CapyRangeDouble rangeX =
    {.min = rect->corners[0].x, .max = rect->corners[1].x};
  CapyRangeDouble rangeY =
    {.min = rect->corners[0].y, .max = rect->corners[1].y};
  loopRange(x, rangeX) loopRange(y, rangeY) {
    double pos[2] = {x, y};
    $(that, drawPoint)(pos, img);
  }
}

// Draw a segment on a CapyImage.
// Input:
//   seg: the segment to draw
//   img: the image on which to draw
static void DrawSegment(
  CapySegment const* const seg,
            CapyImg* const img) {
  methodOf(CapyPen);
  $(that, drawGeometry2D)((CapyGeometry2D const*)seg, img);
}

// Draw a triangle on a CapyImage.
// Input:
//   tri: the triangle to draw
//   img: the image on which to draw
static void DrawTriangle(
  CapyTriangle const* const tri,
             CapyImg* const img) {
  methodOf(CapyPen);
  $(that, drawGeometry2D)((CapyGeometry2D const*)tri, img);
}

// Draw a circle on a CapyImage.
// Input:
//   circle: the circle to draw
//   img: the image on which to draw
static void DrawCircle(
  CapyCircle const* const circle,
           CapyImg* const img) {
  methodOf(CapyPen);
  $(that, drawGeometry2D)((CapyGeometry2D const*)circle, img);
}

// Draw a filled circle on a CapyImage.
// Input:
//   circle: the circle to draw
//   img: the image on which to draw
static void DrawFilledCircle(
  CapyCircle const* const circle,
           CapyImg* const img) {
  methodOf(CapyPen);
  CapyImgPos_t xRange[2] = {
    (CapyImgPos_t)(-circle->radius - 1.0), (CapyImgPos_t)(circle->radius + 1.0)
  };
  CapyImgPos_t yRange[2] = {
    (CapyImgPos_t)(-circle->radius - 1.0), (CapyImgPos_t)(circle->radius + 1.0)
  };
  double radiusSquared = circle->radius * circle->radius;
  for(CapyImgPos_t x = xRange[0]; x <= xRange[1]; ++x) {
    for(CapyImgPos_t y = yRange[0]; y <= yRange[1]; ++y) {
      double d = x * x + y * y;
      if(d <= radiusSquared) {
        double pos[2] = {circle->center.x + x, circle->center.y + y};
        $(that, drawPoint)(pos, img);
      }
    }
  }
}

// Free the memory used by a CapyPen
static void Destruct(void) {
  return;
}

// Draw a Geometry2D on a CapyImage.
// Input:
//   geometry: the geometry to draw
//   img: the image on which to draw
static void DrawGeometry2D(
  CapyGeometry2D const* const geometry,
               CapyImg* const img) {
  methodOf(CapyPen);

  // Approximate the geometry with a Bezier spline and draw it
  CapyBezierSpline spline = $(geometry, getBezierSpline)();
  $(that, drawBezierSpline)(&spline, img);
  $(&spline, destruct)();
}

// Draw a text with a given font
// Input:
//   pos: the coordinates of the top-left corner of the block of text
//   text: the text to draw
//   font: the font of the text
//   img: the image on which to draw
static void DrawText(
    double const* const pos,
      char const* const text,
  CapyFont const* const font,
         CapyImg* const img) {
  methodOf(CapyPen);
  CapyListBezierSpline* textSplines = $(font, textToBezierSpline)(text);
  forEach(spline, textSplines->iter) {
    CapyVec u = {.dim = 2, .vals = (double[2]){pos[0], pos[1]}};
    $(spline, translate)(&u);
    $(that, drawBezierSpline)(spline, img);
  }
  while(textSplines->head) {
    CapyBezierSpline* spline = $(textSplines, pop)();
    CapyBezierSplineFree(&spline);
  }
  CapyListBezierSplineFree(&textSplines);
}

// Create a CapyPen
// Output:
//   Return a CapyPen with default color white and radius 1
CapyPen CapyPenCreate(void) {
  return (CapyPen){
    .color = capyColorRGBAWhite,
    .size = 1.5,
    .hardness = capyPenHardness_HB,
    .destruct = Destruct,
    .drawBezier = DrawBezier,
    .drawBezierSpline = DrawBezierSpline,
    .drawPoint = DrawPoint,
    .drawLine = DrawLine,
    .drawQuadrilateral = DrawQuadrilateral,
    .drawRectangle = DrawRectangle,
    .drawFilledRectangle = DrawFilledRectangle,
    .drawSegment = DrawSegment,
    .drawTriangle = DrawTriangle,
    .drawCircle = DrawCircle,
    .drawFilledCircle = DrawFilledCircle,
    .drawGeometry2D = DrawGeometry2D,
    .drawText = DrawText,
  };
}

// Allocate memory for a new CapyPen and create it
// Output:
//   Return a CapyPen with default color white and radius 1
// Exception:
//   May raise CapyExc_MallocFailed.
CapyPen* CapyPenAlloc(void) {
  CapyPen* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  CapyPen p = CapyPenCreate();
  memcpy(that, &p, sizeof(CapyPen));
  return that;
}

// Free the memory used by a CapyPen* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyPen to free
void CapyPenFree(CapyPen** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
