// ---------------------------------- pen.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_PEN_H
#define CAPY_PEN_H
#include "externalHeaders.h"
#include "color.h"
#include "image.h"
#include "bezier.h"
#include "font.h"

// Description:
// Pen class to draw 2d graphics on an image.

// Pen hardness
typedef enum CapyPenHardness {
  capyPenHardness_9H = 1,
  capyPenHardness_8H,
  capyPenHardness_7H,
  capyPenHardness_6H,
  capyPenHardness_5H,
  capyPenHardness_4H,
  capyPenHardness_3H,
  capyPenHardness_2H,
  capyPenHardness_H,
  capyPenHardness_F,
  capyPenHardness_HB,
  capyPenHardness_B,
  capyPenHardness_2B,
  capyPenHardness_3B,
  capyPenHardness_4B,
  capyPenHardness_5B,
  capyPenHardness_6B,
  capyPenHardness_7B,
  capyPenHardness_8B,
  capyPenHardness_9B,
  capyPenHardness_none,
} CapyPenHardness;

// Pen object
typedef struct CapyPen {

  // Color
  CapyColorData color;

  // Size (radius in pixel)
  double size;

  // Hardness
  CapyPenHardness hardness;
  CapyPad(CapyPenHardness, hardness);

  // Destructor
  void (*destruct)(void);

  // Draw a point on a CapyImage.
  // Input:
  //   pos: the coordinate in pixel of the point
  //   img: the image on which to draw
  void (*drawPoint)(
    double const* const pos,
         CapyImg* const img);

  // Draw a line on a CapyImage.
  // Input:
  //   from: the coordinate in pixel of the start point
  //   to: the coordinate in pixel of the end point
  //   img: the image on which to draw
  void (*drawLine)(
    double const* const from,
    double const* const to,
         CapyImg* const img);

  // Draw a CapyBezier curve on a CapyImage. The curve is expected to
  // have one input (in [0, 1]) and return two outputs (in pixel
  // coordinates)
  // Input:
  //   curve: the curve to draw
  //   img: the image on which to draw
  void (*drawBezier)(
    CapyBezier* const curve,
       CapyImg* const img);

  // Draw a CapyBezierSpline curve on a CapyImage. The curve is expected to
  // have one input (in [0, spline.nbSegment]) and return two outputs (in pixel
  // coordinates)
  // Input:
  //   spline: the spline to draw
  //      img: the image on which to draw
  void (*drawBezierSpline)(
    CapyBezierSpline* const spline,
             CapyImg* const img);

  // Draw a quadrilateral on a CapyImage.
  // Input:
  //   quad: the quadrialteral to draw
  //   img: the image on which to draw
  void (*drawQuadrilateral)(
    CapyQuadrilateral const* const quad,
                    CapyImg* const img);

  // Draw a rectangle on a CapyImage.
  // Input:
  //   rect: the rectangle to draw
  //   img: the image on which to draw
  void (*drawRectangle)(
    CapyRectangle const* const rect,
                CapyImg* const img);

  // Draw a filled rectangle on a CapyImage.
  // Input:
  //   rect: the rectangle to draw
  //   img: the image on which to draw
  void (*drawFilledRectangle)(
    CapyRectangle const* const rect,
                CapyImg* const img);

  // Draw a segment on a CapyImage.
  // Input:
  //   seg: the segment to draw
  //   img: the image on which to draw
  void (*drawSegment)(
    CapySegment const* const seg,
              CapyImg* const img);

  // Draw a triangle on a CapyImage.
  // Input:
  //   tri: the triangle to draw
  //   img: the image on which to draw
  void (*drawTriangle)(
    CapyTriangle const* const tri,
               CapyImg* const img);

  // Draw a circle on a CapyImage.
  // Input:
  //   circle: the circle to draw
  //   img: the image on which to draw
  void (*drawCircle)(
    CapyCircle const* const circle,
             CapyImg* const img);

  // Draw a filled circle on a CapyImage.
  // Input:
  //   circle: the circle to draw
  //   img: the image on which to draw
  void (*drawFilledCircle)(
    CapyCircle const* const circle,
             CapyImg* const img);

  // Draw a Geometry2D on a CapyImage.
  // Input:
  //   geometry: the geometry to draw
  //   img: the image on which to draw
  void (*drawGeometry2D)(
    CapyGeometry2D const* const geometry,
                 CapyImg* const img);

  // Draw a text with a given font
  // Input:
  //   pos: the coordinates of the top-left corner of the block of text
  //   text: the text to draw
  //   font: the font of the text
  //   img: the image on which to draw
  void (*drawText)(
      double const* const pos,
        char const* const text,
    CapyFont const* const font,
           CapyImg* const img);
} CapyPen;

// Create a CapyPen
// Output:
//   Return a CapyPen with default color white, hardness HB and
//   radius 1.5
CapyPen CapyPenCreate(void);

// Allocate memory for a new CapyPen and create it
// Output:
//   Return a CapyPen with default color white, hardness HB and
//   radius 1.5
// Exception:
//   May raise CapyExc_MallocFailed.
CapyPen* CapyPenAlloc(void);

// Free the memory used by a CapyPen* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyPen to free
void CapyPenFree(CapyPen** const that);
#endif
