// ------------------- plyFormat.c --------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "plyFormat.h"

// Load a CapyPointCloud from a file
// Input:
//   file: the file (as an opened CapyStreamIo)
// Output:
//   Return the CapyPointCloud.
// Exceptions:
//   May raise CapyExc_StreamReadError, CapyExc_UnsupportedFormat,
//   CapyExc_InvalidStream
static CapyPointCloud* LoadPointCloud(CapyStreamIo* const file) {
  methodOf(CapyPlyFormat);

  // Variable to memorise the result
  CapyPointCloud* pointCloud = NULL;

  // Load the whole file
  CapyListArrChar* lines = $(file, readLines)();

  // Check the magic number
  CapyArrChar lineMagicNumber = $(lines, get)(0);
  if(strcmp(lineMagicNumber.data, "ply") != 0) {
    raiseExc(CapyExc_InvalidStream);
  }

  // Check the format
  CapyArrChar lineFormat = $(lines, get)(1);
  if(strcmp(lineFormat.data, "format ascii 1.0") == 0) {
    that->isBinary = false;
  } else if(strcmp(lineFormat.data, "format binary 1.0") == 0) {
    that->isBinary = true;
    raiseExc(CapyExc_InvalidStream);
  } else raiseExc(CapyExc_InvalidStream);

  // Move to the vertex line
  CapyListArrCharElem* ptr = lines->head;
  while(
    ptr != NULL &&
    strstr(ptr->data.data, "element vertex ") != ptr->data.data
  ) {
    ptr = ptr->next;
  }
  if (ptr == NULL) raiseExc(CapyExc_InvalidStream);

  // Get the number of vertices
  size_t nbVertex = 0;
  sscanf(ptr->data.data + 15, "%lu", &nbVertex);

  // Count the number of value per vertex
  size_t dim = 0;
  ptr = ptr->next;
  while(
    ptr != NULL &&
    strstr(ptr->data.data, "property ") == ptr->data.data
  ) {
    ptr = ptr->next;
    ++dim;
  }
  if (ptr == NULL || dim == 0) raiseExc(CapyExc_InvalidStream);

  // Create the point cloud
  pointCloud = CapyPointCloudAlloc(dim);

  // Move to the start of the data
  while(
    ptr != NULL &&
    strcmp(ptr->data.data, "end_header") != 0
  ) {
    ptr = ptr->next;
  }
  if (ptr == NULL) raiseExc(CapyExc_InvalidStream);
  ptr = ptr->next;

  // Allocate memory for the data
  safeMalloc(pointCloud->points, nbVertex);
  if(!(pointCloud->points)) return NULL;

  // Decode the data
  loop(iPoint, nbVertex) {
    pointCloud->points[iPoint] = CapyVecCreate(dim);
    ++(pointCloud->size);
    char* valStr = strtok(ptr->data.data, " ");
    loop(iVal, dim) {
      if(valStr == NULL) raiseExc(CapyExc_InvalidStream);
      sscanf(valStr, "%lf", pointCloud->points[iPoint].vals + iVal);
      valStr = strtok(NULL, " ");
    }
    ptr = ptr->next;
    if (ptr == NULL) raiseExc(CapyExc_InvalidStream);
  }

  // Free memory
  while($(lines, isEmpty)() == false) {
    CapyArrChar line = $(lines, pop)();
    $(&line, destruct)();
  }
  CapyListArrCharFree(&lines);

  // Forward eventual exception
  CapyForwardExc();

  // Return the point cloud
  return pointCloud;
}

// Save a CapyPointCloud to a file
// Input:
//   pointCloud: the point cloud to save
//         file: the file (as an opened CapyStreamIo)
// Exceptions:
//   May raise CapyExc_StreamWriteError, CapyExc_UnsupportedFormat
static void SavePointCloud(
   CapyPointCloud const* const pointCloud,
           CapyStreamIo* const file) {
  methodOf(CapyPlyFormat);
  if(that->isBinary) {
    raiseExc(CapyExc_UnsupportedFormat);
    return;
  }

  // Write the magic number
  safeFPrintf(file->stream, "%s\n", "ply");

  // Write the format
  safeFPrintf(file->stream, "%s\n", "format ascii 1.0");

  // Write the number of vertex
  safeFPrintf(file->stream, "element vertex %lu\n", pointCloud->size);

  // Write the property of vertices
  loop(i, pointCloud->dim) {
    safeFPrintf(file->stream, "property float dim%lu\n", i);
  }

  // Write the end of the header
  safeFPrintf(file->stream, "%s\n", "end_header");

  // Write the vertices
  loop(iPoint, pointCloud->size) loop(iDim, pointCloud->dim) {
    safeFPrintf(file->stream, "%lf", pointCloud->points[iPoint].vals[iDim]);
    char space = ' ';
    if(iDim == pointCloud->dim - 1) space = '\n';
    safeFPrintf(file->stream, "%c", space);
  }
}

// Free the memory used by a CapyPlyFormat
static void Destruct(void) {
  methodOf(CapyPlyFormat);
  $(that, destructCapyFileFormat)();
}

// Create a CapyPlyFormat
// Output:
//   Return a CapyPlyFormat
CapyPlyFormat CapyPlyFormatCreate(void) {
  CapyPlyFormat that = {.isBinary = false};
  CapyInherits(that, CapyFileFormat, (capyFileFormat_ply));
  that.destruct = Destruct;
  that.loadPointCloud = LoadPointCloud;
  that.savePointCloud = SavePointCloud;
  return that;
}

// Allocate memory for a new CapyPlyFormat and create it
// Output:
//   Return a CapyPlyFormat
// Exception:
//   May raise CapyExc_MallocFailed.
CapyPlyFormat* CapyPlyFormatAlloc(void) {
  CapyPlyFormat* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyPlyFormatCreate();
  return that;
}

// Free the memory used by a CapyPlyFormat* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyPlyFormat to free
void CapyPlyFormatFree(CapyPlyFormat** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

