// ------------------- pngFormat.c --------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "pngFormat.h"

// Get the index of a pixel in an image given its 2D position
#define PosToIdx(img, x, y) ((y) * (img)->dims.width + (x))

// Load a CapyImg from a file
// Input:
//   file: the file (as an opened CapyStreamIo)
// Output:
//   Return the CapyImg.
// Exceptions:
//   May raise CapyExc_StreamReadError, CapyExc_UnsupportedFormat,
//   CapyExc_InvalidStream
static CapyImg* LoadImg(CapyStreamIo* const file) {

  // Read the header and check it
  png_byte header[8];
  safeFRead(file->stream, 8, header);
  bool validHeader = !png_sig_cmp(header, 0, 8);
  if(validHeader == false) {
    raiseExc(CapyExc_InvalidStream);
    return NULL;
  }

  // Read the png_structp and png_infopp structures
  png_structp pngPtr = png_create_read_struct(
    PNG_LIBPNG_VER_STRING,
    (png_voidp)NULL,
    (png_error_ptr)NULL,
    (png_error_ptr)NULL);
  if(!pngPtr) {
    raiseExc(CapyExc_InvalidStream);
    return NULL;
  }
  png_infop infoPtr = png_create_info_struct(pngPtr);
  if(!infoPtr) {
    png_destroy_read_struct(
      &pngPtr,
      (png_infopp)NULL,
      (png_infopp)NULL);
    raiseExc(CapyExc_InvalidStream);
    return NULL;
  }
  png_infop end_info = png_create_info_struct(pngPtr);
  if(!end_info) {
    png_destroy_read_struct(
      &pngPtr,
      &infoPtr,
      (png_infopp)NULL);
    raiseExc(CapyExc_InvalidStream);
    return NULL;
  }

  // Inform libpng that we have read the header
  png_set_sig_bytes(pngPtr, 8);

  // Read the image
  int png_transforms = PNG_TRANSFORM_IDENTITY;
  png_init_io(pngPtr, file->stream);
  png_read_png(pngPtr, infoPtr, png_transforms, NULL);

  // Set the mode and convert the data to rgb if necessary
  CapyImgMode mode = capyImgMode_rgba;
  png_byte colorType = png_get_color_type(pngPtr, infoPtr);
  png_byte bitDepth = png_get_bit_depth(pngPtr, infoPtr);
  switch(colorType) {
    case PNG_COLOR_TYPE_GRAY:
      mode = capyImgMode_greyscale;

      // Ensure greyscale image are encoded with 8 bits
      if(bitDepth < 8) png_set_expand_gray_1_2_4_to_8(pngPtr);
      break;
    case PNG_COLOR_TYPE_PALETTE:

      // Transform palette colors to rgbs
      png_set_palette_to_rgb(pngPtr);
      break;
  }

  // If the alpha channel is encoded in tRNS update the alpha channel
  if(png_get_valid(pngPtr, infoPtr, PNG_INFO_tRNS)) {
    png_set_tRNS_to_alpha(pngPtr);
  }

  // Get the dimensions of the image
  png_uint_32 width = png_get_image_width(pngPtr, infoPtr);
  png_uint_32 height = png_get_image_height(pngPtr, infoPtr);
  if(width >= INT_MAX || height >= INT_MAX) {
    raiseExc(CapyExc_InvalidStream);
    return NULL;
  }

  // Create the CapyImg
  CapyImgDims dims = {.width = width, .height = height};
  CapyImg* img = CapyImgAlloc(mode, dims);

  // Get the number of byte per pixels
  png_uint_32 nbChannel = 0;
  png_uint_32 nbBytePixel = 0;
  switch(colorType) {
    case PNG_COLOR_TYPE_GRAY:
      nbChannel = 1;
      break;
    case PNG_COLOR_TYPE_RGB:
      nbChannel = 3;
      break;
    case PNG_COLOR_TYPE_RGBA:
      nbChannel = 4;
      break;
    default:
      raiseExc(CapyExc_InvalidStream);
      CapyImgFree(&img);
      return NULL;
  };
  nbBytePixel = nbChannel;
  if(bitDepth == 16) nbBytePixel *= 2;

  // Get the row data of the image and the size in byte of one row
  png_bytepp rowPointers = png_get_rows(pngPtr, infoPtr);

  // Get the maximum value in the row data according to the bit depth
  CapyColorValue_t maxPixelValue = 255.0;
  if(bitDepth == 16) maxPixelValue = 65535.0;

  // Loop on the pixels and set the pixel value in the image, no need to care
  // of colorType here thanks to CapyColorData being a union
  loop(y, height) loop(x, width) {
    loop(rgba, nbChannel) {
      CapyColorValue_t val = 0.0;
      if(bitDepth == 8) {
        val = rowPointers[y][x * nbBytePixel + rgba];
      } else if(bitDepth == 16) {

        // Correct for inverted endianness
        uint16_t* pixel = (uint16_t*)(rowPointers[y] + x * nbBytePixel);
        uint16_t pixelCorr = (pixel[rgba] << 8) | (pixel[rgba] >> 8);
        val = pixelCorr;
      }
      img->pixels[PosToIdx(img, x, y)].RGBA[rgba] = val / maxPixelValue;
    }

    // If there is only one byte per pixel, duplicate the intensity channel
    if(nbChannel == 1) loop(i, 2) {
      img->pixels[PosToIdx(img, x, y)].RGBA[1 + i] =
        img->pixels[PosToIdx(img, x, y)].RGBA[0];
    }

    // If there is no alpha information, set it to fully opaque by default
    if(nbChannel < 4) img->pixels[PosToIdx(img, x, y)].RGBA[3] = 1.0;
  }

  // Get the gamma if available, else assume the gamma is 2.2
  // The gamma value in the chunk is the inverse of the display gamma
  double gamma = 1.0;
  png_uint_32 hasGamma = png_get_gAMA(pngPtr, infoPtr, &gamma);
  if(hasGamma != 0) img->gamma = 1.0 / gamma;
  else img->gamma = 2.2;

  // Free memory
  png_destroy_read_struct(&pngPtr, &infoPtr, &end_info);

  // Return the image
  return img;
}

// Save a CapyImg to a file
// Input:
//    img: the image to save
//   file: the file (as an opened CapyStreamIo)
// Exceptions:
//   May raise CapyExc_StreamWriteError, CapyExc_UnsupportedFormat
static void SaveImg(
   CapyImg const* const img,
    CapyStreamIo* const file) {
  png_structp png_ptr = png_create_write_struct(
    PNG_LIBPNG_VER_STRING,
    (png_voidp)NULL,
    (png_error_ptr)NULL,
    (png_error_ptr)NULL);
  if(png_ptr == NULL) {
    raiseExc(CapyExc_StreamWriteError);
    return;
  }
  png_infop info_ptr = png_create_info_struct(png_ptr);
  if(info_ptr == NULL) {
    png_destroy_write_struct(&png_ptr, (png_infopp)NULL);
    raiseExc(CapyExc_StreamWriteError);
    return;
  }
  png_init_io(png_ptr, file->stream);
  int bit_depth = 8;
  int color_type = PNG_COLOR_TYPE_RGB_ALPHA;
  png_set_IHDR(
    png_ptr,
    info_ptr,
    img->dims.width,
    img->dims.height,
    bit_depth,
    color_type,
    PNG_INTERLACE_NONE,
    PNG_COMPRESSION_TYPE_DEFAULT,
    PNG_FILTER_TYPE_DEFAULT);

  // The gamma value in the chunk is the inverse of the display gamma
  png_set_gAMA(png_ptr, info_ptr, 1.0 / img->gamma);
  size_t nbBytePixel = 4;
  png_bytepp row_pointers = NULL;
  safeMalloc(row_pointers, img->dims.height);
  if(!row_pointers) return;
  loop(y, img->dims.height) {
    safeMalloc(row_pointers[y], (size_t)img->dims.width * nbBytePixel);
    if(!(row_pointers[y])) return;
  }
  loop(x, img->dims.width) loop(y, img->dims.height) {
    loop(rgba, nbBytePixel) {
      CapyColorValue_t val =
        img->pixels[PosToIdx(img, x, y)].RGBA[rgba];
      row_pointers[y][x * nbBytePixel + rgba] = (uint8_t)(val * 255.0);
    }
  }
  png_set_rows(png_ptr, info_ptr, row_pointers);
  png_write_png(png_ptr, info_ptr, PNG_TRANSFORM_IDENTITY, NULL);
  png_write_end(png_ptr, info_ptr);
  png_destroy_write_struct(&png_ptr, &info_ptr);
  loop(y, img->dims.height) free(row_pointers[y]);
  free(row_pointers);
}

// Free the memory used by a CapyPngFormat
static void Destruct(void) {
  methodOf(CapyPngFormat);
  $(that, destructCapyFileFormat)();
}

// Create a CapyPngFormat
// Output:
//   Return a CapyPngFormat
CapyPngFormat CapyPngFormatCreate(void) {
  CapyPngFormat that;
  CapyInherits(that, CapyFileFormat, (capyFileFormat_png));
  that.destruct = Destruct;
  that.loadImg = LoadImg;
  that.saveImg = SaveImg;
  return that;
}

// Allocate memory for a new CapyPngFormat and create it
// Output:
//   Return a CapyPngFormat
// Exception:
//   May raise CapyExc_MallocFailed.
CapyPngFormat* CapyPngFormatAlloc(void) {
  CapyPngFormat* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyPngFormatCreate();
  return that;
}

// Free the memory used by a CapyPngFormat* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyPngFormat to free
void CapyPngFormatFree(CapyPngFormat** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
