// ------------------------------ pointCloud.c ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "pointCloud.h"

// Get the Bezier approximating the point cloud.
// Input:
//   nbIn: the number of dimension from the first dimension of a point
//         treated as inputs of the Bezier (the remaining ones being treated
//         as the output).
//   order: the Bezier order.
// Output:
//   Return a new CapyBezier which takes the first 'nbIn' values of a point
//   as input and return as output the approximation of the remaining values
//   of that point. Return NULL if the Bezier couldn't be created.
// Exception:
//   May raise CapyExc_MallocFailed.
static CapyBezier* GetApproxBezier(
               size_t const nbIn,
    CapyBezierOrder_t const order) {
  methodOf(CapyPointCloud);

  // If there are no output, return null
  if(nbIn >= that->dim) return NULL;
  size_t nbOut = that->dim - nbIn;

  // Variables used to calculate the Bezier
  CapyBezier* bezier = NULL;
  CapyMat W = { .vals = NULL };
  CapyMat inv = { .vals = NULL };
  CapyVec v = { .vals = NULL };
  CapyVec ctrls = { .vals = NULL };
  struct { double* out; } volatiles = {.out = NULL};

  // Try to allocate memory
  try {
    bezier = CapyBezierAlloc(order, nbIn, that->dim - nbIn);
    W = CapyMatCreate(bezier->nbCtrls, that->size);
    inv = CapyMatCreate(that->size, bezier->nbCtrls);
    v = CapyVecCreate(that->size);
    ctrls = CapyVecCreate(bezier->nbCtrls);
    safeMalloc(volatiles.out, nbOut);
  } catchDefault {
    CapyBezierFree(&bezier);
    CapyMatDestruct(&W);
    CapyMatDestruct(&inv);
    CapyVecDestruct(&v);
    CapyVecDestruct(&ctrls);
    free(volatiles.out);
  } endCatch;
  if(!(volatiles.out)) return NULL;
  if(bezier == NULL) return NULL;

  // Create the system, solve it for each output and update the Bezier's ctrl
  loop(iPoint, that->size) {
    $(bezier, eval)(that->points[iPoint].vals, volatiles.out);
    loop(iCtrl, bezier->nbCtrls) {
      W.vals[iPoint * bezier->nbCtrls + iCtrl] =
        bezier->ctrlWeights.vals[iCtrl];
    }
  }
  try {
    CapyMatPseudoInv(&W, &inv);
  } catchDefault {
    CapyBezierFree(&bezier);
    CapyMatDestruct(&W);
    CapyMatDestruct(&inv);
    CapyVecDestruct(&v);
    CapyVecDestruct(&ctrls);
    free(volatiles.out);
  } endCatch;
  if(bezier == NULL) return NULL;
  loop(iOut, nbOut) {
    loop(iPoint, that->size) {
      v.vals[iPoint] = that->points[iPoint].vals[nbIn + iOut];
    }
    CapyMatProdVec(&inv, &v, &ctrls);
    loop(iCtrl, bezier->nbCtrls) {
      bezier->ctrls[iCtrl].vals[iOut] = ctrls.vals[iCtrl];
    }
  }

  // Free memory
  CapyMatDestruct(&W);
  CapyMatDestruct(&inv);
  CapyVecDestruct(&v);
  CapyVecDestruct(&ctrls);
  free(volatiles.out);

  // Return the resulting Bezier
  return bezier;
}

// Calculate the covariance between two dimensions of the dataset
// Input:
//   that: the point cloud
//   i: the first dimension
//   j: the second dimension
// Output:
//   Return the covariance.
static double Covariance(
  CapyPointCloud const* const that,
                 size_t const i,
                 size_t const j) {
  if(that->size == 0) return 0.0;
  double covariance = 0.0;
  double c = 1.0 / (double)(that->size);
  loop(iPoint, that->size) {
    covariance +=
      c *
      (that->points[iPoint].vals[i] - that->mean.vals[i]) *
      (that->points[iPoint].vals[j] - that->mean.vals[j]);
  }
  return covariance;
}

// Update the mean vector
// Output:
//   that->mean is updated.
static void UpdateMean(void) {
  methodOf(CapyPointCloud);
  loop(i, that->dim) that->mean.vals[i] = 0.0;
  double c = 1.0 / (double)(that->size);
  loop(iPoint, that->size) loop(iDim, that->dim) {
    that->mean.vals[iDim] += c * that->points[iPoint].vals[iDim];
  }
}

// Update the stdDev vector
// Output:
//   that->mean and that->stdDev is updated.
static void UpdateStdDev(void) {
  methodOf(CapyPointCloud);
  $(that, updateMean)();
  loop(i, that->dim) that->stdDev.vals[i] = 0.0;
  double c = 1.0 / (double)(that->size);
  loop(iPoint, that->size) loop(iDim, that->dim) {
    that->stdDev.vals[iDim] +=
      c * that->points[iPoint].vals[iDim] * that->points[iPoint].vals[iDim];
  }
  loop(iDim, that->dim) {
    that->stdDev.vals[iDim] = sqrt(
      that->stdDev.vals[iDim] - that->mean.vals[iDim] * that->mean.vals[iDim]);
  }
}

// Update the covariance matrix
// Output:
//   that->covariance and that->mean are updated.
static void UpdateCovariance(void) {
  methodOf(CapyPointCloud);
  $(that, updateMean)();
  loop(i, that->dim) loop(j, that->dim) {
    if(i > j) {
      that->covariance.vals[j * that->dim + i] =
        that->covariance.vals[i * that->dim + j];
    } else {
      that->covariance.vals[j * that->dim + i] = Covariance(that, i, j);
    }
  }
}

// Update the pearson correlation matrix
// Output:
//   that->pearsonCorrelation, that->stdDev, that->covariance and
//   that->mean are updated.
static void UpdatePearsonCorrelation(void) {
  methodOf(CapyPointCloud);
  $(that, updateCovariance)();
  $(that, updateStdDev)();
  loop(i, that->dim) loop(j, that->dim) {
    if(i > j) {
      that->pearsonCorrelation.vals[j * that->dim + i] =
        that->pearsonCorrelation.vals[i * that->dim + j];
    } else {
      that->pearsonCorrelation.vals[j * that->dim + i] =
        that->covariance.vals[j * that->dim + i] /
        (that->stdDev.vals[i] * that->stdDev.vals[j]);
    }
  }
}

// Update the principal components
// Output:
//   that->principalComponent, that eigenValue, that->covariance and
//   that->mean are updated.
static void UpdatePrincipalComponent(void) {
  methodOf(CapyPointCloud);
  $(that, updateCovariance)();
  CapyMatGetEigen(
    &(that->covariance) , &(that->eigenValue), &(that->principalComponent)
  );
}

// Update the ranges
// Output:
//   that->range is updated.
static void UpdateRange(void) {
  methodOf(CapyPointCloud);
  loop(iPoint, that->size) loop(iDim, that->dim) {
    if(iPoint == 0) {
      that->range[iDim].min = that->points[iPoint].vals[iDim];
      that->range[iDim].max = that->range[iDim].min;
    } else {
      if(that->range[iDim].min > that->points[iPoint].vals[iDim]) {
        that->range[iDim].min = that->points[iPoint].vals[iDim];
      }
      if(that->range[iDim].max < that->points[iPoint].vals[iDim]) {
        that->range[iDim].max = that->points[iPoint].vals[iDim];
      }
    }
  }
}

// Get an estimate of the minimum sum of weights along paths linking
// two points (cf CapyPathFinder)
// Inputs:
//   iPointFrom: the start node of the path
//   iPointTo: the end node of the path
// Output:
//   Return the estimate, which must be less or equal to the actual minimum
//   sum of weights.
static double EstimateWeightPath(
  size_t const iPointFrom,
  size_t const iPointTo) {
  methodOf(CapyPointCloud);
  double weight = 0.0;
  loop(iDim, that->dim) {
    double d =
      that->points[iPointTo].vals[iDim] - that->points[iPointFrom].vals[iDim];
    weight += d * d;
  }
  weight = sqrt(weight);
  return weight;
}

// Free the memory used by a CapyPointCloud
static void Destruct(void) {
  methodOf(CapyPointCloud);
  loop(iPoint, that->size) {
    CapyVecDestruct(that->points + iPoint);
  }
  CapyMatDestruct(&(that->covariance));
  CapyMatDestruct(&(that->pearsonCorrelation));
  CapyVecDestruct(&(that->mean));
  CapyVecDestruct(&(that->stdDev));
  CapyMatDestruct(&(that->principalComponent));
  CapyVecDestruct(&(that->eigenValue));
  free(that->points);
  free(that->links);
  loop(iDim, that->dim) $(that->range + iDim, destruct)();
  free(that->range);
}

// Create a CapyPointCloud
// Input:
//   dim: the dimension of a point in the cloud
// Output:
//   Return a CapyPointCloud
// Exception:
//   May raise CapyExc_MallocFailed.
CapyPointCloud CapyPointCloudCreate(size_t const dim) {
  CapyPointCloud that = {
    .dim = dim,
    .size = 0,
    .points = NULL,
    .nbLink = 0,
    .links = NULL,
    .destruct = Destruct,
    .mean = CapyVecCreate(dim),
    .stdDev = CapyVecCreate(dim),
    .covariance = CapyMatCreate(dim, dim),
    .pearsonCorrelation = CapyMatCreate(dim, dim),
    .principalComponent = CapyMatCreate(dim, dim),
    .eigenValue = CapyVecCreate(dim),
    .range = NULL,
    .getApproxBezier = GetApproxBezier,
    .updateMean = UpdateMean,
    .updateStdDev = UpdateStdDev,
    .updateCovariance = UpdateCovariance,
    .updatePearsonCorrelation = UpdatePearsonCorrelation,
    .updatePrincipalComponent = UpdatePrincipalComponent,
    .updateRange = UpdateRange,
    .estimateWeightPath = EstimateWeightPath,
  };
  safeMalloc(that.range, dim);
  if(that.range) loop(i, dim) that.range[i] = CapyRangeDoubleCreate(0.0, 0.0);
  return that;
}

// Allocate memory for a new CapyPointCloud and create it
// Input:
//   dim: the dimension of a point in the cloud
// Output:
//   Return a CapyPointCloud
// Exception:
//   May raise CapyExc_MallocFailed.
CapyPointCloud* CapyPointCloudAlloc(size_t const dim) {
  CapyPointCloud* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyPointCloudCreate(dim);
  return that;
}

// Free the memory used by a CapyPointCloud* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyPointCloud to free
void CapyPointCloudFree(CapyPointCloud** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// BTree of CapyPointCloudNearestNeighbourRes
CapyDefBTree(
  BTreePointCloudNearestNeighbourRes, CapyPointCloudNearestNeighbourRes)

// Comparator for CapyPointCloudNearestNeighbourRes
static int CapyPointCloudNearestNeighbourResCmp(
  CapyPointCloudNearestNeighbourRes const* const elemA,
  CapyPointCloudNearestNeighbourRes const* const elemB) {
  if(elemA->dist < elemB->dist) return -1;
  if(elemA->dist > elemB->dist) return 1;
  else return 0;
}

// Function to search recursively the nearest neighbour in a point cloud
// using a B-Tree.
static CapyPointCloudNearestNeighbourRes QueryNearestNeighbourRec(
      CapyPointCloudNearestNeighbour const* const that,
                             CapyVec const* const query,
  BTreePointCloudNearestNeighbourRes const* const curBtree,
          CapyPointCloudNearestNeighbourRes const curRes,
                           CapyRangeDouble* const range,
                                     double const distQuery) {

  // Variable to memorise the result at that step of the iteration,
  // initialised with the current result
  CapyPointCloudNearestNeighbourRes res = curRes;
  res.nbTestedPoint = 0;

  // Loop on the elements at the root of the B-Tree
  loop(iElem, curBtree->nbElem) {

    // If the element is within the curent search range
    if(
      curBtree->elems[iElem].dist >= range->min &&
      curBtree->elems[iElem].dist <= range->max
    ) {

      // Get the distance of that element to the query
      double dist = CapyVecGetDistance(
        query, that->pointCloud->points + curBtree->elems[iElem].iPoint);

      // Increment the number of tested points
      res.nbTestedPoint += 1;

      // If the element is nearer than the current nearest
      if(dist < res.dist) {

        // Shrink the search range
        if(range->min < distQuery - dist) range->min = distQuery - dist;
        if(range->max > distQuery + dist) range->max = distQuery + dist;

        // Update the result
        res.iPoint = curBtree->elems[iElem].iPoint;
        res.dist = dist;
      }
    }
  }

  // Loop on the childs of the root element
  loop(iChild, curBtree->nbElem + 1) {

    // If the child exists
    if(curBtree->childs[iChild] != NULL) {

      // If the child boundaries intersect the current search range
      if(
        (
          iChild == 0 ||
          curBtree->elems[iChild - 1].dist <= range->max
        ) &&
        (
          iChild == curBtree->nbElem ||
          curBtree->elems[iChild].dist >= range->min
        )
      ) {

        // Search in the child
        CapyPointCloudNearestNeighbourRes subRes = QueryNearestNeighbourRec(
          that, query, curBtree->childs[iChild], res, range, distQuery);

        // If the search in the child improved the result
        if(subRes.dist < res.dist) {

          // Update the result
          res.iPoint = subRes.iPoint;
          res.dist = subRes.dist;
        }

        // Update the number of tested points
        res.nbTestedPoint += subRes.nbTestedPoint;
      }
    }
  }
  return res;
}

// Query the nearest point in the point cloud
// Input:
//   query: the query point
//   iInitPoint: index of the point in the point cloud used for
//               initialisation
// Output:
//   Return the index of the nearest point in the point cloud and its
//   distance to the query.
static CapyPointCloudNearestNeighbourRes QueryNearestNeighbour(
  CapyVec const* const query,
          size_t const iInitPoint) {
  methodOf(CapyPointCloudNearestNeighbour);

  // Variable to memorise the result
  CapyPointCloudNearestNeighbourRes res = {0};

  // Initialise the result with the initi point
  res.iPoint = iInitPoint;
  res.dist = CapyVecGetDistance(query, that->pointCloud->points + res.iPoint);

  // Start the recursive search
  double distQuery = CapyVecGetDistance(&(that->origin), query);
  CapyRangeDouble range = {
    .min = distQuery - res.dist,
    .max = distQuery + res.dist,
  };
  res =
    QueryNearestNeighbourRec(that, query, that->btree, res, &range, distQuery);

  // Return the result
  return res;
}

// Free the memory used by a CapyPointCloudNearestNeighbour
static void DestructPointCloudNearestNeighbour(void) {
  methodOf(CapyPointCloudNearestNeighbour);
  CapyVecDestruct(&(that->origin));
  BTreePointCloudNearestNeighbourResFree(&(that->btree));
}

// Create a CapyPointCloudCloudNearestNeighbour.
// Input:
//   pointCloud: the point cloud
//   nbMaxElem: max number of elements in the underlying BTree
//   origin: reference point for distance calculation
// Output:
//   Return a CapyPointCloudCloudNearestNeighbour
// Exception:
//   May raise CapyExc_MallocFailed.
CapyPointCloudNearestNeighbour CapyPointCloudNearestNeighbourCreate(
    CapyPointCloud const* const pointCloud,
                   size_t const nbMaxElem,
           CapyVec const* const origin) {
  CapyPointCloudNearestNeighbour that = {
    .pointCloud = pointCloud,
    .origin = CapyVecCreate(origin->dim),
    .destruct = DestructPointCloudNearestNeighbour,
    .query = QueryNearestNeighbour,
  };
  loop(i, origin->dim) that.origin.vals[i] = origin->vals[i];
  that.btree = BTreePointCloudNearestNeighbourResAlloc(
    nbMaxElem, CapyPointCloudNearestNeighbourResCmp);
  loop(iPoint, pointCloud->size) {
    CapyPointCloudNearestNeighbourRes elem = {0};
    elem.iPoint = iPoint;
    elem.dist = CapyVecGetDistance(&(that.origin), pointCloud->points + iPoint);
    $(that.btree, add)(elem);
  }
  return that;
}

// Allocate memory for a new CapyPointCloudCloudNearestNeighbour and create it
// Input:
//   pointCloud: the point cloud
//   nbMaxElem: max number of elements in the underlying BTree
//   origin: reference point for distance calculation
// Output:
//   Return a CapyPointCloudCloudNearestNeighbour
// Exception:
//   May raise CapyExc_MallocFailed.
CapyPointCloudNearestNeighbour* CapyPointCloudNearestNeighbourAlloc(
    CapyPointCloud const* const pointCloud,
                   size_t const nbMaxElem,
           CapyVec const* const origin) {
  CapyPointCloudNearestNeighbour* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyPointCloudNearestNeighbourCreate(pointCloud, nbMaxElem, origin);
  return that;
}

// Free the memory used by a CapyPointCloudNearestNeighbour* and reset
// '*that' to NULL
// Input:
//   that: a pointer to the CapyPointCloudNearestNeighbour to free
void CapyPointCloudNearestNeighbourFree(
  CapyPointCloudNearestNeighbour** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
